"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PadEfsStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a mechanism that adds 1GB-sized files containing only zero-bytes to an Amazon EFS filesystem through a given Access Point to that filesystem.
 *
 * This is being
 * provided to give you a way to increase the baseline throughput of an Amazon EFS filesystem
 * that has been deployed in bursting throughput mode (see: https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes).
 * This is most useful for your Amazon EFS filesystems that contain a very small amount of data and
 * have a baseline throughput that exceeds the throughput provided by the size of the filesystem.
 *
 * When deployed in bursting throughput mode, an Amazon EFS filesystem provides you with a baseline
 * throughput that is proportional to the amount of data stored in that filesystem. However, usage
 * of that filesystem is allowed to burst above that throughput; doing so consumes burst credits that
 * are associated with the filesystem. When all burst credits have been expended, then your filesystem
 * is no longer allowed to burst throughput and you will be limited in throughput to the greater of 1MiB/s
 * or the throughput dictated by the amount of data stored in your filesystem; the filesystem will be able
 * to burst again if it is able to accrue burst credits by staying below its baseline throughput for a time.
 *
 * Customers that deploy the Deadline Repository Filesystem on an Amazon EFS filesystem may find that
 * the filesystem does not contain sufficient data to meet the throughput needs of Deadline; evidenced by
 * a downward trend in EFS bursting credits over time. When bursting credits are expended, then the render
 * farm may begin to exhibit failure mode behaviors such as the RenderQueue dropping or refusing connections,
 * or becoming unresponsive.
 *
 * If you find that your Amazon EFS is depleting its burst credits and would like to increase the
 * amount of padding that has been added to it then you can either:
 * - Modify the value of the desired padding property of this construct and redeploy your infrastructure
 *    to force an update; or
 * - Manually invoke the AWS Step Function that has been created by this construct by finding it
 *    in your AWS Console (its name will be prefixed with "<id of this construct>StateMachine"), and
 *    then start an execution of the state machine with the following JSON document as input:
 *    { "desiredPadding": <number of GiB you want to store> }
 *
 * Warning: The implementation of this construct creates and starts an AWS Step Function to add the files
 * to the filesystem. The execution of this Step Function occurs asynchronously from your deployment. We recommend
 * verifying that the step function completed successfully via your Step Functions console.
 *
 * Resources Deployed
 * --------------------------
 * - Two AWS Lambda Functions, with roles, with full access to the given EFS Access Point.
 * - An Elastic Network Interface (ENI) for each Lambda Function in each of the selected VPC Subnets, so
 *    that the Lambda Functions can connect to the given EFS Access Point.
 * - An AWS Step Function to coordinate execution of the two Lambda Functions.
 * - Security Groups for each AWS Lambda Function.
 * - A CloudFormation custom resource that executes StepFunctions.startExecution on the Step Function
 *    whenever the stack containing this construct is created or updated.
 *
 * Security Considerations
 * ---------------------------
 * - The AWS Lambdas that are deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these Lambdas.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - By default, the network interfaces created by this construct's AWS Lambda Functions have Security Groups
 *    that restrict egress access from the Lambda Function into your VPC such that the Lambda Functions can
 *    access only the given EFS Access Point.
 *
 * @stability stable
 */
class PadEfsStorage extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        /*
        Implementation:
         This is implemented as an AWS Step Function that implements the following
         algorithm:
         try {
          du = diskUsage(<efs access point directory>)
          while (du != desiredPadding) {
            if (du < desiredPadding) {
              <grow padding by adding up to 20 1GB numbered files to the filesystem.>
            } else if (du > desiredPadding) {
              <delete 1GB numbered files from the filesystem to reduce the padding to the desired amount>
              // Note: We break here to prevent two separate invocations of the step function (e.g. accidental manual
              // invocations) from looping indefinitely. Without a break, one invocation trying to grow while another
              // tries to shrink will infinitely loop both -- the diskUsage will never settle on the value that either
              // invocation wants.
              break;
            }
            du = diskUsage(<efs access point directory>)
          }
          return success
        } catch (error) {
          return failure
        }
         */
        const diskUsageTimeout = core_1.Duration.minutes(5);
        const paddingTimeout = core_1.Duration.minutes(15);
        // Location in the lambda environment where the EFS will be mounted.
        const efsMountPoint = '/mnt/efs';
        let desiredSize;
        try {
            desiredSize = props.desiredPadding.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
        }
        catch (err) {
            core_1.Annotations.of(this).addError('Failed to round desiredSize to an integer number of GiB. The size must be in GiB.');
        }
        const securityGroup = (_b = props.securityGroup) !== null && _b !== void 0 ? _b : new aws_ec2_1.SecurityGroup(this, 'LambdaSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: false,
        });
        const lambdaProps = {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            runtime: aws_lambda_1.Runtime.NODEJS_14_X,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            // Required for access point...
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            securityGroups: [securityGroup],
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(props.accessPoint, efsMountPoint),
        };
        const diskUsage = new aws_lambda_1.Function(this, 'DiskUsage', {
            description: 'Used by RFDK PadEfsStorage to calculate disk usage of an EFS access point',
            handler: 'pad-efs-storage.getDiskUsage',
            timeout: diskUsageTimeout,
            memorySize: 128,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        diskUsage.node.addDependency(props.accessPoint);
        const doPadding = new aws_lambda_1.Function(this, 'PadFilesystem', {
            description: 'Used by RFDK PadEfsStorage to add or remove numbered 1GB files in an EFS access point',
            handler: 'pad-efs-storage.padFilesystem',
            timeout: paddingTimeout,
            // Execution requires about 70MB for just the lambda, but the filesystem driver will use every available byte.
            // Larger sizes do not seem to make a difference on filesystem write performance.
            // Set to 256MB just to give a buffer.
            memorySize: 256,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        doPadding.node.addDependency(props.accessPoint);
        // Build the step function's state machine.
        const fail = new aws_stepfunctions_1.Fail(this, 'Fail');
        const succeed = new aws_stepfunctions_1.Succeed(this, 'Succeed');
        const diskUsageTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'QueryDiskUsage', {
            lambdaFunction: diskUsage,
            comment: 'Determine the number of GB currently stored in the EFS access point',
            timeout: diskUsageTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.diskUsage',
        });
        const growTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'GrowTask', {
            lambdaFunction: doPadding,
            comment: 'Add up to 20 numbered 1GB files to the EFS access point',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const shrinkTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'ShrinkTask', {
            lambdaFunction: doPadding,
            comment: 'Remove 1GB numbered files from the EFS access point to shrink the padding',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const choice = new aws_stepfunctions_1.Choice(this, 'BranchOnDiskUsage')
            .when(aws_stepfunctions_1.Condition.numberLessThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), growTask)
            .when(aws_stepfunctions_1.Condition.numberGreaterThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), shrinkTask)
            .otherwise(succeed);
        diskUsageTask.next(choice);
        diskUsageTask.addCatch(fail, {
            // See: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-error-handling.html
            errors: ['States.ALL'],
        });
        growTask.next(diskUsageTask);
        growTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        shrinkTask.next(succeed);
        shrinkTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        const statemachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: diskUsageTask,
        });
        // ==========
        // Invoke the step function on stack create & update.
        const invokeCall = {
            action: 'startExecution',
            service: 'StepFunctions',
            apiVersion: '2016-11-23',
            region: core_1.Stack.of(this).region,
            physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('executionArn'),
            parameters: {
                stateMachineArn: statemachine.stateMachineArn,
                input: JSON.stringify({
                    desiredPadding: desiredSize,
                }),
            },
        };
        const resource = new custom_resources_1.AwsCustomResource(this, 'Default', {
            installLatestAwsSdk: true,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            onCreate: invokeCall,
            onUpdate: invokeCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [statemachine.stateMachineArn],
            }),
        });
        resource.node.addDependency(statemachine);
        // Add RFDK tags to the construct tree.
        runtime_info_1.tagConstruct(this);
    }
}
exports.PadEfsStorage = PadEfsStorage;
_a = JSII_RTTI_SYMBOL_1;
PadEfsStorage[_a] = { fqn: "aws-rfdk.PadEfsStorage", version: "0.41.0" };
//# sourceMappingURL=data:application/json;base64,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