"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * An optional EFS access point can be specified for mounting the EFS file-system. For more information on using EFS
 * Access Points, see https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html. For this to work properly, the
 * EFS mount helper is required. The EFS Mount helper comes pre-installed on Amazon Linux 2. For other Linux
 * distributions, the host machine must have the Amazon EFS client installed. We advise installing the Amazon EFS Client
 * when building your AMI. For instructions on installing the Amazon EFS client for other distributions, see
 * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro.
 *
 * NOTE: Without an EFS access point, the file-system is writeable only by the root user.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MountableEfs {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        this.fileSystem = props.filesystem;
        this.accessPoint = props.accessPoint;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        if (core_1.Construct.isConstruct(target)) {
            target.node.addDependency(this.props.filesystem.mountTargetsAvailable);
        }
        if (this.props.accessPoint) {
            const grantActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mount === null || mount === void 0 ? void 0 : mount.permissions);
            if (this.accessPointRequiresClientRootAccess(this.props.accessPoint)) {
                grantActions.push('elasticfilesystem:ClientRootAccess');
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    this.props.filesystem.node.defaultChild.attrArn,
                ],
                actions: grantActions,
                conditions: {
                    StringEquals: {
                        'elasticfilesystem:AccessPointArn': this.props.accessPoint.accessPointArn,
                    },
                },
            }));
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.accessPoint) {
            mountOptions.push('iam', `accesspoint=${this.props.accessPoint.accessPointId}`);
        }
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const resolveMountTargetDnsWithApi = (_b = this.props.resolveMountTargetDnsWithApi) !== null && _b !== void 0 ? _b : false;
        if (resolveMountTargetDnsWithApi) {
            const describeMountTargetResources = [
                this.props.filesystem.node.defaultChild.attrArn,
            ];
            if (this.props.accessPoint) {
                describeMountTargetResources.push(this.props.accessPoint.accessPointArn);
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: describeMountTargetResources,
                actions: ['elasticfilesystem:DescribeMountTargets'],
            }));
        }
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${resolveMountTargetDnsWithApi} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Returns whether the mounted file-system evaluates the UID/GID of the system user accessing the file-system.
     *
     * Some network file-systems provide features to fix a UID/GID for all access to the mounted file-system and ignore
     * the system user accessing the file. If this is the case, an implementing class must indicate this in the return
     * value.
     *
     * @stability stable
     * @inheritdoc true
     */
    usesUserPosixPermissions() {
        if (this.accessPoint) {
            // We cannot determine if the access point forces a POSIX user in the import case
            if (!(this.accessPoint instanceof efs.AccessPoint)) {
                throw new Error(`MountableEfs.usesUserPosixPermissions() only supports efs.AccessPoint instances, got "${this.accessPoint.constructor.name}"`);
            }
            const accessPointResource = this.accessPoint.node.defaultChild;
            return !accessPointResource.posixUser;
        }
        return true;
    }
    /**
     * Uses a CDK escape-hatch to fetch the UID/GID of the access point POSIX user.
     *
     * @param accessPoint The access point to obtain the POSIX user for
     */
    getAccessPointPosixUser(accessPoint) {
        const accessPointResource = accessPoint.node.defaultChild;
        return accessPointResource.posixUser;
    }
    /**
     * Uses a synthesis-time check to determine whether an access point is setting its UID/GID to 0 (root). Mounting such
     * an access point requires the `ClientRootAccess` IAM permission.
     *
     * If this introspection is possible and the access point is determined to require root access, the method returns
     * true.
     *
     * If there is no information at synthesis-time, the method returns false as a secure default.
     *
     * @param accessPoint The access point to introspect
     */
    accessPointRequiresClientRootAccess(accessPoint) {
        if (accessPoint instanceof efs.AccessPoint) {
            const posixUser = this.getAccessPointPosixUser(accessPoint);
            // The following code path is cannot be triggered using the L2 construct for EFS Access Points. It currently
            // accepts a PosixUser struct. We will skip coverage for the time-being.
            /* istanbul ignore next */
            if (core_1.isResolvableObject(posixUser)) {
                // We can't know at synthesis time whether this POSIX user is root. Use secure defaults.
                return false;
            }
            if (!posixUser) {
                // No POSIX user specified we will not grant ClientRootAccess permission to opt on the side of secure defaults.
                return false;
            }
            // We have synthesis-time values for the UID/GID being set in the access point. Return true if either is 0 (root).
            return Number(posixUser.uid) === 0 || Number(posixUser.gid) === 0;
        }
        else {
            // This code path is for imported or custom-implementations of efs.AccessPoint
            // We cannot introspect the access point, so we will impose secure defaults and not grant ClientRootAccess.
            return false;
        }
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
_a = JSII_RTTI_SYMBOL_1;
MountableEfs[_a] = { fqn: "aws-rfdk.MountableEfs", version: "0.41.0" };
//# sourceMappingURL=data:application/json;base64,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