"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and executing commands against it.
 *
 * To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the MongoDB administrator credentials,
 *    and the MongoDB application port. An attacker that can find a way to modify and execute this lambda could use it to
 *    modify or read any data in the database. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g;
        super(scope, id);
        (_b = props.users.x509AuthUsers) === null || _b === void 0 ? void 0 : _b.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            description: `Used by a MongoDbPostInstallSetup ${core_1.Names.uniqueId(this)} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        (_d = props.users.passwordAuthUsers) === null || _d === void 0 ? void 0 : _d.forEach(secret => secret.grantRead(lamdbaFunc));
        (_e = props.users.x509AuthUsers) === null || _e === void 0 ? void 0 : _e.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: (_f = props.users.passwordAuthUsers) === null || _f === void 0 ? void 0 : _f.map(secret => secret.secretArn),
            X509AuthUsers: (_g = props.users.x509AuthUsers) === null || _g === void 0 ? void 0 : _g.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
_a = JSII_RTTI_SYMBOL_1;
MongoDbPostInstallSetup[_a] = { fqn: "aws-rfdk.MongoDbPostInstallSetup", version: "0.41.0" };
//# sourceMappingURL=data:application/json;base64,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