"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    CodeCommitTrigger["NONE"] = "None";
    CodeCommitTrigger["POLL"] = "Poll";
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 */
class CodeCommitSourceAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        var _a;
        const branch = (_a = props.branch) !== null && _a !== void 0 ? _a : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        const baseId = stage.pipeline.node.uniqueId;
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : '-${this.branch}-';
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,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