"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionsProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
let SDK_METADATA = undefined;
/**
 * integ-tests can only depend on '@aws-cdk/core' so
 * this construct creates a lambda function provider using
 * only CfnResource
 */
class LambdaFunctionProvider extends core_2.Construct {
    constructor(scope, id /*, props?: LambdaFunctionProviderProps*/) {
        super(scope, id);
        this.policies = [];
        const staging = new core_1.AssetStaging(this, 'Staging', {
            sourcePath: path.join(__dirname, 'lambda-handler.bundle'),
        });
        const stack = core_1.Stack.of(this);
        const asset = stack.synthesizer.addFileAsset({
            fileName: staging.relativeStagedPath(stack),
            sourceHash: staging.assetHash,
            packaging: core_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const role = new core_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: [
                    {
                        PolicyName: 'Inline',
                        PolicyDocument: {
                            Version: '2012-10-17',
                            Statement: core_1.Lazy.list({ produce: () => this.policies }),
                        },
                    },
                ],
            },
        });
        const handler = new core_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Runtime: 'nodejs14.x',
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: core_1.Duration.minutes(2).toSeconds(),
                Handler: 'index.handler',
                Role: role.getAtt('Arn'),
            },
        });
        this.serviceToken = core_1.Token.asString(handler.getAtt('Arn'));
        this.roleArn = role.getAtt('Arn');
    }
    addPolicies(policies) {
        this.policies.push(...policies);
    }
}
/**
 * Mimic the singletonfunction construct in '@aws-cdk/aws-lambda'
 */
class SingletonFunction extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.policies = [];
        this.lambdaFunction = this.ensureFunction(props);
        this.serviceToken = this.lambdaFunction.serviceToken;
    }
    /**
     * The policies can be added by different constructs
     */
    onPrepare() {
        this.lambdaFunction.addPolicies(this.policies);
    }
    ensureFunction(props) {
        const constructName = 'SingletonFunction' + slugify(props.uuid);
        const existing = core_1.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        return new LambdaFunctionProvider(core_1.Stack.of(this), constructName);
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        if (SDK_METADATA === undefined) {
            // eslint-disable-next-line
            SDK_METADATA = require('./sdk-api-metadata.json');
        }
        const srv = service.toLowerCase();
        const iamService = (SDK_METADATA[srv] && SDK_METADATA[srv].prefix) || srv;
        const iamAction = api.charAt(0).toUpperCase() + api.slice(1);
        this.policies.push({
            Action: [`${iamService}:${iamAction}`],
            Effect: 'Allow',
            Resource: resources || ['*'],
        });
    }
}
/**
 * Represents an assertions provider. The creates a singletone
 * Lambda Function that will create a single function per stack
 * that serves as the custom resource provider for the various
 * assertion providers
 */
class AssertionsProvider extends core_2.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.policies = [];
        this.handler = new SingletonFunction(this, 'AssertionsProvider', {
            uuid: '1488541a-7b23-4664-81b6-9b4408076b81',
            policies: core_1.Lazy.list({ produce: () => this.policies }),
        });
        this.handlerRoleArn = this.handler.lambdaFunction.roleArn;
        this.serviceToken = this.handler.serviceToken;
    }
    /**
     * Encode an object so it can be passed
     * as custom resource parameters. Custom resources will convert
     * all input parameters to strings so we encode non-strings here
     * so we can then decode them correctly in the provider function
     */
    encode(obj) {
        if (!obj) {
            return obj;
        }
        return JSON.parse(JSON.stringify(obj), (_k, v) => {
            switch (v) {
                case true:
                    return 'TRUE:BOOLEAN';
                case false:
                    return 'FALSE:BOOLEAN';
                default:
                    return v;
            }
        });
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        this.handler.addPolicyStatementFromSdkCall(service, api, resources);
    }
}
exports.AssertionsProvider = AssertionsProvider;
_a = JSII_RTTI_SYMBOL_1;
AssertionsProvider[_a] = { fqn: "@aws-cdk/integ-tests.AssertionsProvider", version: "1.163.1" };
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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