#!/usr/bin/env python3

"""Output installed Python packages in requirements format, including also hashes of Python packages."""

import email
import json
import logging
import operator
import os
import platform
from typing import Any, Dict, List, Optional, Tuple, Set
from urllib.parse import urlparse

import daiquiri
from packaging.utils import canonicalize_name

_LOGGER = logging.getLogger(__name__)
daiquiri.setup(level=logging.INFO)

_PYPI_URL_PREFIX = "https://files.pythonhosted.org/packages/"
_PYPI_SIMPLE = "https://pypi.org/simple"
_IS_PROVENANCE_KEY = "_@is_provenance"  # Dirty, used internally to distinguish provenance_url.json and direct_url.json.
_PREAMBLE_TEMPLATE = """\
#
# This file is autogenerated by {title} version {version} with Python {python_version}.
#
"""


def _get_preamble() -> str:
    """Get requirements file preamble."""
    from pip_preserve import __version__
    from pip_preserve import __title__

    return _PREAMBLE_TEMPLATE.format(title=__title__, version=__version__, python_version=platform.python_version())


def _load_url_files(dist_info_dir: str, *, raise_on_error: bool = True) -> Optional[Tuple[str, str, Dict[str, Any]]]:
    """Load provenance_url.json file or direct_url.json file from the specified dist-info directory."""
    if not os.path.isdir(dist_info_dir):
        raise ValueError(f"Path {dist_info_dir!r} does not exist or is not a directory")

    # Metadata
    metadata_path = os.path.join(dist_info_dir, "METADATA")
    if not os.path.isfile(metadata_path):
        msg = f"No metadata found in {dist_info_dir!r}"
        if raise_on_error:
            raise ValueError(msg)

        _LOGGER.warning(msg)
        return None

    with open(metadata_path) as f:
        metadata = email.message_from_file(f)

    name = metadata.get("Name")
    if not name:
        msg = f"No distribution name stated in {metadata_path!r}"
        if raise_on_error:
            raise ValueError(msg)

        _LOGGER.warning(msg)
        return None

    # Always use canonical names.
    name = canonicalize_name(name)

    version = metadata.get("Version")
    if not version:
        msg = f"No distribution version stated for {name!r} in {metadata_path!r}"
        if raise_on_error:
            raise ValueError(msg)

        _LOGGER.warning(msg)
        return None

    # provenance_url.json
    provenance_url_path = os.path.join(dist_info_dir, "provenance_url.json")
    if not os.path.isfile(provenance_url_path):
        _LOGGER.debug("No provenance_url.json found in %r, trying direct_url.json", dist_info_dir)

        direct_url_path = os.path.join(dist_info_dir, "direct_url.json")
        if os.path.isfile(direct_url_path):
            _LOGGER.debug("Found direct_url.json file for %r", dist_info_dir)
            with open(direct_url_path) as f:
                direct_url_content = json.load(f)

            return name, version, direct_url_content
        else:
            msg = f"No provenance_url.json or direct_url.json found for {name!r} in version {version!r}"
            if raise_on_error:
                raise ValueError(msg)

            _LOGGER.warning(msg)
            return None
    else:
        with open(provenance_url_path) as f:
            provenance_url_content = json.load(f)

        # Dirty, but keep for future ourselves, so we simplify generating files.
        provenance_url_content[_IS_PROVENANCE_KEY] = True
        return name, version, provenance_url_content


def _get_hashes(hashes: Dict[str, str]) -> str:
    """Prepare hashes for a distribution so that they can be used in requirements files."""
    digests = [f"--hash={d}:{v}" for d, v in hashes.items()]

    result = ""
    for idx, digest in enumerate(digests):
        if idx != 0:
            result += " \\\n"
        result += f"  {digest}"

    return result


def _get_indexes(distributions_info: List[Tuple[str, str, Dict[str, Any]]]) -> List[str]:
    """Generate requirements.txt with index configuration."""
    index_urls: Set[str] = set()
    for name, version, entry in distributions_info:
        if not entry.get(_IS_PROVENANCE_KEY, False):
            # Skip direct url installations.
            continue

        if entry["url"].startswith(_PYPI_URL_PREFIX):
            index_urls.add(_PYPI_SIMPLE)
        else:
            # A simple guess where packages could be hosted, would eventually deserve better heuristics.
            parsed_url = urlparse(entry["url"])

            path_parts = parsed_url.path.split("/")
            if len(path_parts) < 2:
                _LOGGER.warning(
                    "Cannot find package name in URL for package %r in version %r "
                    "- please check index URL in the output: %s",
                    name,
                    version,
                    entry["url"],
                )
                new_path = parsed_url.path  # Keep the original URL.
            else:
                new_path = "/".join(path_parts[:-2])
                if path_parts[-2] != name:
                    _LOGGER.warning(
                        "URL to package %r in version %r does not state package name "
                        "- please check index URL in the output: %s",
                        name,
                        version,
                        entry["url"],
                    )

            parsed_url = parsed_url._replace(path=new_path)
            index_urls.add(parsed_url.geturl())

    indexes: List[str] = []
    if _PYPI_SIMPLE in index_urls:
        if len(index_urls) == 1:
            # If there is only PyPI, do not state any index configuration.
            return []
        indexes.append(f"--index-url {_PYPI_SIMPLE}")
        index_urls.remove(_PYPI_SIMPLE)

    for idx, index in enumerate(index_urls):
        if idx == 0 and len(indexes) == 0:
            indexes.append(f"--index-url {index}")
            continue

        indexes.append(f"--extra-index-url {index}")

    return indexes


def _print_archive_info(entry: Dict[str, Any]) -> str:
    """Print archive_info entries."""
    result = entry["url"]
    hashes = _get_hashes(entry["archive_info"]["hashes"])
    if hashes:
        result += " \\\n"
        result += hashes

    return result + "\n"


def _print_direct_url(name: str, version: str, entry: Dict[str, Any]) -> str:
    """Print a spec constructed out of a direct URL."""
    result = f"# {name}=={version} installed using a direct URL\n"
    if "vcs_info" in entry:
        _LOGGER.debug("Package %r in version %r is installed from a VCS", name, version)
        vcs_info = entry["vcs_info"]

        url = vcs_info["vcs"] + "+" + entry["url"] + "@" + vcs_info["commit_id"]
        if "subdirectory" in entry:
            url += f"#subdirectory={entry['subdirectory']}"

        return result + url + "\n"
    elif "dir_info" in entry:
        _LOGGER.debug("Package %r in version %r is installed from a directory", name, version)
        if entry["dir_info"].get("editable", False):
            return result + f"-e {entry['url']}" + "\n"
        else:
            return result + entry["url"] + "\n"
    elif "archive_info" in entry:
        _LOGGER.debug("Pacakge %r in version %r is installed using a direct URL")
        return result + _print_archive_info(entry)
    else:
        raise NotImplementedError(f"Unknown direct_url.json content: {entry!r}")


def _print_reqs(name: str, version: str, entry: Dict[str, Any], *, direct_url: bool = False) -> str:
    """Output requirement for the given direct_url.json/provenance_url.json file."""
    result = ""
    if direct_url:
        if entry.get(_IS_PROVENANCE_KEY, False):
            _LOGGER.debug("Package %r in version %r is installed using its name and a version specifier")
            result += _print_archive_info(entry)
        else:
            result += _print_direct_url(name, version, entry)
    else:
        if entry.get(_IS_PROVENANCE_KEY, False):
            _LOGGER.debug("Package %r in version %r is installed using its name and a version specifier")

            req = f"{name}=={version}"
            hashes = _get_hashes(entry["archive_info"]["hashes"])
            if hashes:
                req += " \\\n" + hashes
            result += req + "\n"
        else:
            result += _print_direct_url(name, version, entry)

    return result


def preserve_requirements(site_packages: List[str], *, ignore_errors: bool, direct_url: bool) -> Optional[str]:
    """Output installed Python packages in requirements format, including also hashes of Python packages."""
    distributions_info: List[Tuple[str, str, Dict[str, Any]]] = []
    for site_packages_item in site_packages:
        if not os.path.isdir(site_packages_item):
            _LOGGER.warning("Not a directory: %r", site_packages_item)
            continue

        for item in os.listdir(site_packages_item):
            if not item.endswith(".dist-info"):
                _LOGGER.debug("Skipping %s as it is not a .dist-info directory", item)
                continue

            dist_info_dir = os.path.join(site_packages_item, item)

            _LOGGER.debug("Checking directory %r", dist_info_dir)
            info = _load_url_files(dist_info_dir, raise_on_error=not ignore_errors)

            if info is None:
                _LOGGER.warning(
                    "The generated output will miss information from %r, "
                    "please review any missing packages in the output",
                    dist_info_dir,
                )
                continue

            _LOGGER.debug("Extracted information from .dist-info: %r", info)
            distributions_info.append(info)

    if not distributions_info:
        _LOGGER.error(f"No packages found in %r", site_packages)
        return None

    # Make sure the output is sorted based on distribution name.
    distributions_info.sort(key=operator.itemgetter(0))

    _LOGGER.debug("Generating output...")
    result = _get_preamble()
    if not direct_url:
        indexes = _get_indexes(distributions_info)
        if indexes:
            result += "\n".join(indexes) + "\n"

    for name, version, entry in distributions_info:
        result += _print_reqs(name, version, entry, direct_url=direct_url)

    return result
