# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_assume_role_lib']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.17.51,<2.0.0']

setup_kwargs = {
    'name': 'aws-assume-role-lib',
    'version': '2.10.0',
    'description': 'Assumed role session chaining (with credential refreshing) for boto3',
    'long_description': '# aws-assume-role-lib\n**Assumed role session chaining (with credential refreshing) for boto3**\n\nThe typical way to use boto3 when programmatically assuming a role is to explicitly call [`sts.AssumeRole`](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/sts.html#STS.Client.assume_role) and use the returned credentials to create a new [`boto3.Session`](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/core/session.html) or client.\nIt looks like this mess of code:\n\n```python\nrole_arn = "arn:aws:iam::123456789012:role/MyRole"\nsession = boto3.Session()\n\nsts = session.client("sts")\nresponse = sts.assume_role(\n    RoleArn=role_arn,\n    RoleSessionName="something_you_have_to_think_about"\n)\n\ncredentials = response["Credentials"]\n\nassumed_role_session = boto3.Session(\n    aws_access_key_id=credentials["AccessKeyId"],\n    aws_secret_access_key=credentials["SecretAccessKey"],\n    aws_session_token=credentials["SessionToken"]\n)\n\n# use the session\nprint(assumed_role_session.client("sts").get_caller_identity())\n```\n\nThis code is verbose, requires specifying a role session name even if you don\'t care what it is, and must explicitly handle credential expiration and refreshing if needed (in a Lambda function, this is typically handled by calling `AssumeRole` in every invocation).\n\nWith `aws-assume-role-lib`, all that collapses down to a single line. The assumed role session automatically refreshes expired credentials and generates a role session name if one is not provided.\n\n```python\nrole_arn = "arn:aws:iam::123456789012:role/MyRole"\nsession = boto3.Session()\n\nassumed_role_session = aws_assume_role_lib.assume_role(session, role_arn)\n\n# use the session\nprint(assumed_role_session.client("sts").get_caller_identity())\n```\n\nIn a Lambda function that needs to assume a role, you can create the assumed role session during initialization and use it for the lifetime of the execution environment, with `AssumeRole` calls only being made when necessary, not on every invocation.\n\nNote that in `~/.aws/config`, [you have the option to have profiles that assume a role based on another profile](https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-role.html), and this automatically handles refreshing expired credentials as well.\n\nIf you\'ve only used `boto3.client()` and are not familiar with boto3 sessions, [here\'s an explainer](https://ben11kehoe.medium.com/boto3-sessions-and-why-you-should-use-them-9b094eb5ca8e).\n\n# Installation\n\n```bash\npip install --user aws-assume-role-lib\n```\n\nOr just add [`aws_assume_role_lib.py`](https://raw.githubusercontent.com/benkehoe/aws-assume-role-lib/stable/aws_assume_role_lib/aws_assume_role_lib.py) to your project.\n\nView the release history [here](CHANGELOG.md).\n\n# Usage\n\n```python\nimport boto3\nfrom aws_assume_role_lib import assume_role\n\n# Get a session\nsession = boto3.Session()\n# or with a profile:\n# session = boto3.Session(profile_name="my-profile")\n\n# Assume the session\nassumed_role_session = assume_role(session, "arn:aws:iam::123456789012:role/MyRole")\n\n# do stuff with the original credentials\nprint(session.client("sts").get_caller_identity()["Arn"])\n\n# do stuff with the assumed role\nprint(assumed_role_session.client("sts").get_caller_identity()["Arn"])\n```\n\nIn Lambda, initialize the sessions outside the handler, and `AssumeRole` will only get called when necessary, rather than on every invocation:\n```python\nimport os\nimport boto3\nfrom aws_assume_role_lib import assume_role, generate_lambda_session_name\n\n# Get the Lambda session\nSESSION = boto3.Session()\n\n# Get the config\nROLE_ARN = os.environ["ROLE_ARN"]\nROLE_SESSION_NAME = generate_lambda_session_name() # see below for details\n\n# Assume the session\nASSUMED_ROLE_SESSION = assume_role(SESSION, ROLE_ARN, RoleSessionName=ROLE_SESSION_NAME)\n\ndef handler(event, context):\n    # do stuff with the Lambda role using SESSION\n    print(SESSION.client("sts").get_caller_identity()["Arn"])\n\n    # do stuff with the assumed role using ASSUMED_ROLE_SESSION\n    print(ASSUMED_ROLE_SESSION.client("sts").get_caller_identity()["Arn"])\n```\nLearn more about the benefits of `aws-assume-role-lib` in Lambda functions in the [demo](lambda-demo/README.md).\n\n# Interface\n\n```\nassume_role(\n    # required arguments\n    session:           boto3.Session,\n    RoleArn:           str,\n\n    *,\n    # keyword-only arguments for AssumeRole\n    RoleSessionName:   str                                    = None,\n    PolicyArns:        Union[list[dict[str, str]], list[str]] = None,\n    Policy:            Union[str, dict]                       = None,\n    DurationSeconds:   Union[int, datetime.timedelta]         = None,\n    Tags:              list[dict[str, str]]                   = None,\n    TransitiveTagKeys: list[str]                              = None,\n    ExternalId:        str                                    = None,\n    SerialNumber:      str                                    = None,\n    TokenCode:         str                                    = None,\n    SourceIdentity:    str                                    = None,\n    additional_kwargs: dict                                   = None,\n\n    # keyword-only arguments for returned session\n    region_name:       Union[str, bool]                       = None,\n\n    # keyword-only arguments for assume_role() itself\n    validate:          bool                                   = True,\n    cache:             dict                                   = None,\n)\n```\n\n`assume_role()` takes a session and a role ARN, and optionally [other keyword arguments for `sts.AssumeRole`](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/sts.html#STS.Client.assume_role).\n\nUnlike the `AssumeRole` API call itself, `RoleArn` is required, but `RoleSessionName` is not.\nThe `RoleSessionName` is set for you if it is not provided; it will use the [`SourceIdentity`](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_control-access_monitor.html) if that is provided, otherwise it will generated.\nIf you want this generated value for `RoleSessionName` when `SourceIdentity` is provided (the behavior in v2.8 and before), set `RoleSessionName` to the special value `aws_assume_role_lib.AUTOMATIC_ROLE_SESSION_NAME`.\n\nNote that unlike the boto3 sts client method, you can provide the `Policy` parameter (the [inline session policy](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#policies_session)) as a `dict` instead of as a serialized JSON string, `PolicyArns` as a list of ARNs, and `DurationSeconds` as a `datetime.timedelta` instead of as an integer.\n\nBy default, the session returned by `assume_role()` links its region configuration to the input session.\nIf you would like to set the region explicitly, pass it in the `region_name` parameter.\n\nNote that if the parent session was created without a region passed in to the `Session` constructor, it has an implicit region, based on searching potential configuration locations.\nThis means that the region used by the session can change (for example, if you set or change `os.environ["AWS_DEFAULT_REGION"]`).\nBy default, the child session region is linked to the parent session, so if the parent session has an implicit region, or if the parent session\'s region is changed directly, they would both change.\nIf you would like to fix the child session region to be explicitly the current value, pass `region_name=True`.\nIf, for some reason, you have an explicit region set on the parent, and want the child to have implicit region config, pass `region_name=False`.\n\nBy default, `assume_role()` checks if the parameters are invalid.\nWithout this validation, errors for these issues are more confusingly raised when the child session is first used to make an API call (boto3 doesn\'t make the call to retrieve credentials until they are needed).\nHowever, this incurs a small time penalty, so parameter validation can be disabled by passing `validate=False`.\n\nIf any new arguments are added to `AssumeRole` in the future and this library is not updated to allow them directly, they can be passed in as a dict via the `additional_kwargs` argument.\n\nThe parent session is available on the child session in the `assume_role_parent_session` property.\nNote this property is added by this library; ordinary boto3 sessions do not have it.\n\n# Patching boto3\n\nYou can make the `assume_role()` function available directly in boto3 by calling `patch_boto3()`.\nThis creates a `boto3.assume_role(RoleArn, ...)` function (note that it does not take a session, it uses the same default session as `boto3.client()`), and adds a `boto3.Session.assume_role()` method.\nSo usage for that looks like:\n\n```python\nimport boto3\nimport aws_assume_role_lib\naws_assume_role_lib.patch_boto3()\n\nassumed_role_session = boto3.assume_role("arn:aws:iam::123456789012:role/MyRole")\n# the above is basically equivalent to:\n# aws_assume_role_lib.assume_role(boto3.Session(), "arn:aws:iam::123456789012:role/MyRole")\n\nsession = boto3.Session(profile_name="my-profile")\nassumed_role_session = session.assume_role("arn:aws:iam::123456789012:role/MyRole")\n```\n\n# Role session names for Lambda functions\nLearn more about the benefits of `aws-assume-role-lib` in Lambda functions in the [demo](lambda-demo/README.md).\n\nIf you don\'t provide a role session name, but you provide a [`SourceIdentity`](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_control-access_monitor.html), this value is used for the role session name as well.\nIf `SourceIdentity` is not provided either, the underlying `botocore` library [generates one using a timestamp](https://github.com/boto/botocore/blob/c53072ec257ef47e2fc749c384a9488fd3f3e626/botocore/credentials.py#L730).\nThat\'s the best it can do, because it doesn\'t have any other context.\n\nBut in a Lambda function, we do have additional context, the Lambda function itself.\nIf you call `generate_lambda_session_name()` inside an instance of a Lambda function, it returns a session name that corresponds to the function instance, which you can use when assuming a role in the Lambda function (either with this library\'s `assume_role()` or any other method).\nThe purpose of this is to simplify tracing usage of the session back to the function instance.\n\nThe returned value is in one of the following forms, depending on the length of the values, to keep the session name within the maximum of 64 characters:\n* `{function_name}`\n* `{function_name}.{identifier}`\n* `{function_name}.{function_version}.{identifier}`\n\nThe function version is never included if it is `$LATEST`.\n\nThe maximum role session name length is 64 characters. To ensure this, and\nto provide at least 4 characters of the identifier when it is used, the\nfollowing rules apply, in order:\n1. If the function name is longer than 59 characters, the session name is the truncated function name.\n2. If the function name plus the function version is longer than 59 characters, the session name is the function name plus the identifier, truncated.\n3. Otherwise, the session name is the function name plus the version (if one is found and not $LATEST) plus the identifier, truncated.\n\nThe identifier is the function instance\'s unique ID taken from the CloudWatch log stream name; if this cannot be found, it\'s a timestamp if the identifier can be at least 14 characters long (to provide for second-level precision), otherwise it is a random string.\nThe identifier will not be included unless at least 4 characters\n\nThe values are automatically extracted from [the relevant environment variables](https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-runtime); you can override any of them by providing them as arguments to the function.\n\n# ARN formatting\n`assume_role()` requires a role ARN, and if you know the role name and account id but have trouble remembering the [exact format of role ARNs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_identityandaccessmanagement.html#identityandaccessmanagement-resources-for-iam-policies), there\'s `get_role_arn()` for you.\nThere\'s additionally a `get_assumed_role_session_arn()` for formatting assumed role session ARNs.\n\n```\nget_role_arn(\n    account_id: Union[str, int],\n    role_name:  str,\n    path:       str             = "",\n    partition:  str             = "aws",\n)\n\nget_assumed_role_session_arn(\n    account_id:        Union[str, int],\n    role_name:         str,\n    role_session_name: str,\n    partition:         str             = "aws",\n)\n```\n\nFor `get_role_arn()`, if the role name has a path, it can be provided as part of the name or as the separate `path` argument (but not both).\nAssumed role session ARNs do not include the role path; if it is used in the role name it is removed.\n\n# Caching\n\nIf you would like to cache the credentials on the file system, you can use the `JSONFileCache` class, which will create files under the directory you provide in the constructor (which it will create if it doesn\'t exist).\nUse it like:\n```python\nassumed_role_session = assume_role(session, "arn:aws:iam::123456789012:role/MyRole", cache=JSONFileCache("path/to/dir"))\n```\nYou can also use any `dict`-like object for the cache (supporting `__getitem__`/`__setitem__`/`__contains__`).\n\n# Command line use\n`aws-assume-role-lib` has basic support for retrieving assumed role credentials from the command line.\nIn general, it\'s better to make profiles in `~/.aws/config` for role assumption, like this:\n\n```ini\n# this is a pre-existing profile you already have\n[profile profile-to-call-assume-role-with]\n# maybe it\'s IAM User credentials\n# or AWS SSO config\n# or whatever else you may have\n\n[profile my-assumed-role]\nrole_arn = arn:aws:iam::123456789012:role/MyRole\n# optional: role_session_name = MyRoleSessionName\n\nsource_profile = profile-to-call-assume-role-with\n# or instead of source_profile, you can tell it to\n# use external credentials. one of:\n# credential_source = Environment\n# credential_source = Ec2InstanceMetadata\n# credential_source = EcsContainer\n```\n\nYou can use `my-assumed-role` like any other profile.\nIt uses the AWS SDKs\' built-in support for role assumption, rather than relying on this third party library.\nIt also gets you credential refreshing from the SDKs, where getting the credentials in the manner below cannot refresh them when they expire.\n\nBut if you absolutely must have ad hoc role assumption on the command line, use the module invocation syntax `python -m aws_assume_role_lib ROLE_ARN [OPTIONS]`.\n\nThe options are:\n* `--profile`: use a specific configuration profile.\n* `--env`: print the credentials as environment variables (the default), suitable for `export $(python -m aws_assume_role_lib ...)`.\n* `--json`: print the credentials in [`credential_process`-formatted JSON format](https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-sourcing-external.html). Note that you don\'t normally need to use this as a `credential_process` in a profile, because you can just directly make the profile do role assumption as shown above.\n* The remaining options are the arguments to `assume_role()`:\n  * `--RoleSessionName`\n  * `--PolicyArns`: must be a comma-separated list of ARNs, a JSON list of ARNs, or a JSON object per the API\n  * `--Policy`: must be a JSON object\n  * `--DurationSeconds`\n  * `--Tags`: must be formatted as `Key1=Value1,Key2=Value2`, or a JSON object.\n  * `--TransitiveTagKeys`: must be a comma-separated list or a JSON list.\n  * `--ExternalId`\n  * `--SerialNumber`\n  * `--TokenCode`\n  * `--SourceIdentity`\n  * `--additional-kwargs`: must be a JSON object.\n',
    'author': 'Ben Kehoe',
    'author_email': 'ben@kehoe.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/benkehoe/aws-assume-role-lib',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
