#!/usr/bin/env python3
# This file is part of Xpra.
# Copyright (C) 2016, 2017 Antoine Martin <antoine@xpra.org>
# Xpra is released under the terms of the GNU GPL v2, or, at your option, any
# later version. See the file COPYING for details.

import unittest
import binascii

from xpra.codecs.pillow.decoder import get_image_type


class TestPillowHeader(unittest.TestCase):

    def test_known_header(self):
        examples = {
            "png"   : b"\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x00\x01\x00\x00\x00\x01\x08\x02\x00\x00\x00\x90wS\xde\x00\x00\x00\x0cIDAT\x08\xd7c\xf8\xff\xff?\x00\x05\xfe\x02\xfe\xdc\xccY\xe7\x00\x00\x00\x00IEND\xaeB`\x82'",
            "webp"  : b"RIFF&\x00\x00\x00WEBPVP8 \x1a\x00\x00\x000\x01\x00\x9d\x01*\x01\x00\x01\x00\x00\xc0\x12%\xa4\x00\x03p\x00\xfe\xfe\xddx\x00\x00\x00",
            "jpeg"  : binascii.unhexlify("ffd8ffe000104a46494600010101012c012c0000fffe00134372656174656420776974682047494d50ffdb0043000302020302020303030304030304050805050404050a070706080c0a0c0c0b0a0b0b0d0e12100d0e110e0b0b1016101113141515150c0f171816141812141514ffdb00430103040405040509050509140d0b0d1414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414ffc00011080001000103011100021101031101ffc4001f0000010501010101010100000000000000000102030405060708090a0bffc400b5100002010303020403050504040000017d01020300041105122131410613516107227114328191a1082342b1c11552d1f02433627282090a161718191a25262728292a3435363738393a434445464748494a535455565758595a636465666768696a737475767778797a838485868788898a92939495969798999aa2a3a4a5a6a7a8a9aab2b3b4b5b6b7b8b9bac2c3c4c5c6c7c8c9cad2d3d4d5d6d7d8d9dae1e2e3e4e5e6e7e8e9eaf1f2f3f4f5f6f7f8f9faffc4001f0100030101010101010101010000000000000102030405060708090a0bffc400b51100020102040403040705040400010277000102031104052131061241510761711322328108144291a1b1c109233352f0156272d10a162434e125f11718191a262728292a35363738393a434445464748494a535455565758595a636465666768696a737475767778797a82838485868788898a92939495969798999aa2a3a4a5a6a7a8a9aab2b3b4b5b6b7b8b9bac2c3c4c5c6c7c8c9cad2d3d4d5d6d7d8d9dae2e3e4e5e6e7e8e9eaf2f3f4f5f6f7f8f9faffda000c03010002110311003f00fd53a00fffd9"),
            }
        for encoding, data in examples.items():
            e = get_image_type(data)
            assert encoding==e, "expected %s got %s" % (encoding, e)

    def test_bad_header(self):
        for x in (b"", None, b"abc", b"\x99PNG", b"RIF", b"RIFF&\x00\x00\x00WEBp", binascii.unhexlify("fe0000000000000000000000000000000000000000000000000000")):
            e = get_image_type(x)
            assert e is None, "wrongly detected %s as %s" % (x, e)


def main():
    unittest.main()

if __name__ == '__main__':
    main()
