# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=4 sw=4 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2004-2008,2010,2015 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python 3 (>= 3.7)
# Project  : Cedar Backup, release 3
# Purpose  : Tests filesystem-related classes.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

########################################################################
# Module documentation
########################################################################

"""
Unit tests for CedarBackup3/filesystem.py.

Test Notes
==========

   This module contains individual tests for each of the classes implemented in
   filesystem.py: FilesystemList, BackupFileList and PurgeItemList.

   The BackupFileList and PurgeItemList classes inherit from FilesystemList,
   and the FilesystemList class itself inherits from the standard Python list
   class.  For the most part, I won't spend time testing inherited
   functionality, especially if it's already been tested.  However, I do test
   some of the base list functionality just to ensure that the inheritence has
   been constructed properly and everything seems to work as expected.

   You may look at this code and ask, "Why all of the checks that XXX is in
   list YYY?  Why not just compare what we got to a known list?"  The answer is
   that the order of the list is not significant, only its contents.  We can't
   be positive about the order in which we recurse a directory, but we do need
   to make sure that everything we expect is in the list and nothing more.  We
   do this by checking the count if items and then making sure that exactly
   that many known items exist in the list.

   This file is ridiculously long, almost too long to be worked with easily.  I
   really should split it up into smaller files, but I like having a 1:1
   relationship between a module and its test.

   These tests do pass on Windows, but I've taken the easy way out.  In most cases,
   rather than establising new expected results, I've excluded the tests that require
   soft links or other UNIX-specific behaviors.

Naming Conventions
==================

   I prefer to avoid large unit tests which validate more than one piece of
   functionality.  Instead, I create lots of very small tests that each
   validate one specific thing.  These small tests are then named with an index
   number, yielding something like ``testAddDir_001`` or ``testValidate_023``.
   Each method then has a docstring describing what it's supposed to
   accomplish.  I feel that this makes it easier to judge the extent of a
   problem when one exists.

Full vs. Reduced Tests
======================

   All of the tests in this module are considered safe to be run in an average
   build environment.  There is a no need to use a FILESYSTEMTESTS_FULL
   environment variable to provide a "reduced feature set" test suite as for
   some of the other test modules.

@author Kenneth J. Pronovici <pronovic@ieee.org>
"""


########################################################################
# Import modules and do runtime validations
########################################################################

import hashlib
import os
import tarfile
import tempfile
import unittest

from CedarBackup3.filesystem import BackupFileList, FilesystemList, PurgeItemList, compareContents, normalizeDir
from CedarBackup3.testutil import (
    buildPath,
    changeFileAge,
    configureLogging,
    extractTar,
    failUnlessAssignRaises,
    findResources,
    platformMacOsX,
    platformSupportsLinks,
    randomFilename,
    removedir,
)
from CedarBackup3.util import encodePath

#######################################################################
# Module-wide configuration and constants
#######################################################################

DATA_DIRS = ["./data", "./tests/data"]
RESOURCES = [
    "tree1.tar.gz",
    "tree2.tar.gz",
    "tree3.tar.gz",
    "tree4.tar.gz",
    "tree5.tar.gz",
    "tree6.tar.gz",
    "tree7.tar.gz",
    "tree8.tar.gz",
    "tree9.tar.gz",
    "tree10.tar.gz",
    "tree11.tar.gz",
    "tree12.tar.gz",
    "tree13.tar.gz",
    "tree22.tar.gz",
]

INVALID_FILE = "bogus"  # This file name should never exist
NOMATCH_PATH = "/something"  # This path should never match something we put in a file list
NOMATCH_BASENAME = "something"  # This basename should never match something we put in a file list
NOMATCH_PATTERN = "pattern"  # This pattern should never match something we put in a file list

AGE_1_HOUR = 1 * 60 * 60  # in seconds
AGE_2_HOURS = 2 * 60 * 60  # in seconds
AGE_12_HOURS = 12 * 60 * 60  # in seconds
AGE_23_HOURS = 23 * 60 * 60  # in seconds
AGE_24_HOURS = 24 * 60 * 60  # in seconds
AGE_25_HOURS = 25 * 60 * 60  # in seconds
AGE_47_HOURS = 47 * 60 * 60  # in seconds
AGE_48_HOURS = 48 * 60 * 60  # in seconds
AGE_49_HOURS = 49 * 60 * 60  # in seconds

#######################################################################
# Test Case Classes
#######################################################################

###########################
# TestFilesystemList class
###########################


class TestFilesystemList(unittest.TestCase):

    """Tests for the FilesystemList class."""

    ################
    # Setup methods
    ################

    @classmethod
    def setUpClass(cls):
        configureLogging()

    def setUp(self):
        try:
            self.tmpdir = tempfile.mkdtemp()
            self.resources = findResources(RESOURCES, DATA_DIRS)
        except Exception as e:
            self.fail(e)

    def tearDown(self):
        try:
            removedir(self.tmpdir)
        except:
            pass

    ##################
    # Utility methods
    ##################

    def extractTar(self, tarname):
        """Extracts a tarfile with a particular name."""
        extractTar(self.tmpdir, self.resources["%s.tar.gz" % tarname])

    def buildPath(self, components):
        """Builds a complete search path from a list of components."""
        components.insert(0, self.tmpdir)
        return buildPath(components)

    def pathPattern(self, path):
        """Returns properly-escaped regular expression pattern matching the indicated path."""
        return ".*%s.*" % path.replace("\\", "\\\\")

    def failUnlessAssignRaises(self, exception, obj, prop, value):
        """Equivalent of :any:`failUnlessRaises`, but used for property assignments instead."""
        failUnlessAssignRaises(self, exception, obj, prop, value)

    ############################
    # Test attribute assignment
    ############################

    def testAssignment_001(self):
        """
        Test assignment of excludeFiles attribute, true values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeFiles)
        fsList.excludeFiles = True
        self.assertEqual(True, fsList.excludeFiles)
        fsList.excludeFiles = [
            1,
        ]
        self.assertEqual(True, fsList.excludeFiles)

    def testAssignment_002(self):
        """
        Test assignment of excludeFiles attribute, false values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeFiles)
        fsList.excludeFiles = False
        self.assertEqual(False, fsList.excludeFiles)
        fsList.excludeFiles = []
        self.assertEqual(False, fsList.excludeFiles)

    def testAssignment_003(self):
        """
        Test assignment of excludeLinks attribute, true values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeLinks)
        fsList.excludeLinks = True
        self.assertEqual(True, fsList.excludeLinks)
        fsList.excludeLinks = [
            1,
        ]
        self.assertEqual(True, fsList.excludeLinks)

    def testAssignment_004(self):
        """
        Test assignment of excludeLinks attribute, false values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeLinks)
        fsList.excludeLinks = False
        self.assertEqual(False, fsList.excludeLinks)
        fsList.excludeLinks = []
        self.assertEqual(False, fsList.excludeLinks)

    def testAssignment_005(self):
        """
        Test assignment of excludeDirs attribute, true values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeDirs)
        fsList.excludeDirs = True
        self.assertEqual(True, fsList.excludeDirs)
        fsList.excludeDirs = [
            1,
        ]
        self.assertEqual(True, fsList.excludeDirs)

    def testAssignment_006(self):
        """
        Test assignment of excludeDirs attribute, false values.
        """
        fsList = FilesystemList()
        self.assertEqual(False, fsList.excludeDirs)
        fsList.excludeDirs = False
        self.assertEqual(False, fsList.excludeDirs)
        fsList.excludeDirs = []
        self.assertEqual(False, fsList.excludeDirs)

    def testAssignment_007(self):
        """
        Test assignment of ignoreFile attribute.
        """
        fsList = FilesystemList()
        self.assertEqual(None, fsList.ignoreFile)
        fsList.ignoreFile = "ken"
        self.assertEqual("ken", fsList.ignoreFile)
        fsList.ignoreFile = None
        self.assertEqual(None, fsList.ignoreFile)

    def testAssignment_008(self):
        """
        Test assignment of excludePaths attribute.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList.excludePaths)
        fsList.excludePaths = None
        self.assertEqual([], fsList.excludePaths)
        fsList.excludePaths = [
            "/path/to/something/absolute",
        ]
        self.assertEqual(["/path/to/something/absolute"], fsList.excludePaths)
        fsList.excludePaths = [
            "/path/to/something/absolute",
            "/path/to/something/else",
        ]
        self.assertEqual(["/path/to/something/absolute", "/path/to/something/else"], fsList.excludePaths)
        self.failUnlessAssignRaises(ValueError, fsList, "excludePaths", ["path/to/something/relative"])
        self.failUnlessAssignRaises(
            ValueError, fsList, "excludePaths", ["/path/to/something/absolute", "path/to/something/relative"]
        )
        fsList.excludePaths = [
            "/path/to/something/absolute",
        ]
        self.assertEqual(["/path/to/something/absolute"], fsList.excludePaths)
        fsList.excludePaths.insert(0, "/ken")
        self.assertEqual(["/ken", "/path/to/something/absolute"], fsList.excludePaths)
        fsList.excludePaths.append("/file")
        self.assertEqual(["/ken", "/path/to/something/absolute", "/file"], fsList.excludePaths)
        fsList.excludePaths.extend(["/one", "/two"])
        self.assertEqual(["/ken", "/path/to/something/absolute", "/file", "/one", "/two"], fsList.excludePaths)
        fsList.excludePaths = [
            "/path/to/something/absolute",
        ]
        self.assertRaises(ValueError, fsList.excludePaths.insert, 0, "path/to/something/relative")
        self.assertRaises(ValueError, fsList.excludePaths.append, "path/to/something/relative")
        self.assertRaises(ValueError, fsList.excludePaths.extend, ["path/to/something/relative"])

    def testAssignment_009(self):
        """
        Test assignment of excludePatterns attribute.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList.excludePatterns)
        fsList.excludePatterns = None
        self.assertEqual([], fsList.excludePatterns)
        fsList.excludePatterns = [
            r".*\.jpg",
        ]
        self.assertEqual([r".*\.jpg"], fsList.excludePatterns)
        fsList.excludePatterns = [
            r".*\.jpg",
            "[a-zA-Z0-9]*",
        ]
        self.assertEqual([r".*\.jpg", "[a-zA-Z0-9]*"], fsList.excludePatterns)
        self.failUnlessAssignRaises(ValueError, fsList, "excludePatterns", ["*.jpg"])
        self.failUnlessAssignRaises(ValueError, fsList, "excludePatterns", ["*.jpg", "[a-zA-Z0-9]*"])
        fsList.excludePatterns = [
            r".*\.jpg",
        ]
        self.assertEqual([r".*\.jpg"], fsList.excludePatterns)
        fsList.excludePatterns.insert(0, "ken")
        self.assertEqual(["ken", r".*\.jpg"], fsList.excludePatterns)
        fsList.excludePatterns.append("pattern")
        self.assertEqual(["ken", r".*\.jpg", "pattern"], fsList.excludePatterns)
        fsList.excludePatterns.extend(["one", "two"])
        self.assertEqual(["ken", r".*\.jpg", "pattern", "one", "two"], fsList.excludePatterns)
        fsList.excludePatterns = [
            r".*\.jpg",
        ]
        self.assertRaises(ValueError, fsList.excludePatterns.insert, 0, "*.jpg")
        self.assertEqual([r".*\.jpg"], fsList.excludePatterns)
        self.assertRaises(ValueError, fsList.excludePatterns.append, "*.jpg")
        self.assertEqual([r".*\.jpg"], fsList.excludePatterns)
        self.assertRaises(ValueError, fsList.excludePatterns.extend, ["*.jpg"])
        self.assertEqual([r".*\.jpg"], fsList.excludePatterns)

    def testAssignment_010(self):
        """
        Test assignment of excludeBasenamePatterns attribute.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns = None
        self.assertEqual([], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns = [
            r".*\.jpg",
        ]
        self.assertEqual([r".*\.jpg"], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns = [
            r".*\.jpg",
            "[a-zA-Z0-9]*",
        ]
        self.assertEqual([r".*\.jpg", "[a-zA-Z0-9]*"], fsList.excludeBasenamePatterns)
        self.failUnlessAssignRaises(ValueError, fsList, "excludeBasenamePatterns", ["*.jpg"])
        self.failUnlessAssignRaises(ValueError, fsList, "excludeBasenamePatterns", ["*.jpg", "[a-zA-Z0-9]*"])
        fsList.excludeBasenamePatterns = [
            r".*\.jpg",
        ]
        self.assertEqual([r".*\.jpg"], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns.insert(0, "ken")
        self.assertEqual(["ken", r".*\.jpg"], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns.append("pattern")
        self.assertEqual(["ken", r".*\.jpg", "pattern"], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns.extend(["one", "two"])
        self.assertEqual(["ken", r".*\.jpg", "pattern", "one", "two"], fsList.excludeBasenamePatterns)
        fsList.excludeBasenamePatterns = [
            r".*\.jpg",
        ]
        self.assertRaises(ValueError, fsList.excludeBasenamePatterns.insert, 0, "*.jpg")
        self.assertEqual([r".*\.jpg"], fsList.excludeBasenamePatterns)
        self.assertRaises(ValueError, fsList.excludeBasenamePatterns.append, "*.jpg")
        self.assertEqual([r".*\.jpg"], fsList.excludeBasenamePatterns)
        self.assertRaises(ValueError, fsList.excludeBasenamePatterns.extend, ["*.jpg"])
        self.assertEqual([r".*\.jpg"], fsList.excludeBasenamePatterns)

    ################################
    # Test basic list functionality
    ################################

    def testBasic_001(self):
        """
        Test the append() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("a")
        self.assertEqual(["a"], fsList)
        fsList.append("b")
        self.assertEqual(["a", "b"], fsList)

    def testBasic_002(self):
        """
        Test the insert() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.insert(0, "a")
        self.assertEqual(["a"], fsList)
        fsList.insert(0, "b")
        self.assertEqual(["b", "a"], fsList)

    def testBasic_003(self):
        """
        Test the remove() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.insert(0, "a")
        fsList.insert(0, "b")
        self.assertEqual(["b", "a"], fsList)
        fsList.remove("a")
        self.assertEqual(["b"], fsList)
        fsList.remove("b")
        self.assertEqual([], fsList)

    def testBasic_004(self):
        """
        Test the pop() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("a")
        fsList.append("b")
        fsList.append("c")
        fsList.append("d")
        fsList.append("e")
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)
        self.assertEqual("e", fsList.pop())
        self.assertEqual(["a", "b", "c", "d"], fsList)
        self.assertEqual("d", fsList.pop())
        self.assertEqual(["a", "b", "c"], fsList)
        self.assertEqual("c", fsList.pop())
        self.assertEqual(["a", "b"], fsList)
        self.assertEqual("b", fsList.pop())
        self.assertEqual(["a"], fsList)
        self.assertEqual("a", fsList.pop())
        self.assertEqual([], fsList)

    def testBasic_005(self):
        """
        Test the count() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("a")
        fsList.append("b")
        fsList.append("c")
        fsList.append("d")
        fsList.append("e")
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)
        self.assertEqual(1, fsList.count("a"))

    def testBasic_006(self):
        """
        Test the index() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("a")
        fsList.append("b")
        fsList.append("c")
        fsList.append("d")
        fsList.append("e")
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)
        self.assertEqual(2, fsList.index("c"))

    def testBasic_007(self):
        """
        Test the reverse() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("a")
        fsList.append("b")
        fsList.append("c")
        fsList.append("d")
        fsList.append("e")
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)
        fsList.reverse()
        self.assertEqual(["e", "d", "c", "b", "a"], fsList)
        fsList.reverse()
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)

    def testBasic_008(self):
        """
        Test the sort() method.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("e")
        fsList.append("d")
        fsList.append("c")
        fsList.append("b")
        fsList.append("a")
        self.assertEqual(["e", "d", "c", "b", "a"], fsList)
        fsList.sort()
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)
        fsList.sort()
        self.assertEqual(["a", "b", "c", "d", "e"], fsList)

    def testBasic_009(self):
        """
        Test slicing.
        """
        fsList = FilesystemList()
        self.assertEqual([], fsList)
        fsList.append("e")
        fsList.append("d")
        fsList.append("c")
        fsList.append("b")
        fsList.append("a")
        self.assertEqual(["e", "d", "c", "b", "a"], fsList)
        self.assertEqual(["e", "d", "c", "b", "a"], fsList[:])
        self.assertEqual(["e", "d", "c", "b", "a"], fsList[0:])
        self.assertEqual("e", fsList[0])
        self.assertEqual("a", fsList[4])
        self.assertEqual(["d", "c", "b"], fsList[1:4])

    #################
    # Test addFile()
    #################

    def testAddFile_001(self):
        """
        Attempt to add a file that doesn't exist; no exclusions.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_002(self):
        """
        Attempt to add a directory; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_003(self):
        """
        Attempt to add a soft link; no exclusions.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_004(self):
        """
        Attempt to add an existing file; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_005(self):
        """
        Attempt to add a file that doesn't exist; excludeFiles set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_006(self):
        """
        Attempt to add a directory; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_007(self):
        """
        Attempt to add a soft link; excludeFiles set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_008(self):
        """
        Attempt to add an existing file; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddFile_009(self):
        """
        Attempt to add a file that doesn't exist; excludeDirs set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_010(self):
        """
        Attempt to add a directory; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_011(self):
        """
        Attempt to add a soft link; excludeDirs set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_012(self):
        """
        Attempt to add an existing file; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_013(self):
        """
        Attempt to add a file that doesn't exist; excludeLinks set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_014(self):
        """
        Attempt to add a directory; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addFile, path)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddFile_015(self):
        """
        Attempt to add a soft link; excludeLinks set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_016(self):
        """
        Attempt to add an existing file; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_017(self):
        """
        Attempt to add a file that doesn't exist; with excludePaths including the
        path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_018(self):
        """
        Attempt to add a directory; with excludePaths including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_019(self):
        """
        Attempt to add a soft link; with excludePaths including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_020(self):
        """
        Attempt to add an existing file; with excludePaths including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddFile_021(self):
        """
        Attempt to add a file that doesn't exist; with excludePaths not including
        the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_022(self):
        """
        Attempt to add a directory; with excludePaths not including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_023(self):
        """
        Attempt to add a soft link; with excludePaths not including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_024(self):
        """
        Attempt to add an existing file; with excludePaths not including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_025(self):
        """
        Attempt to add a file that doesn't exist; with excludePatterns matching
        the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_026(self):
        """
        Attempt to add a directory; with excludePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_027(self):
        """
        Attempt to add a soft link; with excludePatterns matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_028(self):
        """
        Attempt to add an existing file; with excludePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddFile_029(self):
        """
        Attempt to add a file that doesn't exist; with excludePatterns not
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_030(self):
        """
        Attempt to add a directory; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_031(self):
        """
        Attempt to add a soft link; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_032(self):
        """
        Attempt to add an existing file; with excludePatterns not matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_033(self):
        """
        Attempt to add an invalid link (i.e. a link that points to something that
        doesn't exist).
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10", "link001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_034(self):
        """
        Attempt to add a file that has spaces in its name.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11", "file with spaces"])
        fsList = FilesystemList()
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_035(self):
        """
        Attempt to add a UTF-8 file.
        """
        self.extractTar("tree12")
        path = self.buildPath(["tree12", "unicode", encodePath(b"\xe2\x99\xaa\xe2\x99\xac")])
        fsList = FilesystemList()
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddFile_036(self):
        """
        Attempt to add a file that doesn't exist; with excludeBasenamePatterns
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [INVALID_FILE]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_037(self):
        """
        Attempt to add a directory; with excludeBasenamePatterns matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "dir001",
        ]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_038(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_039(self):
        """
        Attempt to add an existing file; with excludeBasenamePatterns matching
        the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "file001",
        ]
        count = fsList.addFile(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddFile_040(self):
        """
        Attempt to add a file that doesn't exist; with excludeBasenamePatterns
        not matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_041(self):
        """
        Attempt to add a directory; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_042(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePaths = [NOMATCH_BASENAME]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePaths = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addFile, path)

    def testAddFile_043(self):
        """
        Attempt to add an existing file; with excludeBasenamePatterns not
        matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addFile(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    ################
    # Test addDir()
    ################

    def testAddDir_001(self):
        """
        Attempt to add a directory that doesn't exist; no exclusions.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_002(self):
        """
        Attempt to add a file; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_003(self):
        """
        Attempt to add a soft link; no exclusions.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_004(self):
        """
        Attempt to add an existing directory; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_005(self):
        """
        Attempt to add a directory that doesn't exist; excludeFiles set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_006(self):
        """
        Attempt to add a file; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_007(self):
        """
        Attempt to add a soft link; excludeFiles set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_008(self):
        """
        Attempt to add an existing directory; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_009(self):
        """
        Attempt to add a directory that doesn't exist; excludeDirs set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_010(self):
        """
        Attempt to add a file; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_011(self):
        """
        Attempt to add a soft link; excludeDirs set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_012(self):
        """
        Attempt to add an existing directory; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_013(self):
        """
        Attempt to add a directory that doesn't exist; excludeLinks set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_014(self):
        """
        Attempt to add a file; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDir, path)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDir_015(self):
        """
        Attempt to add a soft link; excludeLinks set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_016(self):
        """
        Attempt to add an existing directory; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_017(self):
        """
        Attempt to add a directory that doesn't exist; with excludePaths
        including the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_018(self):
        """
        Attempt to add a file; with excludePaths including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_019(self):
        """
        Attempt to add a soft link; with excludePaths including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_020(self):
        """
        Attempt to add an existing directory; with excludePaths including the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_021(self):
        """
        Attempt to add a directory that doesn't exist; with excludePaths not
        including the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_022(self):
        """
        Attempt to add a file; with excludePaths not including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_023(self):
        """
        Attempt to add a soft link; with excludePaths not including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_024(self):
        """
        Attempt to add an existing directory; with excludePaths not including the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_025(self):
        """
        Attempt to add a directory that doesn't exist; with excludePatterns
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_026(self):
        """
        Attempt to add a file; with excludePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_027(self):
        """
        Attempt to add a soft link; with excludePatterns matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_028(self):
        """
        Attempt to add an existing directory; with excludePatterns matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_029(self):
        """
        Attempt to add a directory that doesn't exist; with excludePatterns not
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_030(self):
        """
        Attempt to add a file; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_031(self):
        """
        Attempt to add a soft link; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_032(self):
        """
        Attempt to add an existing directory; with excludePatterns not matching
        the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_033(self):
        """
        Attempt to add an invalid link (i.e. a link that points to something that
        doesn't exist).
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10", "link001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_034(self):
        """
        Attempt to add a directory that has spaces in its name.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11", "dir with spaces"])
        fsList = FilesystemList()
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_035(self):
        """
        Attempt to add a directory that doesn't exist; with
        excludeBasenamePatterns matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [INVALID_FILE]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_036(self):
        """
        Attempt to add a file; with excludeBasenamePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "file001",
        ]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_037(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_038(self):
        """
        Attempt to add an existing directory; with excludeBasenamePatterns
        matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "dir001",
        ]
        count = fsList.addDir(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDir_039(self):
        """
        Attempt to add a directory that doesn't exist; with
        excludeBasenamePatterns not matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_040(self):
        """
        Attempt to add a file; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addDir, path)

    def testAddDir_041(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addDir, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDir_042(self):
        """
        Attempt to add an existing directory; with excludeBasenamePatterns not
        matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    ########################
    # Test addDirContents()
    ########################

    def testAddDirContents_001(self):
        """
        Attempt to add a directory that doesn't exist; no exclusions.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_002(self):
        """
        Attempt to add a file; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_003(self):
        """
        Attempt to add a soft link; no exclusions.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        count = fsList.addDir(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_004(self):
        """
        Attempt to add an empty directory containing ignore file; no exclusions.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_005(self):
        """
        Attempt to add an empty directory; no exclusions.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_006(self):
        """
        Attempt to add an non-empty directory containing ignore file; no
        exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_007(self):
        """
        Attempt to add an non-empty directory; no exclusions.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "link001"]) in fsList)

    def testAddDirContents_008(self):
        """
        Attempt to add a directory that doesn't exist; excludeFiles set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_009(self):
        """
        Attempt to add a file; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_010(self):
        """
        Attempt to add a soft link; excludeFiles set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_011(self):
        """
        Attempt to add an empty directory containing ignore file; excludeFiles set.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_012(self):
        """
        Attempt to add an empty directory; excludeFiles set.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_013(self):
        """
        Attempt to add an non-empty directory containing ignore file; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_014(self):
        """
        Attempt to add an non-empty directory; excludeFiles set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(5, count)
        self.assertEqual(5, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)

    def testAddDirContents_015(self):
        """
        Attempt to add a directory that doesn't exist; excludeDirs set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_016(self):
        """
        Attempt to add a file; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_017(self):
        """
        Attempt to add a soft link; excludeDirs set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_018(self):
        """
        Attempt to add an empty directory containing ignore file; excludeDirs set.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_019(self):
        """
        Attempt to add an empty directory; excludeDirs set.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_020(self):
        """
        Attempt to add an non-empty directory containing ignore file; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_021(self):
        """
        Attempt to add an non-empty directory; excludeDirs set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(3, count)
        self.assertEqual(3, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "link001"]) in fsList)

    def testAddDirContents_023(self):
        """
        Attempt to add a directory that doesn't exist; excludeLinks set.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_024(self):
        """
        Attempt to add a file; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_025(self):
        """
        Attempt to add a soft link; excludeLinks set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDirContents, path)

        if platformSupportsLinks():
            path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
            fsList = FilesystemList()
            fsList.excludeLinks = True
            count = fsList.addDirContents(path)
            self.assertEqual(0, count)
            self.assertEqual([], fsList)

    def testAddDirContents_026(self):
        """
        Attempt to add an empty directory containing ignore file; excludeLinks set.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeLinks = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_027(self):
        """
        Attempt to add an empty directory; excludeLinks set.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertTrue(self.buildPath(["tree8", "dir001"]) in fsList)

    def testAddDirContents_028(self):
        """
        Attempt to add an non-empty directory containing ignore file; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeLinks = True
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_029(self):
        """
        Attempt to add an non-empty directory; excludeLinks set.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDirContents(path)
        self.assertEqual(7, count)
        self.assertEqual(7, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)

    def testAddDirContents_030(self):
        """
        Attempt to add a directory that doesn't exist; with excludePaths
        including the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_031(self):
        """
        Attempt to add a file; with excludePaths including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_032(self):
        """
        Attempt to add a soft link; with excludePaths including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_033(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludePaths including the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_034(self):
        """
        Attempt to add an empty directory; with excludePaths including the path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_035(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludePaths including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_036(self):
        """
        Attempt to add an non-empty directory; with excludePaths including the
        main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_037(self):
        """
        Attempt to add a directory that doesn't exist; with excludePaths not
        including the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_038(self):
        """
        Attempt to add a file; with excludePaths not including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_039(self):
        """
        Attempt to add a soft link; with excludePaths not including the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_040(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludePaths not including the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_041(self):
        """
        Attempt to add an empty directory; with excludePaths not including the
        path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_042(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludePaths not including the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_043(self):
        """
        Attempt to add an non-empty directory; with excludePaths not including
        the main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePaths = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "link001"]) in fsList)

    def testAddDirContents_044(self):
        """
        Attempt to add a directory that doesn't exist; with excludePatterns
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_045(self):
        """
        Attempt to add a file; with excludePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_046(self):
        """
        Attempt to add a soft link; with excludePatterns matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_047(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludePatterns matching the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_048(self):
        """
        Attempt to add an empty directory; with excludePatterns matching the
        path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_049(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_050(self):
        """
        Attempt to add an non-empty directory; with excludePatterns matching the
        main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_051(self):
        """
        Attempt to add a directory that doesn't exist; with excludePatterns not
        matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_052(self):
        """
        Attempt to add a file; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_053(self):
        """
        Attempt to add a soft link; with excludePatterns not matching the path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_054(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludePatterns not matching the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_055(self):
        """
        Attempt to add an empty directory; with excludePatterns not matching the
        path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_056(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludePatterns not matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_057(self):
        """
        Attempt to add an non-empty directory; with excludePatterns not matching
        the main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludePatterns = [NOMATCH_PATH]
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "link001"]) in fsList)

    def testAddDirContents_058(self):
        """
        Attempt to add a large tree with no exclusions.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(136, count)
        self.assertEqual(136, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_059(self):
        """
        Attempt to add a large tree, with excludeFiles set.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDirContents(path)
        self.assertEqual(42, count)
        self.assertEqual(42, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_060(self):
        """
        Attempt to add a large tree, with excludeDirs set.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDirContents(path)
        self.assertEqual(94, count)
        self.assertEqual(94, len(fsList))
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_061(self):
        """
        Attempt to add a large tree, with excludeLinks set.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDirContents(path)
        self.assertEqual(96, count)
        self.assertEqual(96, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)

    def testAddDirContents_062(self):
        """
        Attempt to add a large tree, with excludePaths set to exclude some entries.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludePaths = [
            self.buildPath(["tree6", "dir001", "dir002"]),
            self.buildPath(["tree6", "dir002", "dir001", "dir001"]),
            self.buildPath(["tree6", "dir003", "dir002", "file001"]),
            self.buildPath(["tree6", "dir003", "dir002", "file002"]),
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(125, count)
        self.assertEqual(125, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_063(self):
        """
        Attempt to add a large tree, with excludePatterns set to exclude some entries.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludePatterns = [".*file001.*", r".*tree6\/dir002\/dir001.*"]
        count = fsList.addDirContents(path)
        self.assertEqual(108, count)
        self.assertEqual(108, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_064(self):
        """
        Attempt to add a large tree, with ignoreFile set to exclude some directories.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        count = fsList.addDirContents(path)
        self.assertEqual(79, count)
        self.assertEqual(79, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_065(self):
        """
        Attempt to add a link to a file.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9", "dir002", "link003"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_066(self):
        """
        Attempt to add a link to a directory (which should add its contents).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9", "link002"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(9, count)
        self.assertEqual(9, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link004"]) in fsList)

    def testAddDirContents_067(self):
        """
        Attempt to add an invalid link (i.e. a link that points to something that
        doesn't exist).
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10", "link001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_068(self):
        """
        Attempt to add directory containing an invalid link (i.e. a link that
        points to something that doesn't exist).
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(3, count)
        self.assertEqual(3, len(fsList))
        self.assertTrue(self.buildPath(["tree10"]) in fsList)
        self.assertTrue(self.buildPath(["tree10", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree10", "dir002"]) in fsList)

    def testAddDirContents_069(self):
        """
        Attempt to add a directory containing items with spaces.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testAddDirContents_070(self):
        """
        Attempt to add a directory which has a name containing spaces.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11", "dir with spaces"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(5, count)
        self.assertEqual(5, len(fsList))
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testAddDirContents_071(self):
        """
        Attempt to add a directory which has a UTF-8 filename in it.
        """
        self.extractTar("tree12")
        path = self.buildPath(["tree12", "unicode"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(6, count)
        self.assertEqual(6, len(fsList))
        self.assertTrue(self.buildPath(["tree12", "unicode"]) in fsList)
        self.assertTrue(self.buildPath(["tree12", "unicode", "README.strange-name"]) in fsList)
        self.assertTrue(self.buildPath(["tree12", "unicode", "utflist.long.gz"]) in fsList)
        self.assertTrue(self.buildPath(["tree12", "unicode", "utflist.cp437.gz"]) in fsList)
        self.assertTrue(self.buildPath(["tree12", "unicode", "utflist.short.gz"]) in fsList)
        self.assertTrue(self.buildPath(["tree12", "unicode", encodePath(b"\xe2\x99\xaa\xe2\x99\xac")]) in fsList)

    def testAddDirContents_072(self):
        """
        Attempt to add a directory which has several UTF-8 filenames in it.

        This test data was taken from Rick Lowe's problems around the release of
        v1.10.  I don't run the test for Darwin (Mac OS X) because the tarball
        isn't valid on that platform.

        All of the tests with unicode paths were incredibly painful to get
        working with Python 3, but these tests in particular were difficult,
        because character 0x82 is not a valid UTF-8 character.  The key is was to
        get the filename into the same encoding used by methods like os.listdir(),
        which uses a "surrogateescape" fallback for encoding filenames.  Once I
        switched encodePath to do the same thing, this test started passing.
        There's apparently no other way to represent filenames like this.
        """
        if not platformMacOsX():
            self.extractTar("tree13")
            path = self.buildPath(["tree13"])
            fsList = FilesystemList()
            count = fsList.addDirContents(path)
            self.assertEqual(11, count)
            self.assertEqual(11, len(fsList))
            self.assertTrue(self.buildPath(["tree13"]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"Les mouvements de r\x82forme.doc")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"l'\x82nonc\x82.sxw")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"l\x82onard - renvois et bibliographie.sxw")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"l\x82onard copie finale.sxw")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"l\x82onard de vinci - page titre.sxw")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"l\x82onard de vinci.sxw")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"Rammstein - B\x81ck Dich.mp3")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"megaherz - Glas Und Tr\x84nen.mp3")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"Megaherz - Mistst\x81ck.MP3")]) in fsList)
            self.assertTrue(self.buildPath(["tree13", encodePath(b"Rammstein - Mutter - B\x94se.mp3")]) in fsList)

    def testAddDirContents_073(self):
        """
        Attempt to add a large tree with recursive=False.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, recursive=False)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_074(self):
        """
        Attempt to add a directory that doesn't exist; with
        excludeBasenamePatterns matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [INVALID_FILE]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_075(self):
        """
        Attempt to add a file; with excludeBasenamePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "file001",
        ]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_076(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_077(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludeBasenamePatterns matching the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeBasenamePatterns = [
            "dir001",
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_078(self):
        """
        Attempt to add an empty directory; with excludeBasenamePatterns matching
        the path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "dir001",
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_079(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludeBasenamePatterns matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeBasenamePatterns = [
            "dir008",
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_080(self):
        """
        Attempt to add an non-empty directory; with excludeBasenamePatterns
        matching the main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "dir001",
        ]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_081(self):
        """
        Attempt to add a directory that doesn't exist; with
        excludeBasenamePatterns not matching the path.
        """
        path = self.buildPath([INVALID_FILE])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_082(self):
        """
        Attempt to add a file; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "file001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        self.assertRaises(ValueError, fsList.addDirContents, path)

    def testAddDirContents_083(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns not matching the
        path.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_PATH]
        self.assertRaises(ValueError, fsList.addDirContents, path)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_084(self):
        """
        Attempt to add an empty directory containing ignore file; with
        excludeBasenamePatterns not matching the path.
        """
        self.extractTar("tree7")
        path = self.buildPath(["tree7", "dir001"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_085(self):
        """
        Attempt to add an empty directory; with excludeBasenamePatterns not
        matching the path.
        """
        self.extractTar("tree8")
        path = self.buildPath(["tree8", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDirContents(path)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_086(self):
        """
        Attempt to add an non-empty directory containing ignore file; with
        excludeBasenamePatterns not matching the path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir008"])
        fsList = FilesystemList()
        fsList.ignoreFile = "ignore"
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDirContents(path)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_087(self):
        """
        Attempt to add an non-empty directory; with excludeBasenamePatterns not
        matching the main directory path.
        """
        self.extractTar("tree5")
        path = self.buildPath(["tree5", "dir001"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [NOMATCH_BASENAME]
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree5", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree5", "dir001", "link001"]) in fsList)

    def testAddDirContents_088(self):
        """
        Attempt to add a large tree, with excludeBasenamePatterns set to exclude
        some entries.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = ["file001", "dir001"]
        count = fsList.addDirContents(path)
        self.assertEqual(64, count)
        self.assertEqual(64, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_089(self):
        """
        Attempt to add a large tree with no exclusions, addSelf=True.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, addSelf=True)
        self.assertEqual(136, count)
        self.assertEqual(136, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    def testAddDirContents_090(self):
        """
        Attempt to add a large tree with no exclusions, addSelf=False.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, addSelf=False)
        self.assertEqual(135, count)
        self.assertEqual(135, len(fsList))
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_091(self):
        """
        Attempt to add a directory with linkDepth=1.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=1)
        self.assertEqual(165, count)
        self.assertEqual(165, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_092(self):
        """
        Attempt to add a directory with linkDepth=2.
        """
        self.extractTar("tree6")
        path = self.buildPath(["tree6"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=2)
        self.assertEqual(241, count)
        self.assertEqual(241, len(fsList))
        self.assertTrue(self.buildPath(["tree6"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir001", "link001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir001", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "dir003", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link002", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir002", "link005", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "dir003", "link004", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "ignore"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link002", "link001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree6", "link001"]) in fsList)

    def testAddDirContents_093(self):
        """
        Attempt to add a directory with linkDepth=0, dereference=False.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=0, dereference=False)
        self.assertEqual(12, count)
        self.assertEqual(12, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_094(self):
        """
        Attempt to add a directory with linkDepth=1, dereference=False.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=1, dereference=False)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_095(self):
        """
        Attempt to add a directory with linkDepth=2, dereference=False.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=2, dereference=False)
        self.assertEqual(20, count)
        self.assertEqual(20, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_096(self):
        """
        Attempt to add a directory with linkDepth=3, dereference=False.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=3, dereference=False)
        self.assertEqual(20, count)
        self.assertEqual(20, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "link001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003", "link002", "link002"]) in fsList)

    def testAddDirContents_097(self):
        """
        Attempt to add a directory with linkDepth=0, dereference=True.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=0, dereference=True)
        self.assertEqual(12, count)
        self.assertEqual(12, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_098(self):
        """
        Attempt to add a directory with linkDepth=1, dereference=True.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=1, dereference=True)
        self.assertEqual(20, count)
        self.assertEqual(20, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_099(self):
        """
        Attempt to add a directory with linkDepth=2, dereference=True.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=2, dereference=True)
        self.assertEqual(32, count)
        self.assertEqual(32, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "link002"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_100(self):
        """
        Attempt to add a directory with linkDepth=3, dereference=True.
        """
        self.extractTar("tree22")
        path = self.buildPath(["tree22", "dir003"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, linkDepth=3, dereference=True)
        self.assertEqual(35, count)
        self.assertEqual(35, len(fsList))
        self.assertTrue(self.buildPath(["tree22", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "dir001", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir003", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir002", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir004", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir005", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir006", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir007", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree22", "dir008", "file001"]) in fsList)

    def testAddDirContents_101(self):
        """
        Attempt to add a soft link; excludeFiles and dereference set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeFiles = True
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to dir003
        fsList = FilesystemList()
        fsList.excludeFiles = True
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(1, count)
        self.assertEqual([path], fsList)

    def testAddDirContents_102(self):
        """
        Attempt to add a soft link; excludeDirs and dereference set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeDirs = True
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to dir003
        fsList = FilesystemList()
        fsList.excludeDirs = True
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_103(self):
        """
        Attempt to add a soft link; excludeLinks and dereference set.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeLinks = True
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to dir003
        fsList = FilesystemList()
        fsList.excludeLinks = True
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_104(self):
        """
        Attempt to add a soft link; with excludePaths including the path,
        with dereference=True.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to dir003
        fsList = FilesystemList()
        fsList.excludePaths = [path]
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_105(self):
        """
        Attempt to add a soft link; with excludePatterns matching the path,
        with dereference=True.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to dir003
        fsList = FilesystemList()
        fsList.excludePatterns = [self.pathPattern(path)]
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    def testAddDirContents_106(self):
        """
        Attempt to add a link to a file, with dereference=True.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9", "dir002", "link003"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testAddDirContents_107(self):
        """
        Attempt to add a link to a directory (which should add its contents),
        with dereference=True.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9", "link002"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(13, count)
        self.assertEqual(13, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "dir001"]) in fsList)  # duplicated
        self.assertTrue(self.buildPath(["tree9", "link002", "dir002"]) in fsList)  # duplicated
        self.assertTrue(self.buildPath(["tree9", "link002", "file001"]) in fsList)  # duplicated
        self.assertTrue(self.buildPath(["tree9", "link002", "file002"]) in fsList)  # duplicated
        self.assertTrue(self.buildPath(["tree9", "link002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002", "link004"]) in fsList)

    def testAddDirContents_108(self):
        """
        Attempt to add an invalid link (i.e. a link that points to something that
        doesn't exist), and dereference=True.
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10", "link001"])
        fsList = FilesystemList()
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

    def testAddDirContents_109(self):
        """
        Attempt to add directory containing an invalid link (i.e. a link that
        points to something that doesn't exist), and dereference=True.
        """
        self.extractTar("tree10")
        path = self.buildPath(["tree10"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(3, count)
        self.assertEqual(3, len(fsList))
        self.assertTrue(self.buildPath(["tree10"]) in fsList)
        self.assertTrue(self.buildPath(["tree10", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree10", "dir002"]) in fsList)

    def testAddDirContents_110(self):
        """
        Attempt to add a soft link; with excludeBasenamePatterns matching the
        path, and dereference=True.
        """
        self.extractTar("tree5")

        path = self.buildPath(["tree5", "link001"])  # link to a file
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        self.assertRaises(ValueError, fsList.addDirContents, path, True, True, 1, True)

        path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
        fsList = FilesystemList()
        fsList.excludeBasenamePatterns = [
            "link001",
        ]
        count = fsList.addDirContents(path, True, True, 1, True)
        self.assertEqual(0, count)
        self.assertEqual([], fsList)

    #####################
    # Test removeFiles()
    #####################

    def testRemoveFiles_001(self):
        """
        Test with an empty list and a pattern of None.
        """
        fsList = FilesystemList()
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(0, count)

    def testRemoveFiles_002(self):
        """
        Test with an empty list and a non-empty pattern.
        """
        fsList = FilesystemList()
        count = fsList.removeFiles(pattern="pattern")
        self.assertEqual(0, count)
        self.assertRaises(ValueError, fsList.removeFiles, pattern="*.jpg")

    def testRemoveFiles_003(self):
        """
        Test with a non-empty list (files only) and a pattern of None.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(7, count)
        self.assertEqual(1, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)

    def testRemoveFiles_004(self):
        """
        Test with a non-empty list (directories only) and a pattern of None.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveFiles_005(self):
        """
        Test with a non-empty list (files and directories) and a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(44, count)
        self.assertEqual(37, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)

    def testRemoveFiles_006(self):
        """
        Test with a non-empty list (files, directories and links) and a pattern of None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(10, count)
        self.assertEqual(12, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveFiles_007(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=None)
        self.assertEqual(44, count)
        self.assertEqual(38, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)

    def testRemoveFiles_008(self):
        """
        Test with a non-empty list (spaces in path names) and a pattern of None.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveFiles_009(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches none of the files.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveFiles_010(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches none of the files.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveFiles_011(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches none of the files.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveFiles_012(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches none of the files.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveFiles_013(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches none of the files.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveFiles_014(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches none of the files.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveFiles_015(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches some of the files.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*tree1.*file00[67]")
        self.assertEqual(2, count)
        self.assertEqual(6, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)

    def testRemoveFiles_016(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches some of the files.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeFiles(pattern=".*tree2.*")
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveFiles_017(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches some of the files.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*tree4.*dir006.*")
        self.assertEqual(10, count)
        self.assertEqual(71, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveFiles_018(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches some of the files.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeFiles(pattern=".*tree9.*dir002.*")
        self.assertEqual(4, count)
        self.assertEqual(18, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveFiles_019(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches some of the files.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*dir001.*file002.*")
        self.assertEqual(1, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveFiles_020(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches some of the files.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeFiles(pattern=".*with spaces.*")
        self.assertEqual(6, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)

    def testRemoveFiles_021(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches anything.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(7, count)
        self.assertEqual(1, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)

    def testRemoveFiles_022(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches anything.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveFiles_023(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches anything.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(44, count)
        self.assertEqual(37, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)

    def testRemoveFiles_024(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches all of the files.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(10, count)
        self.assertEqual(12, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveFiles_025(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches all of the files.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(44, count)
        self.assertEqual(38, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)

    def testRemoveFiles_026(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches all of the files.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeFiles(pattern=".*")
        self.assertEqual(11, count)
        self.assertEqual(5, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)

    ####################
    # Test removeDirs()
    ####################

    def testRemoveDirs_001(self):
        """
        Test with an empty list and a pattern of None.
        """
        fsList = FilesystemList()
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(0, count)

    def testRemoveDirs_002(self):
        """
        Test with an empty list and a non-empty pattern.
        """
        fsList = FilesystemList()
        count = fsList.removeDirs(pattern="pattern")
        self.assertEqual(0, count)
        self.assertRaises(ValueError, fsList.removeDirs, pattern="*.jpg")

    def testRemoveDirs_003(self):
        """
        Test with a non-empty list (files only) and a pattern of None.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(1, count)
        self.assertEqual(7, len(fsList))
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveDirs_004(self):
        """
        Test with a non-empty list (directories only) and a pattern of None.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(11, count)
        self.assertEqual(0, len(fsList))

    def testRemoveDirs_005(self):
        """
        Test with a non-empty list (files and directories) and a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(37, count)
        self.assertEqual(44, len(fsList))
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_006(self):
        """
        Test with a non-empty list (files, directories and links) and a pattern of None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(12, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)

    def testRemoveDirs_007(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(37, count)
        self.assertEqual(45, len(fsList))
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_008(self):
        """
        Test with a non-empty list (spaces in path names) and a pattern of None.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeDirs(pattern=None)
        self.assertEqual(5, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveDirs_009(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches none of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveDirs_010(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveDirs_011(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_012(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveDirs_013(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_014(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveDirs_015(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches some of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*tree1.file00[67]")
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveDirs_016(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeDirs(pattern=".*dir0[012]0")
        self.assertEqual(1, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)

    def testRemoveDirs_017(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*dir001")
        self.assertEqual(9, count)
        self.assertEqual(72, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_018(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeDirs(pattern=".*tree9.*dir002.*")
        self.assertEqual(6, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveDirs_019(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*dir001")
        self.assertEqual(9, count)
        self.assertEqual(73, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_020(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeDirs(pattern=".*with spaces.*")
        self.assertEqual(1, count)
        self.assertEqual(15, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveDirs_021(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches all of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(1, count)
        self.assertEqual(7, len(fsList))
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveDirs_022(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(11, count)
        self.assertEqual(0, len(fsList))

    def testRemoveDirs_023(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(37, count)
        self.assertEqual(44, len(fsList))
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_024(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(12, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)

    def testRemoveDirs_025(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(37, count)
        self.assertEqual(45, len(fsList))
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveDirs_026(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeDirs(pattern=".*")
        self.assertEqual(5, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    #####################
    # Test removeLinks()
    #####################

    def testRemoveLinks_001(self):
        """
        Test with an empty list and a pattern of None.
        """
        fsList = FilesystemList()
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(0, count)

    def testRemoveLinks_002(self):
        """
        Test with an empty list and a non-empty pattern.
        """
        fsList = FilesystemList()
        count = fsList.removeLinks(pattern="pattern")
        self.assertEqual(0, count)
        self.assertRaises(ValueError, fsList.removeLinks, pattern="*.jpg")

    def testRemoveLinks_003(self):
        """
        Test with a non-empty list (files only) and a pattern of None.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveLinks_004(self):
        """
        Test with a non-empty list (directories only) and a pattern of None.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveLinks_005(self):
        """
        Test with a non-empty list (files and directories) and a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_006(self):
        """
        Test with a non-empty list (files, directories and links) and a pattern of None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(9, count)
        self.assertEqual(13, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)

    def testRemoveLinks_007(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a pattern of None.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_008(self):
        """
        Test with a non-empty list (spaces in path names) and a pattern of None.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeLinks(pattern=None)
        self.assertEqual(6, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)

    def testRemoveLinks_009(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches none of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveLinks_010(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveLinks_011(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveLinks_012(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveLinks_013(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveLinks_014(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveLinks_015(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches some of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*tree1.*file007")
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveLinks_016(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeLinks(pattern=".*tree2.*")
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveLinks_017(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*tree4.*dir006.*")
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_018(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeLinks(pattern=".*tree9.*dir002.*")
        self.assertEqual(4, count)
        self.assertEqual(18, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveLinks_019(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*tree4.*dir006.*")
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_020(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeLinks(pattern=".*with spaces.*")
        self.assertEqual(3, count)
        self.assertEqual(13, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)

    def testRemoveLinks_021(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches all of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveLinks_022(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveLinks_023(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_024(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(9, count)
        self.assertEqual(13, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)

    def testRemoveLinks_025(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveLinks_026(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeLinks(pattern=".*")
        self.assertEqual(6, count)
        self.assertEqual(10, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)

    #####################
    # Test removeMatch()
    #####################

    def testRemoveMatch_001(self):
        """
        Test with an empty list and a pattern of None.
        """
        fsList = FilesystemList()
        self.assertRaises(TypeError, fsList.removeMatch, pattern=None)

    def testRemoveMatch_002(self):
        """
        Test with an empty list and a non-empty pattern.
        """
        fsList = FilesystemList()
        count = fsList.removeMatch(pattern="pattern")
        self.assertEqual(0, count)
        self.assertRaises(ValueError, fsList.removeMatch, pattern="*.jpg")

    def testRemoveMatch_003(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches none of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveMatch_004(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveMatch_005(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveMatch_006(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches none of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveMatch_007(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches none of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveMatch_008(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches none of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    def testRemoveMatch_009(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches some of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*file00[135].*")
        self.assertEqual(3, count)
        self.assertEqual(5, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveMatch_010(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeMatch(pattern=".*dir00[2468].*")
        self.assertEqual(4, count)
        self.assertEqual(7, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveMatch_011(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*tree4.*dir006")
        self.assertEqual(18, count)
        self.assertEqual(63, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveMatch_012(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches some of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeMatch(pattern=".*file001.*")
        self.assertEqual(3, count)
        self.assertEqual(19, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveMatch_013(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches some of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*dir00[46].*")
        self.assertEqual(25, count)
        self.assertEqual(57, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveMatch_014(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches some of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeMatch(pattern=".*with spaces.*")
        self.assertEqual(7, count)
        self.assertEqual(9, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)

    def testRemoveMatch_015(self):
        """
        Test with a non-empty list (files only) and a non-empty pattern that
        matches all of them.
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(8, count)
        self.assertEqual(0, len(fsList))

    def testRemoveMatch_016(self):
        """
        Test with a non-empty list (directories only) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(11, count)
        self.assertEqual(0, len(fsList))

    def testRemoveMatch_017(self):
        """
        Test with a non-empty list (files and directories) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(81, count)
        self.assertEqual(0, len(fsList))

    def testRemoveMatch_019(self):
        """
        Test with a non-empty list (files, directories and links) and a non-empty
        pattern that matches all of them.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(22, count)
        self.assertEqual(0, len(fsList))

    def testRemoveMatch_020(self):
        """
        Test with a non-empty list (files and directories, some nonexistent) and
        a non-empty pattern that matches all of them.
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        fsList.append(self.buildPath(["tree4", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(82, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(82, count)
        self.assertEqual(0, len(fsList))

    def testRemoveMatch_021(self):
        """
        Test with a non-empty list (spaces in path names) and a non-empty pattern
        that matches all of them.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeMatch(pattern=".*")
        self.assertEqual(16, count)
        self.assertEqual(0, len(fsList))

    #######################
    # Test removeInvalid()
    #######################

    def testRemoveInvalid_001(self):
        """
        Test with an empty list.
        """
        fsList = FilesystemList()
        count = fsList.removeInvalid()
        self.assertEqual(0, count)

    def testRemoveInvalid_002(self):
        """
        Test with a non-empty list containing only invalid entries (some with spaces).
        """
        self.extractTar("tree9")
        fsList = FilesystemList()
        fsList.append(self.buildPath(["tree9", "%s-1" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-2" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-3" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-4" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", " %s 5  " % INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(5, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", " %s 5  " % INVALID_FILE]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(5, count)
        self.assertEqual(0, len(fsList))

    def testRemoveInvalid_003(self):
        """
        Test with a non-empty list containing only valid entries (files only).
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(0, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testRemoveInvalid_004(self):
        """
        Test with a non-empty list containing only valid entries (directories
        only).
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(0, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testRemoveInvalid_005(self):
        """
        Test with a non-empty list containing only valid entries (files and
        directories).
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(0, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testRemoveInvalid_006(self):
        """
        Test with a non-empty list containing only valid entries (files,
        directories and links).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(0, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveInvalid_007(self):
        """
        Test with a non-empty list containing valid and invalid entries (files,
        directories and links).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        fsList.append(self.buildPath(["tree9", "%s-1" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-2" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-3" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-4" % INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(26, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(4, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testRemoveInvalid_008(self):
        """
        Test with a non-empty list containing only valid entries (files,
        directories and links, some with spaces).
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        count = fsList.removeInvalid()
        self.assertEqual(0, count)
        self.assertEqual(16, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)

    ###################
    # Test normalize()
    ###################

    def testNormalize_001(self):
        """
        Test with an empty list.
        """
        fsList = FilesystemList()
        self.assertEqual(0, len(fsList))
        fsList.normalize()
        self.assertEqual(0, len(fsList))

    def testNormalize_002(self):
        """
        Test with a list containing one entry.
        """
        fsList = FilesystemList()
        fsList.append("one")
        self.assertEqual(1, len(fsList))
        fsList.normalize()
        self.assertEqual(1, len(fsList))
        self.assertTrue("one" in fsList)

    def testNormalize_003(self):
        """
        Test with a list containing two entries, no duplicates.
        """
        fsList = FilesystemList()
        fsList.append("one")
        fsList.append("two")
        self.assertEqual(2, len(fsList))
        fsList.normalize()
        self.assertEqual(2, len(fsList))
        self.assertTrue("one" in fsList)
        self.assertTrue("two" in fsList)

    def testNormalize_004(self):
        """
        Test with a list containing two entries, with duplicates.
        """
        fsList = FilesystemList()
        fsList.append("one")
        fsList.append("one")
        self.assertEqual(2, len(fsList))
        fsList.normalize()
        self.assertEqual(1, len(fsList))
        self.assertTrue("one" in fsList)

    def testNormalize_005(self):
        """
        Test with a list containing many entries, no duplicates.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        fsList.normalize()
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testNormalize_006(self):
        """
        Test with a list containing many entries, with duplicates.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(44, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        fsList.normalize()
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    ################
    # Test verify()
    ################

    def testVerify_001(self):
        """
        Test with an empty list.
        """
        fsList = FilesystemList()
        ok = fsList.verify()
        self.assertEqual(True, ok)

    def testVerify_002(self):
        """
        Test with a non-empty list containing only invalid entries.
        """
        self.extractTar("tree9")
        fsList = FilesystemList()
        fsList.append(self.buildPath(["tree9", "%s-1" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-2" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-3" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-4" % INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(4, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)
        ok = fsList.verify()
        self.assertEqual(False, ok)
        self.assertEqual(4, len(fsList))
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)

    def testVerify_003(self):
        """
        Test with a non-empty list containing only valid entries (files only).
        """
        self.extractTar("tree1")
        path = self.buildPath(["tree1"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(8, count)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(True, ok)
        self.assertEqual(8, len(fsList))
        self.assertTrue(self.buildPath(["tree1"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree1", "file007"]) in fsList)

    def testVerify_004(self):
        """
        Test with a non-empty list containing only valid entries (directories
        only).
        """
        self.extractTar("tree2")
        path = self.buildPath(["tree2"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(11, count)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(True, ok)
        self.assertEqual(11, len(fsList))
        self.assertTrue(self.buildPath(["tree2"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir009"]) in fsList)
        self.assertTrue(self.buildPath(["tree2", "dir010"]) in fsList)

    def testVerify_005(self):
        """
        Test with a non-empty list containing only valid entries (files and
        directories).
        """
        self.extractTar("tree4")
        path = self.buildPath(["tree4"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(81, count)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(True, ok)
        self.assertEqual(81, len(fsList))
        self.assertTrue(self.buildPath(["tree4"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir001", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir002", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir003", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir004", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "dir008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir005", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "dir005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file007"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file008"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file009"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "dir006", "file010"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file004"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file005"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file006"]) in fsList)
        self.assertTrue(self.buildPath(["tree4", "file007"]) in fsList)

    def testVerify_006(self):
        """
        Test with a non-empty list containing only valid entries (files,
        directories and links).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(True, ok)
        self.assertEqual(22, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testVerify_007(self):
        """
        Test with a non-empty list containing valid and invalid entries (files,
        directories and links).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(22, count)
        self.assertEqual(22, len(fsList))
        fsList.append(self.buildPath(["tree9", "%s-1" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-2" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-3" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree9", "%s-4" % INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(26, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(False, ok)
        self.assertEqual(26, len(fsList))
        self.assertTrue(self.buildPath(["tree9"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fsList)

    def testVerify_008(self):
        """
        Test with a non-empty list containing valid and invalid entries (some
        containing spaces).
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        fsList = FilesystemList()
        count = fsList.addDirContents(path)
        self.assertEqual(16, count)
        self.assertEqual(16, len(fsList))
        fsList.append(self.buildPath(["tree11", "dir with spaces", "%s-1" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree11", "dir with spaces", "%s-2" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree11", "dir with spaces", "%s-3" % INVALID_FILE]))  # file won't exist on disk
        fsList.append(self.buildPath(["tree11", "dir with spaces", "%s-4" % INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(20, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)
        ok = fsList.verify()
        self.assertEqual(False, ok)
        self.assertEqual(20, len(fsList))
        self.assertTrue(self.buildPath(["tree11"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-1" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-2" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-3" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "%s-4" % INVALID_FILE]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fsList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fsList)


###########################
# TestBackupFileList class
###########################


class TestBackupFileList(unittest.TestCase):

    """Tests for the BackupFileList class."""

    ################
    # Setup methods
    ################

    @classmethod
    def setUpClass(cls):
        configureLogging()

    def setUp(self):
        try:
            self.tmpdir = tempfile.mkdtemp()
            self.resources = findResources(RESOURCES, DATA_DIRS)
        except Exception as e:
            self.fail(e)

    def tearDown(self):
        try:
            removedir(self.tmpdir)
        except:
            pass

    ##################
    # Utility methods
    ##################

    def extractTar(self, tarname):
        """Extracts a tarfile with a particular name."""
        extractTar(self.tmpdir, self.resources["%s.tar.gz" % tarname])

    def buildPath(self, components):
        """Builds a complete search path from a list of components."""
        components.insert(0, self.tmpdir)
        return buildPath(components)

    def tarPath(self, components):
        """Builds a complete search path from a list of components, compatible with Python tar output."""
        result = self.buildPath(components)
        if result[0:1] == os.path.sep:
            return result[1:]
        return result

    def buildRandomPath(self, maxlength, extension):
        """Builds a complete, randomly-named search path."""
        maxlength -= len(self.tmpdir)
        maxlength -= len(extension)
        components = [
            self.tmpdir,
            randomFilename(maxlength, suffix=extension),
        ]
        return buildPath(components)

    ################
    # Test addDir()
    ################

    def testAddDir_001(self):
        """
        Test that function is overridden, no exclusions.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        if platformSupportsLinks():
            dirPath = self.buildPath(["tree5", "dir002", "link001"])
            count = backupList.addDir(dirPath)
            self.assertEqual(1, count)
            self.assertEqual([dirPath], backupList)

    def testAddDir_002(self):
        """
        Test that function is overridden, excludeFiles set.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        backupList.excludeFiles = True
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        if platformSupportsLinks():
            dirPath = self.buildPath(["tree5", "dir002", "link001"])
            count = backupList.addDir(dirPath)
            self.assertEqual(1, count)
            self.assertEqual([dirPath], backupList)

    def testAddDir_003(self):
        """
        Test that function is overridden, excludeDirs set.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        backupList.excludeDirs = True
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        dirPath = self.buildPath(["tree5", "dir002", "link001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))

    def testAddDir_004(self):
        """
        Test that function is overridden, excludeLinks set.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        backupList.excludeLinks = True
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        dirPath = self.buildPath(["tree5", "dir002", "link001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))

    def testAddDir_005(self):
        """
        Test that function is overridden, excludePaths set.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        backupList.excludePaths = [NOMATCH_PATH]
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        if platformSupportsLinks():
            dirPath = self.buildPath(["tree5", "dir002", "link001"])
            count = backupList.addDir(dirPath)
            self.assertEqual(1, count)
            self.assertEqual([dirPath], backupList)

    def testAddDir_006(self):
        """
        Test that function is overridden, excludePatterns set.

        Since this function calls the superclass by definition, we can
        skimp a bit on validation and only ensure that it seems to be
        overridden properly.
        """
        self.extractTar("tree5")
        backupList = BackupFileList()
        backupList.excludePatterns = [NOMATCH_PATH]
        dirPath = self.buildPath(["tree5", "dir001"])
        count = backupList.addDir(dirPath)
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        if platformSupportsLinks():
            dirPath = self.buildPath(["tree5", "dir002", "link001"])
            count = backupList.addDir(dirPath)
            self.assertEqual(1, count)
            self.assertEqual([dirPath], backupList)

    ###################
    # Test totalSize()
    ###################

    def testTotalSize_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        size = backupList.totalSize()
        self.assertEqual(0, size)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testTotalSize_002(self):
        """
        Test on a non-empty list containing only valid entries.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        size = backupList.totalSize()
        self.assertEqual(1116, size)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testTotalSize_004(self):
        """
        Test on a non-empty list (some containing spaces).
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        size = backupList.totalSize()
        self.assertEqual(1085, size)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testTotalSize_005(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        size = backupList.totalSize()
        self.assertEqual(1116, size)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testTotalSize_006(self):
        """
        Test on a non-empty list containing a non-existent file.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        size = backupList.totalSize()
        self.assertEqual(1116, size)

    #########################
    # Test generateSizeMap()
    #########################

    def testGenerateSizeMap_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        sizeMap = backupList.generateSizeMap()
        self.assertEqual(0, len(sizeMap))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSizeMap_002(self):
        """
        Test on a non-empty list containing only valid entries.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        sizeMap = backupList.generateSizeMap()
        self.assertEqual(15, len(sizeMap))
        self.assertEqual(243, sizeMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual(268, sizeMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link003"])])
        self.assertEqual(134, sizeMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual(74, sizeMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link003"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link004"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual(242, sizeMap[self.buildPath(["tree9", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSizeMap_004(self):
        """
        Test on a non-empty list (some containing spaces).
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        sizeMap = backupList.generateSizeMap()
        self.assertEqual(13, len(sizeMap))
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "file001"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "file with spaces"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "link003"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "link with spaces"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "dir002", "file001"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "dir002", "file002"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "dir002", "file003"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "dir with spaces", "file001"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree11", "dir with spaces", "file with spaces"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "dir with spaces", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree11", "dir with spaces", "link with spaces"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSizeMap_005(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        sizeMap = backupList.generateSizeMap()
        self.assertEqual(15, len(sizeMap))
        self.assertEqual(243, sizeMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual(268, sizeMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link003"])])
        self.assertEqual(134, sizeMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual(74, sizeMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link003"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link004"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual(242, sizeMap[self.buildPath(["tree9", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSizeMap_006(self):
        """
        Test on a non-empty list containing a non-existent file.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        sizeMap = backupList.generateSizeMap()
        self.assertEqual(15, len(sizeMap))
        self.assertEqual(243, sizeMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual(268, sizeMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir001", "link003"])])
        self.assertEqual(134, sizeMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual(74, sizeMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link003"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "dir002", "link004"])])
        self.assertEqual(155, sizeMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual(242, sizeMap[self.buildPath(["tree9", "file002"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link001"])])
        self.assertEqual(0, sizeMap[self.buildPath(["tree9", "link002"])])

    ###########################
    # Test generateDigestMap()
    ###########################

    def testGenerateDigestMap_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        digestMap = backupList.generateDigestMap()
        self.assertEqual(0, len(digestMap))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_002(self):
        """
        Test on a non-empty list containing only valid entries.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        digestMap = backupList.generateDigestMap()
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath(["tree9", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_003(self):
        """
        Test on a non-empty list containing only valid entries (some containing
        spaces).
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        digestMap = backupList.generateDigestMap()
        self.assertEqual(7, len(digestMap))
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "file001"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "file with spaces"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "dir002", "file001"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "dir002", "file003"])])
        self.assertEqual(
            "3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree11", "dir with spaces", "file001"])]
        )
        self.assertEqual(
            "3ef0b16a6237af9200b7a46c1987d6a555973847",
            digestMap[self.buildPath(["tree11", "dir with spaces", "file with spaces"])],
        )

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_004(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        digestMap = backupList.generateDigestMap()
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath(["tree9", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_005(self):
        """
        Test on a non-empty list containing a non-existent file.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        digestMap = backupList.generateDigestMap()
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath(["tree9", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath(["tree9", "file002"])])

    def testGenerateDigestMap_006(self):
        """
        Test on an empty list, passing stripPrefix not None.
        """
        backupList = BackupFileList()
        prefix = "whatever"
        digestMap = backupList.generateDigestMap(stripPrefix=prefix)
        self.assertEqual(0, len(digestMap))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_007(self):
        """
        Test on a non-empty list containing only valid entries, passing
        stripPrefix not None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        prefix = normalizeDir(self.buildPath(["tree9"]))
        digestMap = backupList.generateDigestMap(stripPrefix=prefix)
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath(["/", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath(["/", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath(["/", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath(["/", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath(["/", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_008(self):
        """
        Test on a non-empty list containing only valid entries (some containing
        spaces), passing stripPrefix not None.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        prefix = normalizeDir(self.buildPath(["tree11"]))
        digestMap = backupList.generateDigestMap(stripPrefix=prefix)
        self.assertEqual(7, len(digestMap))
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "file001"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "file with spaces"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "dir002", "file001"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "dir002", "file003"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "dir with spaces", "file001"])])
        self.assertEqual(
            "3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "dir with spaces", "file with spaces"])]
        )

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_009(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible), passing stripPrefix not None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        prefix = normalizeDir(self.buildPath(["tree9"]))
        digestMap = backupList.generateDigestMap(stripPrefix=prefix)
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath(["/", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath(["/", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath(["/", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath(["/", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath(["/", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateDigestMap_010(self):
        """
        Test on a non-empty list containing a non-existent file, passing
        stripPrefix not None.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        prefix = normalizeDir(self.buildPath(["tree9"]))
        digestMap = backupList.generateDigestMap(stripPrefix=prefix)
        self.assertEqual(6, len(digestMap))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath(["/", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath(["/", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath(["/", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath(["/", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath(["/", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath(["/", "file002"])])

    ########################
    # Test generateFitted()
    ########################

    def testGenerateFitted_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        fittedList = backupList.generateFitted(2000)
        self.assertEqual(0, len(fittedList))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_002(self):
        """
        Test on a non-empty list containing only valid entries, all of which fit.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        fittedList = backupList.generateFitted(2000)
        self.assertEqual(15, len(fittedList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fittedList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_003(self):
        """
        Test on a non-empty list containing only valid entries (some containing
        spaces), all of which fit.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        fittedList = backupList.generateFitted(2000)
        self.assertEqual(13, len(fittedList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in fittedList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_004(self):
        """
        Test on a non-empty list containing only valid entries, some of which
        fit.

        We can get some strange behavior on Windows, which hits the "links not
        supported" case.  The file tree9/dir002/file002 is 74 bytes, and is
        supposed to be the only file included because links are not recognized.
        However, link004 points at file002, and apparently Windows (sometimes?)
        sees link004 as a real file with a size of 74 bytes.  Since only one of
        the two fits in the fitted list, we just check for one or the other.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        fittedList = backupList.generateFitted(80)
        self.assertEqual(10, len(fittedList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fittedList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_005(self):
        """
        Test on a non-empty list containing only valid entries, none of which
        fit.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        fittedList = backupList.generateFitted(0)
        self.assertEqual(0, len(fittedList))
        fittedList = backupList.generateFitted(50)
        self.assertEqual(9, len(fittedList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fittedList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_006(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        fittedList = backupList.generateFitted(2000)
        self.assertEqual(15, len(fittedList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fittedList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateFitted_007(self):
        """
        Test on a non-empty list containing a non-existent file.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        fittedList = backupList.generateFitted(2000)
        self.assertEqual(15, len(fittedList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in fittedList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in fittedList)

    ######################
    # Test generateSpan()
    ######################

    def testGenerateSpan_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        spanSet = backupList.generateSpan(2000)
        self.assertEqual(0, len(spanSet))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSpan_002(self):
        """
        Test a set of files that all fit in one span item.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        spanSet = backupList.generateSpan(2000)
        self.assertEqual(1, len(spanSet))
        spanItem = spanSet[0]
        self.assertEqual(15, len(spanItem.fileList))
        self.assertEqual(1116, spanItem.size)
        self.assertEqual(2000, spanItem.capacity)
        self.assertEqual((1116.0 / 2000.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in spanItem.fileList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSpan_003(self):
        """
        Test a set of files that all fit in two span items.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        spanSet = backupList.generateSpan(760, "best_fit")
        self.assertEqual(2, len(spanSet))
        spanItem = spanSet[0]
        self.assertEqual(12, len(spanItem.fileList))
        self.assertEqual(753, spanItem.size)
        self.assertEqual(760, spanItem.capacity)
        self.assertEqual((753.0 / 760.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in spanItem.fileList)
        spanItem = spanSet[1]
        self.assertEqual(3, len(spanItem.fileList))
        self.assertEqual(363, spanItem.size)
        self.assertEqual(760, spanItem.capacity)
        self.assertEqual((363.0 / 760.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in spanItem.fileList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSpan_004(self):
        """
        Test a set of files that all fit in three span items.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        spanSet = backupList.generateSpan(515, "best_fit")
        self.assertEqual(3, len(spanSet))
        spanItem = spanSet[0]
        self.assertEqual(11, len(spanItem.fileList))
        self.assertEqual(511, spanItem.size)
        self.assertEqual(515, spanItem.capacity)
        self.assertEqual((511.0 / 515.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in spanItem.fileList)
        spanItem = spanSet[1]
        self.assertEqual(3, len(spanItem.fileList))
        self.assertEqual(471, spanItem.size)
        self.assertEqual(515, spanItem.capacity)
        self.assertEqual((471.0 / 515.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in spanItem.fileList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in spanItem.fileList)
        spanItem = spanSet[2]
        self.assertEqual(1, len(spanItem.fileList))
        self.assertEqual(134, spanItem.size)
        self.assertEqual(515, spanItem.capacity)
        self.assertEqual((134.0 / 515.0) * 100.0, spanItem.utilization)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in spanItem.fileList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateSpan_005(self):
        """
        Test a set of files where one of the files does not fit in the capacity.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        self.assertRaises(ValueError, backupList.generateSpan, 250, "best_fit")

    #########################
    # Test generateTarfile()
    #########################

    def testGenerateTarfile_001(self):
        """
        Test on an empty list.
        """
        backupList = BackupFileList()
        tarPath = self.buildPath(["file.tar"])
        self.assertRaises(ValueError, backupList.generateTarfile, tarPath)
        self.assertTrue(not os.path.exists(tarPath))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_002(self):
        """
        Test on a non-empty list containing a directory (which shouldn't be
        possible).
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", "dir001"]))  # back-door around addDir()
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(16, len(tarList))
        self.assertTrue(
            self.tarPath(["tree9", "dir001/"]) in tarList
            or self.tarPath(["tree9", "dir001//"]) in tarList  # Grr... Python 2.5 behavior differs
            or self.tarPath(["tree9", "dir001"]) in tarList
        )  # Grr... Python 2.6 behavior differs
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_003(self):
        """
        Test on a non-empty list containing a non-existent file, ignore=False.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        self.assertRaises(tarfile.TarError, backupList.generateTarfile, tarPath, ignore=False)
        self.assertTrue(not os.path.exists(tarPath))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_004(self):
        """
        Test on a non-empty list containing a non-existent file, ignore=True.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        backupList.append(self.buildPath(["tree9", INVALID_FILE]))  # file won't exist on disk
        self.assertEqual(16, len(backupList))
        self.assertTrue(self.buildPath(["tree9", INVALID_FILE]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath, ignore=True)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_005(self):
        """
        Test on a non-empty list containing only valid entries, with an invalid mode.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        self.assertRaises(ValueError, backupList.generateTarfile, tarPath, mode="bogus")
        self.assertTrue(not os.path.exists(tarPath))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_006(self):
        """
        Test on a non-empty list containing only valid entries, default mode.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_007(self):
        """
        Test on a non-empty list (some containing spaces), default mode.
        """
        self.extractTar("tree11")
        path = self.buildPath(["tree11"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(13, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "link with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "link with spaces"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(13, len(tarList))
        self.assertTrue(self.tarPath(["tree11", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "file with spaces"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "link with spaces"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir002", "file003"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir with spaces", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir with spaces", "file with spaces"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir with spaces", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree11", "dir with spaces", "link with spaces"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_008(self):
        """
        Test on a non-empty list containing only valid entries, 'tar' mode.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_009(self):
        """
        Test on a non-empty list containing only valid entries, 'targz' mode.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar.gz"])
        backupList.generateTarfile(tarPath, mode="targz")
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_010(self):
        """
        Test on a non-empty list containing only valid entries, 'tarbz2' mode.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildPath(["file.tar.bz2"])
        backupList.generateTarfile(tarPath, mode="tarbz2")
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_011(self):
        """
        Test on a non-empty list containing only valid entries, 'tar' mode, long target name.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildRandomPath(255, ".tar")
        backupList.generateTarfile(tarPath, mode="tar")
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_012(self):
        """
        Test on a non-empty list containing only valid entries, 'targz' mode, long target name.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildRandomPath(255, ".tar")
        backupList.generateTarfile(tarPath, mode="targz")
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testGenerateTarfile_013(self):
        """
        Test on a non-empty list containing only valid entries, 'tarbz2' mode, long target name.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        tarPath = self.buildRandomPath(255, ".tar")
        backupList.generateTarfile(tarPath, mode="tarbz2")
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(15, len(tarList))
        self.assertTrue(self.tarPath(["tree9", "dir001", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir001", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link003"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "dir002", "link004"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "file002"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link001"]) in tarList)
        self.assertTrue(self.tarPath(["tree9", "link002"]) in tarList)

    def testGenerateTarfile_014(self):
        """
        Test behavior of the flat flag.
        """
        self.extractTar("tree11")
        backupList = BackupFileList()
        path = self.buildPath(["tree11", "dir with spaces", "file with spaces"])
        backupList.addFile(path)
        path = self.buildPath(["tree11", "dir with spaces", "file001"])
        backupList.addFile(path)
        path = self.buildPath(["tree11", "dir002", "file002"])
        backupList.addFile(path)
        path = self.buildPath(["tree11", "dir002", "file003"])
        backupList.addFile(path)
        self.assertEqual(4, len(backupList))
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file with spaces"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir with spaces", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree11", "dir002", "file003"]) in backupList)
        tarPath = self.buildPath(["file.tar"])
        backupList.generateTarfile(tarPath, flat=True)
        self.assertTrue(tarfile.is_tarfile(tarPath))
        with tarfile.open(tarPath) as tarFile:
            tarList = tarFile.getnames()
        self.assertEqual(4, len(tarList))
        self.assertTrue("file with spaces" in tarList)
        self.assertTrue("file001" in tarList)
        self.assertTrue("file002" in tarList)
        self.assertTrue("file003" in tarList)

    #########################
    # Test removeUnchanged()
    #########################

    def testRemoveUnchanged_001(self):
        """
        Test on an empty list with an empty digest map.
        """
        digestMap = {}
        backupList = BackupFileList()
        self.assertEqual(0, len(backupList))
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))

    def testRemoveUnchanged_002(self):
        """
        Test on an empty list with an non-empty digest map.
        """
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir002", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir002", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file002"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        self.assertEqual(0, len(backupList))
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_003(self):
        """
        Test on an non-empty list with an empty digest map.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {}
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_004(self):
        """
        Test with a digest map containing only entries that are not in the list.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir003", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir003", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir004", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir004", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file003"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file004"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_005(self):
        """
        Test with a digest map containing only entries that are in the list, with
        non-matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e8AAAAAAAAAAAAAAAAAA7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecAAAAAAAAAAAAAAAAAA95d1d6cba",
            self.buildPath(["tree9", "dir002", "file001"]): "2f68cdda26b64AAAAAAAAAAAAAAAAAA5b8786c4b",
            self.buildPath(["tree9", "dir002", "file002"]): "0cc03b3014d1cAAAAAAAAAAAAAAAAAA5d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237aAAAAAAAAAAAAAAAAAA555973847",
            self.buildPath(["tree9", "file002"]): "fae89085ee97bAAAAAAAAAAAAAAAAAAbb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_006(self):
        """
        Test with a digest map containing only entries that are in the list, with
        matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir002", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir002", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file002"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(6, count)
        self.assertEqual(9, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_007(self):
        """
        Test with a digest map containing both entries that are and are not in
        the list, with non-matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531cCCCCCCCCCCCCCCCCCCCCCCCCCe77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a2CCCCCCCCCCCCCCCCCCCCCCCCCd6cba",
            self.buildPath(["tree9", "dir003", "file001"]): "2f68cdda26CCCCCCCCCCCCCCCCCCCCCCCCC86c4b",
            self.buildPath(["tree9", "dir003", "file002"]): "0cc03b3014CCCCCCCCCCCCCCCCCCCCCCCCCd26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a62CCCCCCCCCCCCCCCCCCCCCCCCC73847",
            self.buildPath(["tree9", "file003"]): "fae89085eeCCCCCCCCCCCCCCCCCCCCCCCCC769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_008(self):
        """
        Test with a digest map containing both entries that are and are not in
        the list, with matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir003", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir003", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file003"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(3, count)
        self.assertEqual(12, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_009(self):
        """
        Test with a digest map containing both entries that are and are not in
        the list, with matching and non-matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531AAAAAAAAAAAAAAAAAAAAAAAe21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir003", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir003", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file003"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        count = backupList.removeUnchanged(digestMap)
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(2, count)
        self.assertEqual(13, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)

    def testRemoveUnchanged_010(self):
        """
        Test on an empty list with an empty digest map.
        """
        digestMap = {}
        backupList = BackupFileList()
        self.assertEqual(0, len(backupList))
        (count, newDigest) = backupList.removeUnchanged(digestMap, captureDigest=True)  # pylint: disable=W0633
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        self.assertEqual(0, len(newDigest))

    def testRemoveUnchanged_011(self):
        """
        Test on an empty list with an non-empty digest map.
        """
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir002", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir002", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file002"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        self.assertEqual(0, len(backupList))
        (count, newDigest) = backupList.removeUnchanged(digestMap, captureDigest=True)  # pylint: disable=W0633
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(0, len(backupList))
        self.assertEqual(0, len(newDigest))

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_012(self):
        """
        Test on an non-empty list with an empty digest map.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {}
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        (count, newDigest) = backupList.removeUnchanged(digestMap, captureDigest=True)  # pylint: disable=W0633
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        self.assertEqual(6, len(newDigest))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", newDigest[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", newDigest[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", newDigest[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", newDigest[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", newDigest[self.buildPath(["tree9", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", newDigest[self.buildPath(["tree9", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_013(self):
        """
        Test with a digest map containing only entries that are not in the list.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir003", "file001"]): "4ff529531c7e897cd3df90ed76355de7e21e77ee",
            self.buildPath(["tree9", "dir003", "file002"]): "9d473094a22ecf2ae299c25932c941795d1d6cba",
            self.buildPath(["tree9", "dir004", "file001"]): "2f68cdda26b643ca0e53be6348ae1255b8786c4b",
            self.buildPath(["tree9", "dir004", "file002"]): "0cc03b3014d1ca7188264677cf01f015d72d26cb",
            self.buildPath(["tree9", "file003"]): "3ef0b16a6237af9200b7a46c1987d6a555973847",
            self.buildPath(["tree9", "file004"]): "fae89085ee97b57ccefa7e30346c573bb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        (count, newDigest) = backupList.removeUnchanged(digestMap, captureDigest=True)  # pylint: disable=W0633
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        self.assertEqual(6, len(newDigest))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", newDigest[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", newDigest[self.buildPath(["tree9", "dir001", "file002"])])
        self.assertEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", newDigest[self.buildPath(["tree9", "dir002", "file001"])])
        self.assertEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", newDigest[self.buildPath(["tree9", "dir002", "file002"])])
        self.assertEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", newDigest[self.buildPath(["tree9", "file001"])])
        self.assertEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", newDigest[self.buildPath(["tree9", "file002"])])

    @unittest.skipUnless(platformSupportsLinks(), "Requires soft links")
    def testRemoveUnchanged_014(self):
        """
        Test with a digest map containing only entries that are in the list, with
        non-matching digests.
        """
        self.extractTar("tree9")
        path = self.buildPath(["tree9"])
        digestMap = {
            self.buildPath(["tree9", "dir001", "file001"]): "4ff529531c7e8AAAAAAAAAAAAAAAAAA7e21e77ee",
            self.buildPath(["tree9", "dir001", "file002"]): "9d473094a22ecAAAAAAAAAAAAAAAAAA95d1d6cba",
            self.buildPath(["tree9", "dir002", "file001"]): "2f68cdda26b64AAAAAAAAAAAAAAAAAA5b8786c4b",
            self.buildPath(["tree9", "dir002", "file002"]): "0cc03b3014d1cAAAAAAAAAAAAAAAAAA5d72d26cb",
            self.buildPath(["tree9", "file001"]): "3ef0b16a6237aAAAAAAAAAAAAAAAAAA555973847",
            self.buildPath(["tree9", "file002"]): "fae89085ee97bAAAAAAAAAAAAAAAAAAbb0a769db",
        }
        backupList = BackupFileList()
        count = backupList.addDirContents(path)
        self.assertEqual(15, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        (count, newDigest) = backupList.removeUnchanged(digestMap, captureDigest=True)  # pylint: disable=W0633
        self.assertTrue(isinstance(backupList, BackupFileList))  # make sure we just replaced it
        self.assertEqual(0, count)
        self.assertEqual(15, len(backupList))
        self.assertTrue(self.buildPath(["tree9", "dir001", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir001", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link003"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "dir002", "link004"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "file002"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link001"]) in backupList)
        self.assertTrue(self.buildPath(["tree9", "link002"]) in backupList)
        self.assertEqual(6, len(newDigest))
        self.assertEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", newDigest[self.buildPath(["tree9", "dir001", "file001"])])
        self.assertEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", newDigest[self.buildPath(["tree9", "dir001", "file002"