"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const asg = require("aws-cdk-lib/aws-autoscaling");
const ec2 = require("aws-cdk-lib/aws-ec2");
const gitlab_runner_autoscaling_1 = require("./gitlab-runner-autoscaling");
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, 'TestStackAutoscaling', { env });
const vpc = ec2.Vpc.fromLookup(stack, 'DefaultVpc', {
    isDefault: true,
});
const defaultCapacity = parseInt(stack.node.tryGetContext('CAPACITY'));
const runner = new gitlab_runner_autoscaling_1.GitlabRunnerAutoscaling(stack, 'TestRunnerAutoscaling', {
    gitlabToken: stack.node.tryGetContext('GITLAB_TOKEN'),
    ebsSize: 10,
    vpc: vpc,
    dockerVolumes: [{
            hostPath: '/tmp/cache',
            containerPath: '/tmp/cache',
        }],
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
    spotInstance: true,
    alarms: [
        {
            AlarmName: 'GitlabRunnerDiskUsage',
            MetricName: 'disk_used_percent',
            Threshold: 50,
        },
    ],
    gitlabRunnerVersion: '15.10',
});
/**
 * Scheduled scaling
 * https://docs.aws.amazon.com/cdk/api/latest/docs/aws-autoscaling-readme.html#scheduled-scaling
 */
runner.autoscalingGroup.scaleOnSchedule('StopOnWeekends', {
    schedule: asg.Schedule.cron({ weekDay: 'fri', hour: '18', minute: '0' }),
    minCapacity: 0,
    maxCapacity: 0,
});
runner.autoscalingGroup.scaleOnSchedule('WorkOnWeekdays', {
    schedule: asg.Schedule.cron({ weekDay: 'mon', hour: '9', minute: '0' }),
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
});
new aws_cdk_lib_1.CfnOutput(stack, 'role', { value: runner.instanceRole.roleArn });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZ2l0bGFiLXJ1bm5lci1hdXRvc2NhbGluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9pbnRlZy5naXRsYWItcnVubmVyLWF1dG9zY2FsaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNkNBQW9EO0FBQ3BELG1EQUFtRDtBQUNuRCwyQ0FBMkM7QUFDM0MsMkVBQXNFO0FBRXRFLE1BQU0sR0FBRyxHQUFHO0lBQ1YsTUFBTSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCO0lBQ3RDLE9BQU8sRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQjtDQUN6QyxDQUFDO0FBRUYsTUFBTSxHQUFHLEdBQUcsSUFBSSxpQkFBRyxFQUFFLENBQUM7QUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxDQUFDLEdBQUcsRUFBRSxzQkFBc0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7QUFFOUQsTUFBTSxHQUFHLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtJQUNsRCxTQUFTLEVBQUUsSUFBSTtDQUNoQixDQUFDLENBQUM7QUFDSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztBQUN2RSxNQUFNLE1BQU0sR0FBRyxJQUFJLG1EQUF1QixDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRTtJQUN6RSxXQUFXLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDO0lBQ3JELE9BQU8sRUFBRSxFQUFFO0lBQ1gsR0FBRyxFQUFFLEdBQUc7SUFDUixhQUFhLEVBQUUsQ0FBQztZQUNkLFFBQVEsRUFBRSxZQUFZO1lBQ3RCLGFBQWEsRUFBRSxZQUFZO1NBQzVCLENBQUM7SUFDRixXQUFXLEVBQUUsZUFBZTtJQUM1QixXQUFXLEVBQUUsZUFBZTtJQUM1QixZQUFZLEVBQUUsSUFBSTtJQUNsQixNQUFNLEVBQUU7UUFDTjtZQUNFLFNBQVMsRUFBRSx1QkFBdUI7WUFDbEMsVUFBVSxFQUFFLG1CQUFtQjtZQUMvQixTQUFTLEVBQUUsRUFBRTtTQUNkO0tBQ0Y7SUFDRCxtQkFBbUIsRUFBRSxPQUFPO0NBQzdCLENBQUMsQ0FBQztBQUVIOzs7R0FHRztBQUNILE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxlQUFlLENBQUMsZ0JBQWdCLEVBQUU7SUFDeEQsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUUsQ0FBQztJQUN4RSxXQUFXLEVBQUUsQ0FBQztJQUNkLFdBQVcsRUFBRSxDQUFDO0NBQ2YsQ0FBQyxDQUFDO0FBRUgsTUFBTSxDQUFDLGdCQUFnQixDQUFDLGVBQWUsQ0FBQyxnQkFBZ0IsRUFBRTtJQUN4RCxRQUFRLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUFDO0lBQ3ZFLFdBQVcsRUFBRSxlQUFlO0lBQzVCLFdBQVcsRUFBRSxlQUFlO0NBQzdCLENBQUMsQ0FBQztBQUdILElBQUksdUJBQVMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFwcCwgU3RhY2ssIENmbk91dHB1dCB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGFzZyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0IHsgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcgfSBmcm9tICcuL2dpdGxhYi1ydW5uZXItYXV0b3NjYWxpbmcnO1xuXG5jb25zdCBlbnYgPSB7XG4gIHJlZ2lvbjogcHJvY2Vzcy5lbnYuQ0RLX0RFRkFVTFRfUkVHSU9OLFxuICBhY2NvdW50OiBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5ULFxufTtcblxuY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnVGVzdFN0YWNrQXV0b3NjYWxpbmcnLCB7IGVudiB9KTtcblxuY29uc3QgdnBjID0gZWMyLlZwYy5mcm9tTG9va3VwKHN0YWNrLCAnRGVmYXVsdFZwYycsIHtcbiAgaXNEZWZhdWx0OiB0cnVlLFxufSk7XG5jb25zdCBkZWZhdWx0Q2FwYWNpdHkgPSBwYXJzZUludChzdGFjay5ub2RlLnRyeUdldENvbnRleHQoJ0NBUEFDSVRZJykpO1xuY29uc3QgcnVubmVyID0gbmV3IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nKHN0YWNrLCAnVGVzdFJ1bm5lckF1dG9zY2FsaW5nJywge1xuICBnaXRsYWJUb2tlbjogc3RhY2subm9kZS50cnlHZXRDb250ZXh0KCdHSVRMQUJfVE9LRU4nKSxcbiAgZWJzU2l6ZTogMTAsXG4gIHZwYzogdnBjLFxuICBkb2NrZXJWb2x1bWVzOiBbe1xuICAgIGhvc3RQYXRoOiAnL3RtcC9jYWNoZScsXG4gICAgY29udGFpbmVyUGF0aDogJy90bXAvY2FjaGUnLFxuICB9XSxcbiAgbWluQ2FwYWNpdHk6IGRlZmF1bHRDYXBhY2l0eSxcbiAgbWF4Q2FwYWNpdHk6IGRlZmF1bHRDYXBhY2l0eSxcbiAgc3BvdEluc3RhbmNlOiB0cnVlLFxuICBhbGFybXM6IFtcbiAgICB7XG4gICAgICBBbGFybU5hbWU6ICdHaXRsYWJSdW5uZXJEaXNrVXNhZ2UnLFxuICAgICAgTWV0cmljTmFtZTogJ2Rpc2tfdXNlZF9wZXJjZW50JyxcbiAgICAgIFRocmVzaG9sZDogNTAsXG4gICAgfSxcbiAgXSxcbiAgZ2l0bGFiUnVubmVyVmVyc2lvbjogJzE1LjEwJyxcbn0pO1xuXG4vKipcbiAqIFNjaGVkdWxlZCBzY2FsaW5nXG4gKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS9sYXRlc3QvZG9jcy9hd3MtYXV0b3NjYWxpbmctcmVhZG1lLmh0bWwjc2NoZWR1bGVkLXNjYWxpbmdcbiAqL1xucnVubmVyLmF1dG9zY2FsaW5nR3JvdXAuc2NhbGVPblNjaGVkdWxlKCdTdG9wT25XZWVrZW5kcycsIHtcbiAgc2NoZWR1bGU6IGFzZy5TY2hlZHVsZS5jcm9uKHsgd2Vla0RheTogJ2ZyaScsIGhvdXI6ICcxOCcsIG1pbnV0ZTogJzAnIH0pLFxuICBtaW5DYXBhY2l0eTogMCxcbiAgbWF4Q2FwYWNpdHk6IDAsXG59KTtcblxucnVubmVyLmF1dG9zY2FsaW5nR3JvdXAuc2NhbGVPblNjaGVkdWxlKCdXb3JrT25XZWVrZGF5cycsIHtcbiAgc2NoZWR1bGU6IGFzZy5TY2hlZHVsZS5jcm9uKHsgd2Vla0RheTogJ21vbicsIGhvdXI6ICc5JywgbWludXRlOiAnMCcgfSksXG4gIG1pbkNhcGFjaXR5OiBkZWZhdWx0Q2FwYWNpdHksXG4gIG1heENhcGFjaXR5OiBkZWZhdWx0Q2FwYWNpdHksXG59KTtcblxuXG5uZXcgQ2ZuT3V0cHV0KHN0YWNrLCAncm9sZScsIHsgdmFsdWU6IHJ1bm5lci5pbnN0YW5jZVJvbGUucm9sZUFybiB9KTtcbiJdfQ==