#  Copyright 2022-present, the Waterdip Labs Pvt. Ltd.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
from dcs_core.core.common.models.metric import MetricsType
from dcs_core.core.datasource.search_datasource import SearchIndexDataSource
from dcs_core.core.datasource.sql_datasource import SQLDataSource
from dcs_core.core.metric.base import FieldMetrics, Metric, MetricIdentity


class DocumentCountMetric(Metric):
    """
    DocumentCountMetrics is a class that represents a metric that is generated by a data source.
    """

    def validate_data_source(self):
        return isinstance(self.data_source, SearchIndexDataSource)

    def get_metric_identity(self):
        return MetricIdentity.generate_identity(
            metric_type=MetricsType.DOCUMENT_COUNT,
            metric_name=self.name,
            data_source=self.data_source,
            index_name=self.index_name,
        )

    def _generate_metric_value(self):
        if isinstance(self.data_source, SearchIndexDataSource):
            return self.data_source.query_get_document_count(
                index_name=self.index_name,
                filters=self.filter_query if self.filter_query else None,
            )
        else:
            raise ValueError("Invalid data source type")


class RowCountMetric(Metric):

    """
    RowCountMetrics is a class that represents a metric that is generated by a data source.
    """

    def get_metric_identity(self):
        return MetricIdentity.generate_identity(
            metric_type=MetricsType.ROW_COUNT,
            metric_name=self.name,
            data_source=self.data_source,
            table_name=self.table_name,
        )

    def _generate_metric_value(self):
        if isinstance(self.data_source, SQLDataSource):
            return self.data_source.query_get_row_count(
                table=self.table_name,
                filters=self.filter_query if self.filter_query else None,
            )
        else:
            raise ValueError("Invalid data source type")


class FreshnessValueMetric(FieldMetrics):
    """
    FreshnessMetric is a class that represents a metric that is generated by a data source.
    """

    def get_metric_identity(self):
        return MetricIdentity.generate_identity(
            metric_type=MetricsType.FRESHNESS,
            metric_name=self.name,
            data_source=self.data_source,
            field_name=self.field_name,
            table_name=self.table_name,
            index_name=self.index_name,
        )

    def _generate_metric_value(self):
        if isinstance(self.data_source, SQLDataSource):
            return self.data_source.query_get_time_diff(
                table=self.table_name, field=self.field_name
            )
        elif isinstance(self.data_source, SearchIndexDataSource):
            return self.data_source.query_get_time_diff(
                index_name=self.index_name, field=self.field_name
            )
        else:
            raise ValueError("Invalid data source type")
