#  Copyright 2022-present, the Waterdip Labs Pvt. Ltd.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

from dataclasses import dataclass
from typing import List, Optional

from dcs_core.core.common.models.metric import MetricsType, MetricValue


@dataclass
class NumericFieldProfile:
    """
    NumericFieldProfile is a class that represents a numeric field profile that is generated by a data source.
    """

    field_name: str
    data_type: str
    min: Optional[MetricValue] = None
    max: Optional[MetricValue] = None
    avg: Optional[MetricValue] = None
    sum: Optional[MetricValue] = None
    stddev: Optional[MetricValue] = None
    variance: Optional[MetricValue] = None
    missing_count: Optional[MetricValue] = None
    distinct_count: Optional[MetricValue] = None
    skewness: Optional[MetricValue] = None
    kurtosis: Optional[MetricValue] = None

    @property
    def get_metric_values(self) -> List[MetricValue]:
        """
        Get all metric values from the profile.
        """
        values = []
        for key, value in self.__dict__.items():
            if isinstance(value, MetricValue):
                values.append(value)
        return values


@dataclass
class TextFieldProfile:
    """
    TextFieldProfile is a class that represents a text field profile that is generated by a data source.
    """

    field_name: str
    data_type: str
    min_length: Optional[MetricValue] = None
    max_length: Optional[MetricValue] = None
    avg_length: Optional[MetricValue] = None
    distinct_count: Optional[MetricValue] = None
    missing_count: Optional[MetricValue] = None

    @property
    def get_metric_values(self) -> List[MetricValue]:
        """
        Get all metric values from the profile.
        """
        values = []
        for key, value in self.__dict__.items():
            if isinstance(value, MetricValue):
                values.append(value)
        return values
