"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./aws-credentials");
const github_action_step_1 = require("./steps/github-action-step");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.jobSettings = props.jobSettings;
        this.awsCredentials = this.getAwsCredentials(props);
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Parse AWS credential configuration from deprecated properties For backwards compatibility.
     */
    getAwsCredentials(props) {
        if (props.gitHubActionRoleArn) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove githubActionRoleArn)');
            }
            return aws_credentials_1.AwsCredentials.fromOpenIdConnect({
                gitHubActionRoleArn: props.gitHubActionRoleArn,
            });
        }
        if (props.awsCredentials) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove awsCredentials)');
            }
            return aws_credentials_1.AwsCredentials.fromGitHubSecrets({
                accessKeyId: 'AWS_ACCESS_KEY_ID',
                secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
                ...props.awsCredentials,
            });
        }
        return props.awsCreds ?? aws_credentials_1.AwsCredentials.fromGitHubSecrets();
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const contextValue = this.node.tryGetContext('cdk-pipelines-github:diffProtection');
        const diffProtection = contextValue === 'false' ? false : contextValue ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || this.workflowFile.toYaml() !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Workflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else if (node.data.step instanceof github_action_step_1.GitHubActionStep) {
                    return this.jobForGitHubActionStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -ex'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '${ASSET_HASH_NAME}=${assetId}' >> $GITHUB_OUTPUT`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.jobSettings,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.publishAssetsAuthRegion),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.jobSettings,
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                ...(this.stackProperties[stack.stackArtifactId]?.environment ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(region, assumeRoleArn),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1.1.0',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v3',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v3',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    jobForGitHubActionStep(node, step) {
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.WRITE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: step.jobSteps,
            },
        };
    }
    stepsToConfigureAws(region, assumeRoleArn) {
        return this.awsCredentials.credentialSteps(region, assumeRoleArn);
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v2',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v3',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v3',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v3',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.3.135" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcGlwZWxpbmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwyQkFBd0U7QUFDeEUsNkJBQTZCO0FBQzdCLDZDQUFvQztBQUNwQywrQ0FBNkQ7QUFDN0QscURBQTZKO0FBQzdKLGlGQUF1RztBQUV2Ryx5Q0FBeUM7QUFDekMsdURBQTJFO0FBRzNFLG1FQUE4RDtBQUM5RCw0Q0FBNEM7QUFDNUMsMkNBQXVDO0FBRXZDLE1BQU0sZUFBZSxHQUFHLFNBQVMsQ0FBQztBQUNsQyxNQUFNLGVBQWUsR0FBRyxZQUFZLENBQUM7QUE2SXJDOztHQUVHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsd0JBQVk7SUFvQjlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFSVCxlQUFVLEdBQTJDLEVBQUUsQ0FBQztRQUN4RCxpQkFBWSxHQUEyQixFQUFFLENBQUM7UUFHMUMsb0JBQWUsR0FBd0MsRUFBRSxDQUFDO1FBTXpFLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUN6QyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO1FBQzVDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztRQUMzQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksRUFBRSxDQUFDO1FBQy9DLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7UUFDakQsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBRXJDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXBELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLElBQUksRUFBRSxDQUFDO1FBRXZELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSw4QkFBOEIsQ0FBQztRQUN6RSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxDQUFDLENBQUM7U0FDaEU7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsRUFBRTtZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLHlGQUF5RixDQUFDLENBQUM7U0FDNUc7UUFFRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksb0JBQVEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLFFBQVEsQ0FBQztRQUNuRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixJQUFJO1lBQ2hELElBQUksRUFBRSxFQUFFLFFBQVEsRUFBRSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQzVCLGdCQUFnQixFQUFFLEVBQUU7U0FDckIsQ0FBQztRQUVGLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQztRQUMxRCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixJQUFJLFdBQVcsQ0FBQztJQUM5RSxDQUFDO0lBRUQ7O09BRUc7SUFDSyxpQkFBaUIsQ0FBQyxLQUEwQjtRQUNsRCxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7Z0JBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0VBQStFLENBQUMsQ0FBQzthQUNsRztZQUNELE9BQU8sZ0NBQWMsQ0FBQyxpQkFBaUIsQ0FBQztnQkFDdEMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjthQUMvQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7Z0JBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQzthQUM3RjtZQUNELE9BQU8sZ0NBQWMsQ0FBQyxpQkFBaUIsQ0FBQztnQkFDdEMsV0FBVyxFQUFFLG1CQUFtQjtnQkFDaEMsZUFBZSxFQUFFLHVCQUF1QjtnQkFDeEMsR0FBRyxLQUFLLENBQUMsY0FBYzthQUN4QixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDLFFBQVEsSUFBSSxnQ0FBYyxDQUFDLGlCQUFpQixFQUFFLENBQUM7SUFDOUQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0kseUJBQXlCLENBQUMsS0FBWSxFQUFFLE9BQStCO1FBQzVFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXRELHdDQUF3QztRQUN4QyxNQUFNLE1BQU0sR0FBRyxlQUFlLENBQUMsTUFBTSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDdkUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUU3RCxPQUFPLGVBQWUsQ0FBQztJQUN6QixDQUFDO0lBRU8sYUFBYSxDQUFDLE1BQXlCLEVBQUUsR0FBVyxFQUFFLEtBQVU7UUFDdEUsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQUUsT0FBTztTQUFFO1FBQ3BDLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFO1lBQzFCLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxHQUFHO2dCQUM1QyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQztnQkFDOUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxLQUFLO2FBQ2IsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVTLGVBQWU7UUFDdkIsTUFBTSxHQUFHLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0IsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUNELE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUM7UUFFN0IsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLEVBQU8sQ0FBQztRQUU5QixNQUFNLFNBQVMsR0FBRyxJQUFJLGdDQUFhLENBQUMsSUFBSSxFQUFFO1lBQ3hDLFlBQVksRUFBRSxLQUFLO1lBQ25CLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLFdBQVcsRUFBRSxLQUFLO1NBQ25CLENBQUMsQ0FBQztRQUVILEtBQUssTUFBTSxTQUFTLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUMsRUFBRTtZQUNqRSxJQUFJLENBQUMsMEJBQU8sQ0FBQyxTQUFTLENBQUMsRUFBRTtnQkFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyQ0FBMkMsU0FBUyxHQUFHLENBQUMsQ0FBQzthQUMxRTtZQUVELE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUUxQyxLQUFLLE1BQU0sT0FBTyxJQUFJLFFBQVEsRUFBRTtnQkFDOUIsS0FBSyxNQUFNLElBQUksSUFBSSxPQUFPLEVBQUU7b0JBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFO3dCQUNoQyxXQUFXLEVBQUUsU0FBUzt3QkFDdEIsU0FBUztxQkFDVixDQUFDLENBQUM7b0JBRUgsSUFBSSxHQUFHLEVBQUU7d0JBQ1AsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO1FBRUQsOERBQThEO1FBQzlELE1BQU0sTUFBTSxHQUErQixFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDdEIsSUFBSSxHQUFHLENBQUMsRUFBRSxJQUFJLE1BQU0sRUFBRTtnQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7YUFDL0M7WUFDRCxNQUFNLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDaEQ7UUFFRCw4Q0FBOEM7UUFDOUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTlCLE1BQU0sUUFBUSxHQUFHO1lBQ2YsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3ZCLEVBQUUsRUFBRSxhQUFhLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQztZQUM3QyxJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUM7UUFFRix1QkFBdUI7UUFDdkIsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFbkMsd0NBQXdDO1FBQ3hDLGNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRWhFLHNFQUFzRTtRQUN0RSxtSEFBbUg7UUFDbkgsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUNwRixNQUFNLGNBQWMsR0FBRyxZQUFZLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUM7UUFDL0UsSUFBSSxjQUFjLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLEtBQUssSUFBSSxDQUFDLFlBQVksRUFBRTtZQUN2RSxxQ0FBcUM7WUFDckMsSUFBSSxDQUFDLGVBQVUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsS0FBSyxpQkFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLEVBQUU7Z0JBQzVHLE1BQU0sSUFBSSxLQUFLLENBQUMsMkNBQTJDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsNENBQTRDLENBQUMsQ0FBQzthQUNySjtTQUNGO1FBRUQsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUNoQyxDQUFDO0lBRU8sZ0JBQWdCLENBQUMsTUFBa0M7UUFDekQsS0FBSyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ2pFLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRztnQkFDZCxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUM7Z0JBQ2hCLE9BQU8sRUFBRTtvQkFDUCxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPO29CQUN4QixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7aUJBQ3JDO2FBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVPLGdCQUFnQixDQUFDLE9BQStCO1FBQ3RELE1BQU0sZUFBZSxHQUEyQixFQUFFLENBQUM7UUFDbkQsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDNUIsZUFBZSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxjQUFjLE1BQU0sQ0FBQyxNQUFNLFlBQVksTUFBTSxDQUFDLFVBQVUsS0FBSyxDQUFDO1NBQ3BHO1FBQ0QsT0FBTyxlQUFlLENBQUM7SUFDekIsQ0FBQztJQUVEOztPQUVHO0lBQ0ssVUFBVSxDQUFDLElBQWdCLEVBQUUsT0FBZ0I7UUFDbkQsUUFBUSxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRTtZQUN2Qiw2RUFBNkU7WUFDN0UsS0FBSyxPQUFPLENBQUM7WUFDYixLQUFLLGFBQWEsQ0FBQztZQUNuQixLQUFLLFNBQVM7Z0JBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBRXZGLEtBQUssYUFBYTtnQkFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1lBRXJFLEtBQUssZ0JBQWdCO2dCQUNuQixPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFFbEUsS0FBSyxTQUFTO2dCQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztZQUVwRSxLQUFLLFNBQVM7Z0JBQ1osT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBRTVFLEtBQUssTUFBTTtnQkFDVCxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFO29CQUN6QixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ25EO3FCQUFNLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLFlBQVkscUJBQVMsRUFBRTtvQkFDOUMsT0FBTyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ3BEO3FCQUFNLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLFlBQVkscUNBQWdCLEVBQUU7b0JBQ3JELE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUMxRDtxQkFBTTtvQkFDTCxNQUFNLElBQUksS0FBSyxDQUFDLDBCQUEwQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztpQkFDOUU7WUFFSDtnQkFDRSxpRUFBaUU7Z0JBQ2pFLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXlELElBQUksQ0FBQyxJQUFZLEVBQUUsSUFBSSx5RkFBeUYsQ0FBQyxDQUFDO1NBQzlMO0lBQ0gsQ0FBQztJQUVPLGtCQUFrQixDQUFDLElBQWdCLEVBQUUsTUFBb0IsRUFBRSxPQUFnQjtRQUNqRixJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUVELE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDekUsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUN0QyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQzVCLE1BQU0sT0FBTyxHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFFbEMsbUVBQW1FO1FBQ25FLE1BQU0sZ0JBQWdCLEdBQXFCLEVBQUUsQ0FBQztRQUM5QyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxJQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzlFLEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLGlCQUFpQixFQUFFO2dCQUMxQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsc0JBQXNCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQzthQUM5RDtTQUNGO1FBRUQsb0NBQW9DO1FBQ3BDLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxDQUFTLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNoSCxNQUFNLFlBQVksR0FBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7WUFDckUsT0FBTywwQkFBMEIsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLHdCQUF3QixLQUFLLENBQUMsYUFBYSxHQUFHLENBQUM7UUFDN0gsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVKLG1EQUFtRDtRQUNuRCxJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQztRQUNuQyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsZUFBZSxJQUFJLE9BQU8scUJBQXFCLENBQUMsQ0FBQztRQUU1RSxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxXQUFXLEtBQUssVUFBVSxDQUFDLENBQUM7UUFDekUsY0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUM5RCxrQkFBYSxDQUFDLGVBQWUsRUFBRSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFFL0UsTUFBTSxXQUFXLEdBQW1CO1lBQ2xDLEVBQUUsRUFBRSxTQUFTO1lBQ2IsSUFBSSxFQUFFLFdBQVcsS0FBSyxFQUFFO1lBQ3hCLEdBQUcsRUFBRSx1QkFBdUIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsZUFBZSxDQUFDLEVBQUU7U0FDeEUsQ0FBQztRQUVGLE9BQU87WUFDTCxFQUFFLEVBQUUsS0FBSztZQUNULFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsa0JBQWtCLEtBQUssRUFBRTtnQkFDL0IsR0FBRyxJQUFJLENBQUMsV0FBVztnQkFDbkIsS0FBSyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUM7Z0JBQ3BDLFdBQVcsRUFBRTtvQkFDWCxRQUFRLEVBQUUsTUFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJO29CQUNuQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLEVBQUU7aUJBQzdDO2dCQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07Z0JBQzFCLE9BQU8sRUFBRTtvQkFDUCxDQUFDLGVBQWUsQ0FBQyxFQUFFLDhCQUE4QixlQUFlLEtBQUs7aUJBQ3RFO2dCQUNELEtBQUssRUFBRTtvQkFDTCxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxTQUFTLENBQUM7b0JBQzFDO3dCQUNFLElBQUksRUFBRSxTQUFTO3dCQUNmLEdBQUcsRUFBRSxtQ0FBbUMsYUFBYSxFQUFFO3FCQUN4RDtvQkFDRCxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUM7b0JBQ3pELEdBQUcsZ0JBQWdCO29CQUNuQixXQUFXO2lCQUNaO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVPLFlBQVksQ0FBQyxJQUFnQixFQUFFLEtBQXNCLEVBQUUsZUFBd0I7UUFDckYsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUM1QixNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzlCLElBQUksQ0FBQyxNQUFNLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FDeEY7UUFFRCxNQUFNLE9BQU8sR0FBRyxDQUFDLENBQVMsRUFBVSxFQUFFO1lBQ3BDLE9BQU8sZ0NBQXVCLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRTtnQkFDeEMsU0FBUyxFQUFFLE9BQU87Z0JBQ2xCLE1BQU0sRUFBRSxNQUFNO2dCQUNkLFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxRQUFnQixFQUFFLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUM5RCxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO2dCQUN6QyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixJQUFJLGFBQWEsQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsT0FBTyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxjQUFjLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFlBQVksZUFBZSxLQUFLLENBQUMsQ0FBQztRQUN2RyxDQUFDLENBQUM7UUFFRixNQUFNLE1BQU0sR0FBd0I7WUFDbEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3ZCLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ3hELDRCQUE0QixFQUFFLEdBQUc7U0FDbEMsQ0FBQztRQUVGLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFLFlBQVksQ0FBQztRQUMvRSxJQUFJLFlBQVksRUFBRTtZQUNoQixNQUFNLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDckQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1NBQ3REO1FBQ0QsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRXJGLE9BQU87WUFDTCxFQUFFLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDakIsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRSxVQUFVLEtBQUssQ0FBQyxlQUFlLEVBQUU7Z0JBQ3ZDLEdBQUcsSUFBSSxDQUFDLFdBQVc7Z0JBQ25CLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLEVBQUUsUUFBUTtnQkFDeEQsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQ25DLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsRUFBRTtpQkFDN0M7Z0JBQ0QsR0FBRyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUM7b0JBQzdELFdBQVcsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxXQUFXO2lCQUNyRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ1AsS0FBSyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUM7Z0JBQ3BDLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07Z0JBQzFCLEtBQUssRUFBRTtvQkFDTCxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDO29CQUNsRDt3QkFDRSxFQUFFLEVBQUUsUUFBUTt3QkFDWixJQUFJLEVBQUUscURBQXFEO3dCQUMzRCxJQUFJLEVBQUUsTUFBTTtxQkFDYjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFTyxlQUFlLENBQUMsSUFBZ0IsRUFBRSxJQUFVO1FBQ2xELElBQUksQ0FBQyxDQUFDLElBQUksWUFBWSxxQkFBUyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1NBQ25EO1FBRUQsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUUvQixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3RELElBQUksRUFBRSxTQUFTO2dCQUNmLEdBQUcsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7YUFDckMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFUixPQUFPO1lBQ0wsRUFBRSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ2pCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsWUFBWTtnQkFDbEIsR0FBRyxJQUFJLENBQUMsV0FBVztnQkFDbkIsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQ25DLHVGQUF1RjtvQkFDdkYsOENBQThDO29CQUM5QyxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLEVBQUU7aUJBQzdDO2dCQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07Z0JBQzFCLEtBQUssRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDO2dCQUNwQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjO2dCQUM5QixLQUFLLEVBQUU7b0JBQ0wsR0FBRyxJQUFJLENBQUMsZUFBZSxFQUFFO29CQUN6QixHQUFHLElBQUksQ0FBQyxhQUFhO29CQUNyQixHQUFHLFlBQVk7b0JBQ2Y7d0JBQ0UsSUFBSSxFQUFFLE9BQU87d0JBQ2IsR0FBRyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztxQkFDOUI7b0JBQ0QsR0FBRyxJQUFJLENBQUMsY0FBYztvQkFDdEIsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQztpQkFDaEQ7YUFDRjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ssaUJBQWlCLENBQUMsR0FBeUIsRUFBRSxJQUFnQjtRQUNuRSxLQUFLLE1BQU0sR0FBRyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDOUIsSUFBSSxHQUFHLENBQUMsSUFBSSxFQUFFLElBQUksS0FBSyxTQUFTLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUNwRSxPQUFPLEdBQUcsQ0FBQyxRQUFRLENBQUM7YUFDckI7U0FDRjtRQUNELHNCQUFzQjtRQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLGNBQWMsR0FBRyxDQUFDLFVBQVUsb0RBQW9ELENBQUMsQ0FBQztJQUNwRyxDQUFDO0lBRU8sWUFBWSxDQUFDLEtBQWEsRUFBRSxNQUE0QjtRQUM5RCxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ3hDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNuQzthQUFNO1lBQ0wsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDckM7SUFDSCxDQUFDO0lBRU8sZ0JBQWdCLENBQUMsSUFBZ0IsRUFBRSxJQUFlO1FBQ3hELE1BQU0sWUFBWSxHQUEyQixFQUFFLENBQUM7UUFDaEQsS0FBSyxNQUFNLENBQUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUU7WUFDbkUsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUNoRCxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRTtnQkFDdkIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVO2dCQUMxQixNQUFNLEVBQUUsUUFBUTthQUNqQixDQUFDLENBQUM7WUFDSCxZQUFZLENBQUMsT0FBTyxDQUFDLEdBQUcsY0FBYyxLQUFLLFlBQVksR0FBRyxDQUFDLFVBQVUsS0FBSyxDQUFDO1NBQzVFO1FBRUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxLQUFLLEVBQWtCLENBQUM7UUFDbkQsTUFBTSxhQUFhLEdBQUcsSUFBSSxLQUFLLEVBQWtCLENBQUM7UUFFbEQsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQy9CLGNBQWMsQ0FBQyxJQUFJLENBQUM7Z0JBQ2xCLElBQUksRUFBRSw4QkFBOEI7Z0JBQ3BDLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFO29CQUN0QixJQUFJLEVBQUUsS0FBSyxDQUFDLFNBQVM7aUJBQ3RCO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDakMsYUFBYSxDQUFDLElBQUksQ0FBQztnQkFDakIsSUFBSSxFQUFFLDRCQUE0QjtnQkFDbEMsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLEVBQUU7b0JBQ3ZCLElBQUksRUFBRSxNQUFNLENBQUMsU0FBUztpQkFDdkI7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdEQsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsR0FBRyxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQzthQUNyQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVSLE9BQU87WUFDTCxFQUFFLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDakIsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRSxJQUFJLENBQUMsRUFBRTtnQkFDYixHQUFHLElBQUksQ0FBQyxXQUFXO2dCQUNuQixXQUFXLEVBQUU7b0JBQ1gsUUFBUSxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSTtpQkFDcEM7Z0JBQ0QsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTtnQkFDMUIsS0FBSyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUM7Z0JBQ3BDLEdBQUcsRUFBRTtvQkFDSCxHQUFHLElBQUksQ0FBQyxHQUFHO29CQUNYLEdBQUcsWUFBWTtpQkFDaEI7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLEdBQUcsY0FBYztvQkFDakIsR0FBRyxZQUFZO29CQUNmLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNqQyxHQUFHLGFBQWE7aUJBQ2pCO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVPLHNCQUFzQixDQUFDLElBQWdCLEVBQUUsSUFBc0I7UUFDckUsT0FBTztZQUNMLEVBQUUsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUNqQixVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFLElBQUksQ0FBQyxFQUFFO2dCQUNiLEdBQUcsSUFBSSxDQUFDLFdBQVc7Z0JBQ25CLFdBQVcsRUFBRTtvQkFDWCxRQUFRLEVBQUUsTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFLO2lCQUNyQztnQkFDRCxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUMxQixLQUFLLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQztnQkFDcEMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLEtBQUssRUFBRSxJQUFJLENBQUMsUUFBUTthQUNyQjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sbUJBQW1CLENBQUMsTUFBYyxFQUFFLGFBQXNCO1FBQ2hFLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxlQUFlLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxnQkFBa0M7UUFDL0QsSUFBSSxNQUEyQixDQUFDO1FBRWhDLElBQUksZ0JBQWdCLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtZQUN0QyxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixnQkFBZ0IsQ0FBQyxXQUFXLEtBQUs7Z0JBQzNELFFBQVEsRUFBRSxnQkFBZ0IsZ0JBQWdCLENBQUMsV0FBVyxLQUFLO2FBQzVELENBQUM7U0FDSDthQUFNLElBQUksZ0JBQWdCLENBQUMsSUFBSSxLQUFLLEtBQUssRUFBRTtZQUMxQyxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7YUFDcEMsQ0FBQztTQUNIO2FBQU07WUFDTCxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7Z0JBQ25DLFFBQVEsRUFBRSxnQkFBZ0IsZ0JBQWdCLENBQUMsV0FBVyxLQUFLO2dCQUMzRCxRQUFRLEVBQUUsZ0JBQWdCLGdCQUFnQixDQUFDLFdBQVcsS0FBSzthQUM1RCxDQUFDO1NBQ0g7UUFFRCxPQUFPO1lBQ0w7Z0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtnQkFDOUIsSUFBSSxFQUFFLE1BQU07YUFDYjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sdUJBQXVCLENBQUMsU0FBaUI7UUFDL0MsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE9BQU8sSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO1NBQy9CO1FBRUQsT0FBTyxDQUFDO2dCQUNOLElBQUksRUFBRSxZQUFZLGVBQWUsRUFBRTtnQkFDbkMsSUFBSSxFQUFFLDhCQUE4QjtnQkFDcEMsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxlQUFlO29CQUNyQixJQUFJLEVBQUUsU0FBUztpQkFDaEI7YUFDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sZUFBZTtRQUNyQixPQUFPLENBQUM7Z0JBQ04sSUFBSSxFQUFFLFVBQVU7Z0JBQ2hCLElBQUksRUFBRSxxQkFBcUI7YUFDNUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLHFCQUFxQixDQUFDLEdBQVc7UUFDdkMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE9BQU8sRUFBRSxDQUFDO1NBQ1g7UUFFRCxPQUFPLENBQUM7Z0JBQ04sSUFBSSxFQUFFLFVBQVUsZUFBZSxFQUFFO2dCQUNqQyxJQUFJLEVBQUUsNEJBQTRCO2dCQUNsQyxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLElBQUksRUFBRSxHQUFHO2lCQUNWO2FBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGtCQUFrQixDQUFDLElBQWdCO1FBQ3pDLE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxFQUFjLENBQUM7UUFFckMsS0FBSyxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQzVCLElBQUksQ0FBQyxZQUFZLHdCQUFLLEVBQUU7Z0JBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDbkM7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUNkO1NBQ0Y7UUFFRCxPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDbkMsQ0FBQzs7QUExbUJILHdDQTJtQkM7OztBQW1CRCxTQUFTLGFBQWEsQ0FBYyxHQUFNLEVBQUUsR0FBRyxHQUFHLEdBQUc7SUFDbkQsSUFBSSxPQUFPLEdBQUcsS0FBSyxRQUFRLElBQUksR0FBRyxJQUFJLElBQUksRUFBRTtRQUMxQyxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3RCLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQVEsQ0FBQztLQUNuRDtJQUVELE1BQU0sTUFBTSxHQUE0QixFQUFFLENBQUM7SUFDM0MsS0FBSyxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEMsb0RBQW9EO1FBQ3BELElBQUksQ0FBQyxLQUFLLEtBQUssSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxJQUFJLElBQUksRUFBRTtZQUNyRCxDQUFDLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3RCO1FBQ0QsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUMvQztJQUNELE9BQU8sTUFBYSxDQUFDO0FBQ3ZCLENBQUM7QUFzQkQsUUFBZSxDQUFDLENBQUMsT0FBTyxDQUFJLEVBQWlCO0lBQzNDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2xCLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2pCLE1BQU0sQ0FBQyxDQUFDO1NBQ1Q7S0FDRjtBQUNILENBQUM7QUFORCwwQkFNQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IG1rZGlyU3luYywgd3JpdGVGaWxlU3luYywgcmVhZEZpbGVTeW5jLCBleGlzdHNTeW5jIH0gZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IFN0YWdlIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgRW52aXJvbm1lbnRQbGFjZWhvbGRlcnMgfSBmcm9tICdhd3MtY2RrLWxpYi9jeC1hcGknO1xuaW1wb3J0IHsgUGlwZWxpbmVCYXNlLCBQaXBlbGluZUJhc2VQcm9wcywgU2hlbGxTdGVwLCBTdGFja0Fzc2V0LCBTdGFja0RlcGxveW1lbnQsIFN0YWNrT3V0cHV0UmVmZXJlbmNlLCBTdGFnZURlcGxveW1lbnQsIFN0ZXAgfSBmcm9tICdhd3MtY2RrLWxpYi9waXBlbGluZXMnO1xuaW1wb3J0IHsgQUdyYXBoTm9kZSwgUGlwZWxpbmVHcmFwaCwgR3JhcGgsIGlzR3JhcGggfSBmcm9tICdhd3MtY2RrLWxpYi9waXBlbGluZXMvbGliL2hlbHBlcnMtaW50ZXJuYWwnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBkZWNhbWVsaXplIGZyb20gJ2RlY2FtZWxpemUnO1xuaW1wb3J0IHsgQXdzQ3JlZGVudGlhbHMsIEF3c0NyZWRlbnRpYWxzUHJvdmlkZXIgfSBmcm9tICcuL2F3cy1jcmVkZW50aWFscyc7XG5pbXBvcnQgeyBEb2NrZXJDcmVkZW50aWFsIH0gZnJvbSAnLi9kb2NrZXItY3JlZGVudGlhbHMnO1xuaW1wb3J0IHsgQWRkR2l0SHViU3RhZ2VPcHRpb25zIH0gZnJvbSAnLi9zdGFnZS1vcHRpb25zJztcbmltcG9ydCB7IEdpdEh1YkFjdGlvblN0ZXAgfSBmcm9tICcuL3N0ZXBzL2dpdGh1Yi1hY3Rpb24tc3RlcCc7XG5pbXBvcnQgKiBhcyBnaXRodWIgZnJvbSAnLi93b3JrZmxvd3MtbW9kZWwnO1xuaW1wb3J0IHsgWWFtbEZpbGUgfSBmcm9tICcuL3lhbWwtZmlsZSc7XG5cbmNvbnN0IENES09VVF9BUlRJRkFDVCA9ICdjZGsub3V0JztcbmNvbnN0IEFTU0VUX0hBU0hfTkFNRSA9ICdhc3NldC1oYXNoJztcblxuLyoqXG4gKiBKb2IgbGV2ZWwgc2V0dGluZ3MgYXBwbGllZCB0byBhbGwgam9icyBpbiB0aGUgd29ya2Zsb3cuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iU2V0dGluZ3Mge1xuICAvKipcbiAgICogam9icy48am9iX2lkPi5pZi5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9hY3Rpb25zL3VzaW5nLXdvcmtmbG93cy93b3JrZmxvdy1zeW50YXgtZm9yLWdpdGh1Yi1hY3Rpb25zI2pvYnNqb2JfaWRpZlxuICAgKi9cbiAgcmVhZG9ubHkgaWY/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcHMgZm9yIGBHaXRIdWJXb3JrZmxvd2AuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR2l0SHViV29ya2Zsb3dQcm9wcyBleHRlbmRzIFBpcGVsaW5lQmFzZVByb3BzIHtcbiAgLyoqXG4gICAqIEZpbGUgcGF0aCBmb3IgdGhlIEdpdEh1YiB3b3JrZmxvdy5cbiAgICpcbiAgICogQGRlZmF1bHQgXCIuZ2l0aHViL3dvcmtmbG93cy9kZXBsb3kueW1sXCJcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93UGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGUgd29ya2Zsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiZGVwbG95XCJcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogR2l0SHViIHdvcmtmbG93IHRyaWdnZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEJ5IGRlZmF1bHQsIHdvcmtmbG93IGlzIHRyaWdnZXJlZCBvbiBwdXNoIHRvIHRoZSBgbWFpbmAgYnJhbmNoXG4gICAqIGFuZCBjYW4gYWxzbyBiZSB0cmlnZ2VyZWQgbWFudWFsbHkgKGB3b3JrZmxvd19kaXNwYXRjaGApLlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2Zsb3dUcmlnZ2Vycz86IGdpdGh1Yi5Xb3JrZmxvd1RyaWdnZXJzO1xuXG4gIC8qKlxuICAgKiBWZXJzaW9uIG9mIHRoZSBDREsgQ0xJIHRvIHVzZS5cbiAgICogQGRlZmF1bHQgLSBhdXRvbWF0aWNcbiAgICovXG4gIHJlYWRvbmx5IGNka0NsaVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiB0aGUgcmVwb3NpdG9yeSBhbHJlYWR5IGNvbnRhaW5zIGEgc3ludGhlc2l6ZWQgYGNkay5vdXRgIGRpcmVjdG9yeSwgaW4gd2hpY2hcbiAgICogY2FzZSB3ZSB3aWxsIHNpbXBseSBjaGVja291dCB0aGUgcmVwbyBpbiBqb2JzIHRoYXQgcmVxdWlyZSBgY2RrLm91dGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwcmVTeW50aGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ29uZmlndXJlIHByb3ZpZGVyIGZvciBBV1MgY3JlZGVudGlhbHMgdXNlZCBmb3IgZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBHZXQgQVdTIGNyZWRlbnRpYWxzIGZyb20gR2l0SHViIHNlY3JldHMgYEFXU19BQ0NFU1NfS0VZX0lEYCBhbmQgYEFXU19TRUNSRVRfQUNDRVNTX0tFWWAuXG4gICAqL1xuICByZWFkb25seSBhd3NDcmVkcz86IEF3c0NyZWRlbnRpYWxzUHJvdmlkZXI7XG5cbiAgLyoqXG4gICAqIE5hbWVzIG9mIEdpdEh1YiByZXBvc2l0b3J5IHNlY3JldHMgdGhhdCBpbmNsdWRlIEFXUyBjcmVkZW50aWFscyBmb3JcbiAgICogZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgQVdTX0FDQ0VTU19LRVlfSURgIGFuZCBgQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZYC5cbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBhd3NDcmVkcy5mcm9tR2l0SHViU2VjcmV0cygpYCBpbnN0ZWFkLlxuICAgKi9cbiAgcmVhZG9ubHkgYXdzQ3JlZGVudGlhbHM/OiBBd3NDcmVkZW50aWFsc1NlY3JldHM7XG5cbiAgLyoqXG4gICAqIEEgcm9sZSB0aGF0IHV0aWxpemVzIHRoZSBHaXRIdWIgT0lEQyBJZGVudGl0eSBQcm92aWRlciBpbiB5b3VyIEFXUyBhY2NvdW50LlxuICAgKiBJZiBzdXBwbGllZCwgdGhpcyB3aWxsIGJlIHVzZWQgaW5zdGVhZCBvZiBgYXdzQ3JlZGVudGlhbHNgLlxuICAgKlxuICAgKiBZb3UgY2FuIGNyZWF0ZSB5b3VyIG93biByb2xlIGluIHRoZSBjb25zb2xlIHdpdGggdGhlIG5lY2Vzc2FyeSB0cnVzdCBwb2xpY3lcbiAgICogdG8gYWxsb3cgZ2l0SHViIGFjdGlvbnMgZnJvbSB5b3VyIGdpdEh1YiByZXBvc2l0b3J5IHRvIGFzc3VtZSB0aGUgcm9sZSwgb3JcbiAgICogeW91IGNhbiB1dGlsaXplIHRoZSBgR2l0SHViQWN0aW9uUm9sZWAgY29uc3RydWN0IHRvIGNyZWF0ZSBhIHJvbGUgZm9yIHlvdS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBHaXRIdWIgcmVwb3NpdG9yeSBzZWNyZXRzIGFyZSB1c2VkIGluc3RlYWQgb2YgT3BlbklkIENvbm5lY3Qgcm9sZS5cbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBhd3NDcmVkcy5mcm9tT3BlbklkQ29ubmVjdCgpYCBpbnN0ZWFkLlxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0SHViQWN0aW9uUm9sZUFybj86IHN0cmluZztcblxuICAvKipcbiAgICogQnVpbGQgY29udGFpbmVyIG9wdGlvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gR2l0SHViIGRlZmF1bHRzXG4gICAqL1xuICByZWFkb25seSBidWlsZENvbnRhaW5lcj86IGdpdGh1Yi5Db250YWluZXJPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBHaXRIdWIgd29ya2Zsb3cgc3RlcHMgdG8gZXhlY3V0ZSBiZWZvcmUgYnVpbGQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBwcmVCdWlsZFN0ZXBzPzogZ2l0aHViLkpvYlN0ZXBbXTtcblxuICAvKipcbiAgICogR2l0SHViIHdvcmtmbG93IHN0ZXBzIHRvIGV4ZWN1dGUgYWZ0ZXIgYnVpbGQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBwb3N0QnVpbGRTdGVwcz86IGdpdGh1Yi5Kb2JTdGVwW107XG5cbiAgLyoqXG4gICAqIFRoZSBEb2NrZXIgQ3JlZGVudGlhbHMgdG8gdXNlIHRvIGxvZ2luLiBJZiB5b3Ugc2V0IHRoaXMgdmFyaWFibGUsXG4gICAqIHlvdSB3aWxsIGJlIGxvZ2dlZCBpbiB0byBkb2NrZXIgd2hlbiB5b3UgdXBsb2FkIERvY2tlciBBc3NldHMuXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJDcmVkZW50aWFscz86IERvY2tlckNyZWRlbnRpYWxbXTtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgcnVubmVyIHRvIHJ1biB0aGUgam9iIG9uLiBUaGUgcnVubmVyIGNhbiBiZSBlaXRoZXIgYVxuICAgKiBHaXRIdWItaG9zdGVkIHJ1bm5lciBvciBhIHNlbGYtaG9zdGVkIHJ1bm5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgUnVubmVyLlVCVU5UVV9MQVRFU1RcbiAgICovXG4gIHJlYWRvbmx5IHJ1bm5lcj86IGdpdGh1Yi5SdW5uZXI7XG5cbiAgLyoqXG4gICAqIFdpbGwgYXNzdW1lIHRoZSBHaXRIdWJBY3Rpb25Sb2xlIGluIHRoaXMgcmVnaW9uIHdoZW4gcHVibGlzaGluZyBhc3NldHMuXG4gICAqIFRoaXMgaXMgTk9UIHRoZSByZWdpb24gaW4gd2hpY2ggdGhlIGFzc2V0cyBhcmUgcHVibGlzaGVkLlxuICAgKlxuICAgKiBJbiBtb3N0IGNhc2VzLCB5b3UgZG8gbm90IGhhdmUgdG8gd29ycnkgYWJvdXQgdGhpcyBwcm9wZXJ0eSwgYW5kIGNhbiBzYWZlbHlcbiAgICogaWdub3JlIGl0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInVzLXdlc3QtMlwiXG4gICAqL1xuICByZWFkb25seSBwdWJsaXNoQXNzZXRzQXV0aFJlZ2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogSm9iIGxldmVsIHNldHRpbmdzIHRoYXQgd2lsbCBiZSBhcHBsaWVkIHRvIGFsbCBqb2JzIGluIHRoZSB3b3JrZmxvdyxcbiAgICogaW5jbHVkaW5nIHN5bnRoIGFuZCBhc3NldCBkZXBsb3kgam9icy4gQ3VycmVudGx5IHRoZSBvbmx5IHZhbGlkIHNldHRpbmdcbiAgICogaXMgJ2lmJy4gWW91IGNhbiB1c2UgdGhpcyB0byBydW4gam9icyBvbmx5IGluIHNwZWNpZmljIHJlcG9zaXRvcmllcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9hY3Rpb25zL3VzaW5nLXdvcmtmbG93cy93b3JrZmxvdy1zeW50YXgtZm9yLWdpdGh1Yi1hY3Rpb25zI2V4YW1wbGUtb25seS1ydW4tam9iLWZvci1zcGVjaWZpYy1yZXBvc2l0b3J5XG4gICAqL1xuICByZWFkb25seSBqb2JTZXR0aW5ncz86IEpvYlNldHRpbmdzO1xufVxuXG4vKipcbiAqIENESyBQaXBlbGluZXMgZm9yIEdpdEh1YiB3b3JrZmxvd3MuXG4gKi9cbmV4cG9ydCBjbGFzcyBHaXRIdWJXb3JrZmxvdyBleHRlbmRzIFBpcGVsaW5lQmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSB3b3JrZmxvd1BhdGg6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHdvcmtmbG93TmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgd29ya2Zsb3dGaWxlOiBZYW1sRmlsZTtcblxuICBwcml2YXRlIHJlYWRvbmx5IHdvcmtmbG93VHJpZ2dlcnM6IGdpdGh1Yi5Xb3JrZmxvd1RyaWdnZXJzO1xuICBwcml2YXRlIHJlYWRvbmx5IHByZVN5bnRoZWQ6IGJvb2xlYW47XG4gIHByaXZhdGUgcmVhZG9ubHkgYXdzQ3JlZGVudGlhbHM6IEF3c0NyZWRlbnRpYWxzUHJvdmlkZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgZG9ja2VyQ3JlZGVudGlhbHM6IERvY2tlckNyZWRlbnRpYWxbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBjZGtDbGlWZXJzaW9uPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGJ1aWxkQ29udGFpbmVyPzogZ2l0aHViLkNvbnRhaW5lck9wdGlvbnM7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlQnVpbGRTdGVwczogZ2l0aHViLkpvYlN0ZXBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBwb3N0QnVpbGRTdGVwczogZ2l0aHViLkpvYlN0ZXBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBqb2JPdXRwdXRzOiBSZWNvcmQ8c3RyaW5nLCBnaXRodWIuSm9iU3RlcE91dHB1dFtdPiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IGFzc2V0SGFzaE1hcDogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IHJ1bm5lcjogZ2l0aHViLlJ1bm5lcjtcbiAgcHJpdmF0ZSByZWFkb25seSBwdWJsaXNoQXNzZXRzQXV0aFJlZ2lvbjogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHN0YWNrUHJvcGVydGllczogUmVjb3JkPHN0cmluZywgUmVjb3JkPHN0cmluZywgYW55Pj4gPSB7fTtcbiAgcHJpdmF0ZSByZWFkb25seSBqb2JTZXR0aW5ncz86IEpvYlNldHRpbmdzO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBHaXRIdWJXb3JrZmxvd1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmNka0NsaVZlcnNpb24gPSBwcm9wcy5jZGtDbGlWZXJzaW9uO1xuICAgIHRoaXMucHJlU3ludGhlZCA9IHByb3BzLnByZVN5bnRoZWQgPz8gZmFsc2U7XG4gICAgdGhpcy5idWlsZENvbnRhaW5lciA9IHByb3BzLmJ1aWxkQ29udGFpbmVyO1xuICAgIHRoaXMucHJlQnVpbGRTdGVwcyA9IHByb3BzLnByZUJ1aWxkU3RlcHMgPz8gW107XG4gICAgdGhpcy5wb3N0QnVpbGRTdGVwcyA9IHByb3BzLnBvc3RCdWlsZFN0ZXBzID8/IFtdO1xuICAgIHRoaXMuam9iU2V0dGluZ3MgPSBwcm9wcy5qb2JTZXR0aW5ncztcblxuICAgIHRoaXMuYXdzQ3JlZGVudGlhbHMgPSB0aGlzLmdldEF3c0NyZWRlbnRpYWxzKHByb3BzKTtcblxuICAgIHRoaXMuZG9ja2VyQ3JlZGVudGlhbHMgPSBwcm9wcy5kb2NrZXJDcmVkZW50aWFscyA/PyBbXTtcblxuICAgIHRoaXMud29ya2Zsb3dQYXRoID0gcHJvcHMud29ya2Zsb3dQYXRoID8/ICcuZ2l0aHViL3dvcmtmbG93cy9kZXBsb3kueW1sJztcbiAgICBpZiAoIXRoaXMud29ya2Zsb3dQYXRoLmVuZHNXaXRoKCcueW1sJykgJiYgIXRoaXMud29ya2Zsb3dQYXRoLmVuZHNXaXRoKCcueWFtbCcpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3dvcmtmbG93IGZpbGUgaXMgZXhwZWN0ZWQgdG8gYmUgYSB5YW1sIGZpbGUnKTtcbiAgICB9XG4gICAgaWYgKCF0aGlzLndvcmtmbG93UGF0aC5pbmNsdWRlcygnLmdpdGh1Yi93b3JrZmxvd3MvJykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignd29ya2Zsb3cgZmlsZXMgbXVzdCBiZSBzdG9yZWQgaW4gdGhlIFxcJy5naXRodWIvd29ya2Zsb3dzXFwnIGRpcmVjdG9yeSBvZiB5b3VyIHJlcG9zaXRvcnknKTtcbiAgICB9XG5cbiAgICB0aGlzLndvcmtmbG93RmlsZSA9IG5ldyBZYW1sRmlsZSh0aGlzLndvcmtmbG93UGF0aCk7XG4gICAgdGhpcy53b3JrZmxvd05hbWUgPSBwcm9wcy53b3JrZmxvd05hbWUgPz8gJ2RlcGxveSc7XG4gICAgdGhpcy53b3JrZmxvd1RyaWdnZXJzID0gcHJvcHMud29ya2Zsb3dUcmlnZ2VycyA/PyB7XG4gICAgICBwdXNoOiB7IGJyYW5jaGVzOiBbJ21haW4nXSB9LFxuICAgICAgd29ya2Zsb3dEaXNwYXRjaDoge30sXG4gICAgfTtcblxuICAgIHRoaXMucnVubmVyID0gcHJvcHMucnVubmVyID8/IGdpdGh1Yi5SdW5uZXIuVUJVTlRVX0xBVEVTVDtcbiAgICB0aGlzLnB1Ymxpc2hBc3NldHNBdXRoUmVnaW9uID0gcHJvcHMucHVibGlzaEFzc2V0c0F1dGhSZWdpb24gPz8gJ3VzLXdlc3QtMic7XG4gIH1cblxuICAvKipcbiAgICogUGFyc2UgQVdTIGNyZWRlbnRpYWwgY29uZmlndXJhdGlvbiBmcm9tIGRlcHJlY2F0ZWQgcHJvcGVydGllcyBGb3IgYmFja3dhcmRzIGNvbXBhdGliaWxpdHkuXG4gICAqL1xuICBwcml2YXRlIGdldEF3c0NyZWRlbnRpYWxzKHByb3BzOiBHaXRIdWJXb3JrZmxvd1Byb3BzKSB7XG4gICAgaWYgKHByb3BzLmdpdEh1YkFjdGlvblJvbGVBcm4pIHtcbiAgICAgIGlmIChwcm9wcy5hd3NDcmVkcykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1BsZWFzZSBwcm92aWRlIG9ubHkgb25lIG1ldGhvZCBvZiBhdXRoZW50aWNhdGlvbiAocmVtb3ZlIGdpdGh1YkFjdGlvblJvbGVBcm4pJyk7XG4gICAgICB9XG4gICAgICByZXR1cm4gQXdzQ3JlZGVudGlhbHMuZnJvbU9wZW5JZENvbm5lY3Qoe1xuICAgICAgICBnaXRIdWJBY3Rpb25Sb2xlQXJuOiBwcm9wcy5naXRIdWJBY3Rpb25Sb2xlQXJuLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmF3c0NyZWRlbnRpYWxzKSB7XG4gICAgICBpZiAocHJvcHMuYXdzQ3JlZHMpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQbGVhc2UgcHJvdmlkZSBvbmx5IG9uZSBtZXRob2Qgb2YgYXV0aGVudGljYXRpb24gKHJlbW92ZSBhd3NDcmVkZW50aWFscyknKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiBBd3NDcmVkZW50aWFscy5mcm9tR2l0SHViU2VjcmV0cyh7XG4gICAgICAgIGFjY2Vzc0tleUlkOiAnQVdTX0FDQ0VTU19LRVlfSUQnLFxuICAgICAgICBzZWNyZXRBY2Nlc3NLZXk6ICdBV1NfU0VDUkVUX0FDQ0VTU19LRVknLFxuICAgICAgICAuLi5wcm9wcy5hd3NDcmVkZW50aWFscyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBwcm9wcy5hd3NDcmVkcyA/PyBBd3NDcmVkZW50aWFscy5mcm9tR2l0SHViU2VjcmV0cygpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlcGxveSBhIHNpbmdsZSBTdGFnZSBieSBpdHNlbGYgd2l0aCBvcHRpb25zIGZvciBmdXJ0aGVyIEdpdEh1YiBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBBZGQgYSBTdGFnZSB0byB0aGUgcGlwZWxpbmUsIHRvIGJlIGRlcGxveWVkIGluIHNlcXVlbmNlIHdpdGggb3RoZXIgU3RhZ2VzIGFkZGVkIHRvIHRoZSBwaXBlbGluZS5cbiAgICogQWxsIFN0YWNrcyBpbiB0aGUgc3RhZ2Ugd2lsbCBiZSBkZXBsb3llZCBpbiBhbiBvcmRlciBhdXRvbWF0aWNhbGx5IGRldGVybWluZWQgYnkgdGhlaXIgcmVsYXRpdmUgZGVwZW5kZW5jaWVzLlxuICAgKi9cbiAgcHVibGljIGFkZFN0YWdlV2l0aEdpdEh1Yk9wdGlvbnMoc3RhZ2U6IFN0YWdlLCBvcHRpb25zPzogQWRkR2l0SHViU3RhZ2VPcHRpb25zKTogU3RhZ2VEZXBsb3ltZW50IHtcbiAgICBjb25zdCBzdGFnZURlcGxveW1lbnQgPSB0aGlzLmFkZFN0YWdlKHN0YWdlLCBvcHRpb25zKTtcblxuICAgIC8vIGtlZXAgdHJhY2sgb2YgR2l0SHViIHNwZWNpZmljIG9wdGlvbnNcbiAgICBjb25zdCBzdGFja3MgPSBzdGFnZURlcGxveW1lbnQuc3RhY2tzO1xuICAgIHRoaXMuYWRkU3RhY2tQcm9wcyhzdGFja3MsICdlbnZpcm9ubWVudCcsIG9wdGlvbnM/LmdpdEh1YkVudmlyb25tZW50KTtcbiAgICB0aGlzLmFkZFN0YWNrUHJvcHMoc3RhY2tzLCAnY2FwYWJpbGl0aWVzJywgb3B0aW9ucz8uc3RhY2tDYXBhYmlsaXRpZXMpO1xuICAgIHRoaXMuYWRkU3RhY2tQcm9wcyhzdGFja3MsICdzZXR0aW5ncycsIG9wdGlvbnM/LmpvYlNldHRpbmdzKTtcblxuICAgIHJldHVybiBzdGFnZURlcGxveW1lbnQ7XG4gIH1cblxuICBwcml2YXRlIGFkZFN0YWNrUHJvcHMoc3RhY2tzOiBTdGFja0RlcGxveW1lbnRbXSwga2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpIHtcbiAgICBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkgeyByZXR1cm47IH1cbiAgICBmb3IgKGNvbnN0IHN0YWNrIG9mIHN0YWNrcykge1xuICAgICAgdGhpcy5zdGFja1Byb3BlcnRpZXNbc3RhY2suc3RhY2tBcnRpZmFjdElkXSA9IHtcbiAgICAgICAgLi4udGhpcy5zdGFja1Byb3BlcnRpZXNbc3RhY2suc3RhY2tBcnRpZmFjdElkXSxcbiAgICAgICAgW2tleV06IHZhbHVlLFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwcm90ZWN0ZWQgZG9CdWlsZFBpcGVsaW5lKCkge1xuICAgIGNvbnN0IGFwcCA9IFN0YWdlLm9mKHRoaXMpO1xuICAgIGlmICghYXBwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSBHaXRIdWIgV29ya2Zsb3cgbXVzdCBiZSBkZWZpbmVkIGluIHRoZSBzY29wZSBvZiBhbiBBcHAnKTtcbiAgICB9XG4gICAgY29uc3QgY2Rrb3V0RGlyID0gYXBwLm91dGRpcjtcblxuICAgIGNvbnN0IGpvYnMgPSBuZXcgQXJyYXk8Sm9iPigpO1xuXG4gICAgY29uc3Qgc3RydWN0dXJlID0gbmV3IFBpcGVsaW5lR3JhcGgodGhpcywge1xuICAgICAgc2VsZk11dGF0aW9uOiBmYWxzZSxcbiAgICAgIHB1Ymxpc2hUZW1wbGF0ZTogdHJ1ZSxcbiAgICAgIHByZXBhcmVTdGVwOiBmYWxzZSwgLy8gd2UgY3JlYXRlIGFuZCBleGVjdXRlIHRoZSBjaGFuZ2VzZXQgaW4gYSBzaW5nbGUgam9iXG4gICAgfSk7XG5cbiAgICBmb3IgKGNvbnN0IHN0YWdlTm9kZSBvZiBmbGF0dGVuKHN0cnVjdHVyZS5ncmFwaC5zb3J0ZWRDaGlsZHJlbigpKSkge1xuICAgICAgaWYgKCFpc0dyYXBoKHN0YWdlTm9kZSkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUb3AtbGV2ZWwgY2hpbGRyZW4gbXVzdCBiZSBncmFwaHMsIGdvdCAnJHtzdGFnZU5vZGV9J2ApO1xuICAgICAgfVxuXG4gICAgICBjb25zdCB0cmFuY2hlcyA9IHN0YWdlTm9kZS5zb3J0ZWRMZWF2ZXMoKTtcblxuICAgICAgZm9yIChjb25zdCB0cmFuY2hlIG9mIHRyYW5jaGVzKSB7XG4gICAgICAgIGZvciAoY29uc3Qgbm9kZSBvZiB0cmFuY2hlKSB7XG4gICAgICAgICAgY29uc3Qgam9iID0gdGhpcy5qb2JGb3JOb2RlKG5vZGUsIHtcbiAgICAgICAgICAgIGFzc2VtYmx5RGlyOiBjZGtvdXREaXIsXG4gICAgICAgICAgICBzdHJ1Y3R1cmUsXG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgICBpZiAoam9iKSB7XG4gICAgICAgICAgICBqb2JzLnB1c2goam9iKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBjb252ZXJ0IGpvYnMgdG8gYSBtYXAgYW5kIG1ha2Ugc3VyZSB0aGVyZSBhcmUgbm8gZHVwbGljYXRlc1xuICAgIGNvbnN0IGpvYm1hcDogUmVjb3JkPHN0cmluZywgZ2l0aHViLkpvYj4gPSB7fTtcbiAgICBmb3IgKGNvbnN0IGpvYiBvZiBqb2JzKSB7XG4gICAgICBpZiAoam9iLmlkIGluIGpvYm1hcCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGR1cGxpY2F0ZSBqb2IgaWQgJHtqb2IuaWR9YCk7XG4gICAgICB9XG4gICAgICBqb2JtYXBbam9iLmlkXSA9IHNuYWtlQ2FzZUtleXMoam9iLmRlZmluaXRpb24pO1xuICAgIH1cblxuICAgIC8vIFVwZGF0ZSBqb2JzIHdpdGggbGF0ZS1ib3VuZCBvdXRwdXQgcmVxdWVzdHNcbiAgICB0aGlzLmluc2VydEpvYk91dHB1dHMoam9ibWFwKTtcblxuICAgIGNvbnN0IHdvcmtmbG93ID0ge1xuICAgICAgbmFtZTogdGhpcy53b3JrZmxvd05hbWUsXG4gICAgICBvbjogc25ha2VDYXNlS2V5cyh0aGlzLndvcmtmbG93VHJpZ2dlcnMsICdfJyksXG4gICAgICBqb2JzOiBqb2JtYXAsXG4gICAgfTtcblxuICAgIC8vIHdyaXRlIGFzIGEgeWFtbCBmaWxlXG4gICAgdGhpcy53b3JrZmxvd0ZpbGUudXBkYXRlKHdvcmtmbG93KTtcblxuICAgIC8vIGNyZWF0ZSBkaXJlY3RvcnkgaWYgaXQgZG9lcyBub3QgZXhpc3RcbiAgICBta2RpclN5bmMocGF0aC5kaXJuYW1lKHRoaXMud29ya2Zsb3dQYXRoKSwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG5cbiAgICAvLyBHSVRIVUJfV09SS0ZMT1cgaXMgc2V0IHdoZW4gR2l0SHViIEFjdGlvbnMgaXMgcnVubmluZyB0aGUgd29ya2Zsb3cuXG4gICAgLy8gc2VlOiBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9hY3Rpb25zL2xlYXJuLWdpdGh1Yi1hY3Rpb25zL2Vudmlyb25tZW50LXZhcmlhYmxlcyNkZWZhdWx0LWVudmlyb25tZW50LXZhcmlhYmxlc1xuICAgIGNvbnN0IGNvbnRleHRWYWx1ZSA9IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdjZGstcGlwZWxpbmVzLWdpdGh1YjpkaWZmUHJvdGVjdGlvbicpO1xuICAgIGNvbnN0IGRpZmZQcm90ZWN0aW9uID0gY29udGV4dFZhbHVlID09PSAnZmFsc2UnID8gZmFsc2UgOiBjb250ZXh0VmFsdWUgPz8gdHJ1ZTtcbiAgICBpZiAoZGlmZlByb3RlY3Rpb24gJiYgcHJvY2Vzcy5lbnYuR0lUSFVCX1dPUktGTE9XID09PSB0aGlzLndvcmtmbG93TmFtZSkge1xuICAgICAgLy8gY2hlY2sgaWYgd29ya2Zsb3cgZmlsZSBoYXMgY2hhbmdlZFxuICAgICAgaWYgKCFleGlzdHNTeW5jKHRoaXMud29ya2Zsb3dQYXRoKSB8fCB0aGlzLndvcmtmbG93RmlsZS50b1lhbWwoKSAhPT0gcmVhZEZpbGVTeW5jKHRoaXMud29ya2Zsb3dQYXRoLCAndXRmOCcpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgUGxlYXNlIGNvbW1pdCB0aGUgdXBkYXRlZCB3b3JrZmxvdyBmaWxlICR7cGF0aC5yZWxhdGl2ZShfX2Rpcm5hbWUsIHRoaXMud29ya2Zsb3dQYXRoKX0gd2hlbiB5b3UgY2hhbmdlIHlvdXIgcGlwZWxpbmUgZGVmaW5pdGlvbi5gKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLndvcmtmbG93RmlsZS53cml0ZUZpbGUoKTtcbiAgfVxuXG4gIHByaXZhdGUgaW5zZXJ0Sm9iT3V0cHV0cyhqb2JtYXA6IFJlY29yZDxzdHJpbmcsIGdpdGh1Yi5Kb2I+KSB7XG4gICAgZm9yIChjb25zdCBbam9iSWQsIGpvYk91dHB1dHNdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuam9iT3V0cHV0cykpIHtcbiAgICAgIGpvYm1hcFtqb2JJZF0gPSB7XG4gICAgICAgIC4uLmpvYm1hcFtqb2JJZF0sXG4gICAgICAgIG91dHB1dHM6IHtcbiAgICAgICAgICAuLi5qb2JtYXBbam9iSWRdLm91dHB1dHMsXG4gICAgICAgICAgLi4udGhpcy5yZW5kZXJKb2JPdXRwdXRzKGpvYk91dHB1dHMpLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckpvYk91dHB1dHMob3V0cHV0czogZ2l0aHViLkpvYlN0ZXBPdXRwdXRbXSkge1xuICAgIGNvbnN0IHJlbmRlcmVkT3V0cHV0czogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICAgIGZvciAoY29uc3Qgb3V0cHV0IG9mIG91dHB1dHMpIHtcbiAgICAgIHJlbmRlcmVkT3V0cHV0c1tvdXRwdXQub3V0cHV0TmFtZV0gPSBgXFwke3sgc3RlcHMuJHtvdXRwdXQuc3RlcElkfS5vdXRwdXRzLiR7b3V0cHV0Lm91dHB1dE5hbWV9IH19YDtcbiAgICB9XG4gICAgcmV0dXJuIHJlbmRlcmVkT3V0cHV0cztcbiAgfVxuXG4gIC8qKlxuICAgKiBNYWtlIGFuIGFjdGlvbiBmcm9tIHRoZSBnaXZlbiBub2RlIGFuZC9vciBzdGVwXG4gICAqL1xuICBwcml2YXRlIGpvYkZvck5vZGUobm9kZTogQUdyYXBoTm9kZSwgb3B0aW9uczogQ29udGV4dCk6IEpvYiB8IHVuZGVmaW5lZCB7XG4gICAgc3dpdGNoIChub2RlLmRhdGE/LnR5cGUpIHtcbiAgICAgIC8vIE5vdGhpbmcgZm9yIHRoZXNlLCB0aGV5IGFyZSBncm91cGluZ3MgKHNob3VsZG4ndCBldmVuIGhhdmUgcG9wcGVkIHVwIGhlcmUpXG4gICAgICBjYXNlICdncm91cCc6XG4gICAgICBjYXNlICdzdGFjay1ncm91cCc6XG4gICAgICBjYXNlIHVuZGVmaW5lZDpcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBqb2JGb3JOb2RlOiBkaWQgbm90IGV4cGVjdCB0byBnZXQgZ3JvdXAgbm9kZXM6ICR7bm9kZS5kYXRhPy50eXBlfWApO1xuXG4gICAgICBjYXNlICdzZWxmLXVwZGF0ZSc6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignR2l0SHViIFdvcmtmbG93cyBkb2VzIG5vdCBzdXBwb3J0IHNlbGYgbXV0YXRpb24nKTtcblxuICAgICAgY2FzZSAncHVibGlzaC1hc3NldHMnOlxuICAgICAgICByZXR1cm4gdGhpcy5qb2JGb3JBc3NldFB1Ymxpc2gobm9kZSwgbm9kZS5kYXRhLmFzc2V0cywgb3B0aW9ucyk7XG5cbiAgICAgIGNhc2UgJ3ByZXBhcmUnOlxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wicHJlcGFyZVwiIGlzIG5vdCBzdXBwb3J0ZWQgYnkgR2l0SHViIFdvcmtmbG93cycpO1xuXG4gICAgICBjYXNlICdleGVjdXRlJzpcbiAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yRGVwbG95KG5vZGUsIG5vZGUuZGF0YS5zdGFjaywgbm9kZS5kYXRhLmNhcHR1cmVPdXRwdXRzKTtcblxuICAgICAgY2FzZSAnc3RlcCc6XG4gICAgICAgIGlmIChub2RlLmRhdGEuaXNCdWlsZFN0ZXApIHtcbiAgICAgICAgICByZXR1cm4gdGhpcy5qb2JGb3JCdWlsZFN0ZXAobm9kZSwgbm9kZS5kYXRhLnN0ZXApO1xuICAgICAgICB9IGVsc2UgaWYgKG5vZGUuZGF0YS5zdGVwIGluc3RhbmNlb2YgU2hlbGxTdGVwKSB7XG4gICAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yU2NyaXB0U3RlcChub2RlLCBub2RlLmRhdGEuc3RlcCk7XG4gICAgICAgIH0gZWxzZSBpZiAobm9kZS5kYXRhLnN0ZXAgaW5zdGFuY2VvZiBHaXRIdWJBY3Rpb25TdGVwKSB7XG4gICAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yR2l0SHViQWN0aW9uU3RlcChub2RlLCBub2RlLmRhdGEuc3RlcCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB1bnN1cHBvcnRlZCBzdGVwIHR5cGU6ICR7bm9kZS5kYXRhLnN0ZXAuY29uc3RydWN0b3IubmFtZX1gKTtcbiAgICAgICAgfVxuXG4gICAgICBkZWZhdWx0OlxuICAgICAgICAvLyBUaGUgJ2FzIGFueScgaXMgdGVtcG9yYXJ5LCB1bnRpbCB0aGUgY2hhbmdlIHVwc3RyZWFtIHJvbGxzIG91dFxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEdpdEh1YldvcmZrbG93IGRvZXMgbm90IHN1cHBvcnQgZ3JhcGggbm9kZXMgb2YgdHlwZSAnJHsobm9kZS5kYXRhIGFzIGFueSk/LnR5cGV9Jy4gWW91IGFyZSBwcm9iYWJseSB1c2luZyBhIGZlYXR1cmUgdGhpcyBDREsgUGlwZWxpbmVzIGltcGxlbWVudGF0aW9uIGRvZXMgbm90IHN1cHBvcnQuYCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBqb2JGb3JBc3NldFB1Ymxpc2gobm9kZTogQUdyYXBoTm9kZSwgYXNzZXRzOiBTdGFja0Fzc2V0W10sIG9wdGlvbnM6IENvbnRleHQpOiBKb2Ige1xuICAgIGlmIChhc3NldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Fzc2V0IFB1Ymxpc2ggc3RlcCBtdXN0IGhhdmUgYXQgbGVhc3QgMSBhc3NldCcpO1xuICAgIH1cblxuICAgIGNvbnN0IGluc3RhbGxTdWZmaXggPSB0aGlzLmNka0NsaVZlcnNpb24gPyBgQCR7dGhpcy5jZGtDbGlWZXJzaW9ufWAgOiAnJztcbiAgICBjb25zdCBjZGtvdXREaXIgPSBvcHRpb25zLmFzc2VtYmx5RGlyO1xuICAgIGNvbnN0IGpvYklkID0gbm9kZS51bmlxdWVJZDtcbiAgICBjb25zdCBhc3NldElkID0gYXNzZXRzWzBdLmFzc2V0SWQ7XG5cbiAgICAvLyBjaGVjayBpZiBhc3NldCBpcyBkb2NrZXIgYXNzZXQgYW5kIGlmIHdlIGhhdmUgZG9ja2VyIGNyZWRlbnRpYWxzXG4gICAgY29uc3QgZG9ja2VyTG9naW5TdGVwczogZ2l0aHViLkpvYlN0ZXBbXSA9IFtdO1xuICAgIGlmIChub2RlLnVuaXF1ZUlkLmluY2x1ZGVzKCdEb2NrZXJBc3NldCcpICYmIHRoaXMuZG9ja2VyQ3JlZGVudGlhbHMubGVuZ3RoID4gMCkge1xuICAgICAgZm9yIChjb25zdCBjcmVkcyBvZiB0aGlzLmRvY2tlckNyZWRlbnRpYWxzKSB7XG4gICAgICAgIGRvY2tlckxvZ2luU3RlcHMucHVzaCguLi50aGlzLnN0ZXBzVG9Db25maWd1cmVEb2NrZXIoY3JlZHMpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBjcmVhdGUgb25lIGZpbGUgYW5kIG1ha2Ugb25lIHN0ZXBcbiAgICBjb25zdCByZWxhdGl2ZVRvQXNzZW1ibHkgPSAocDogc3RyaW5nKSA9PiBwYXRoLnBvc2l4LmpvaW4oY2Rrb3V0RGlyLCBwYXRoLnJlbGF0aXZlKHBhdGgucmVzb2x2ZShjZGtvdXREaXIpLCBwKSk7XG4gICAgY29uc3QgZmlsZUNvbnRlbnRzOiBzdHJpbmdbXSA9IFsnc2V0IC1leCddLmNvbmNhdChhc3NldHMubWFwKChhc3NldCkgPT4ge1xuICAgICAgcmV0dXJuIGBucHggY2RrLWFzc2V0cyAtLXBhdGggXCIke3JlbGF0aXZlVG9Bc3NlbWJseShhc3NldC5hc3NldE1hbmlmZXN0UGF0aCl9XCIgLS12ZXJib3NlIHB1Ymxpc2ggXCIke2Fzc2V0LmFzc2V0U2VsZWN0b3J9XCJgO1xuICAgIH0pKTtcblxuICAgIC8vIHdlIG5lZWQgdGhlIGpvYklkIHRvIHJlZmVyZW5jZSB0aGUgb3V0cHV0cyBsYXRlclxuICAgIHRoaXMuYXNzZXRIYXNoTWFwW2Fzc2V0SWRdID0gam9iSWQ7XG4gICAgZmlsZUNvbnRlbnRzLnB1c2goYGVjaG8gJyR7QVNTRVRfSEFTSF9OQU1FfT0ke2Fzc2V0SWR9JyA+PiAkR0lUSFVCX09VVFBVVGApO1xuXG4gICAgY29uc3QgcHVibGlzaFN0ZXBGaWxlID0gcGF0aC5qb2luKGNka291dERpciwgYHB1Ymxpc2gtJHtqb2JJZH0tc3RlcC5zaGApO1xuICAgIG1rZGlyU3luYyhwYXRoLmRpcm5hbWUocHVibGlzaFN0ZXBGaWxlKSwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgd3JpdGVGaWxlU3luYyhwdWJsaXNoU3RlcEZpbGUsIGZpbGVDb250ZW50cy5qb2luKCdcXG4nKSwgeyBlbmNvZGluZzogJ3V0Zi04JyB9KTtcblxuICAgIGNvbnN0IHB1Ymxpc2hTdGVwOiBnaXRodWIuSm9iU3RlcCA9IHtcbiAgICAgIGlkOiAnUHVibGlzaCcsXG4gICAgICBuYW1lOiBgUHVibGlzaCAke2pvYklkfWAsXG4gICAgICBydW46IGAvYmluL2Jhc2ggLi9jZGsub3V0LyR7cGF0aC5yZWxhdGl2ZShjZGtvdXREaXIsIHB1Ymxpc2hTdGVwRmlsZSl9YCxcbiAgICB9O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiBqb2JJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogYFB1Ymxpc2ggQXNzZXRzICR7am9iSWR9YCxcbiAgICAgICAgLi4udGhpcy5qb2JTZXR0aW5ncyxcbiAgICAgICAgbmVlZHM6IHRoaXMucmVuZGVyRGVwZW5kZW5jaWVzKG5vZGUpLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIGlkVG9rZW46IHRoaXMuYXdzQ3JlZGVudGlhbHMuam9iUGVybWlzc2lvbigpLFxuICAgICAgICB9LFxuICAgICAgICBydW5zT246IHRoaXMucnVubmVyLnJ1bnNPbixcbiAgICAgICAgb3V0cHV0czoge1xuICAgICAgICAgIFtBU1NFVF9IQVNIX05BTUVdOiBgXFwke3sgc3RlcHMuUHVibGlzaC5vdXRwdXRzLiR7QVNTRVRfSEFTSF9OQU1FfSB9fWAsXG4gICAgICAgIH0sXG4gICAgICAgIHN0ZXBzOiBbXG4gICAgICAgICAgLi4udGhpcy5zdGVwc1RvRG93bmxvYWRBc3NlbWJseShjZGtvdXREaXIpLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICdJbnN0YWxsJyxcbiAgICAgICAgICAgIHJ1bjogYG5wbSBpbnN0YWxsIC0tbm8tc2F2ZSBjZGstYXNzZXRzJHtpbnN0YWxsU3VmZml4fWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAuLi50aGlzLnN0ZXBzVG9Db25maWd1cmVBd3ModGhpcy5wdWJsaXNoQXNzZXRzQXV0aFJlZ2lvbiksXG4gICAgICAgICAgLi4uZG9ja2VyTG9naW5TdGVwcyxcbiAgICAgICAgICBwdWJsaXNoU3RlcCxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgam9iRm9yRGVwbG95KG5vZGU6IEFHcmFwaE5vZGUsIHN0YWNrOiBTdGFja0RlcGxveW1lbnQsIF9jYXB0dXJlT3V0cHV0czogYm9vbGVhbik6IEpvYiB7XG4gICAgY29uc3QgcmVnaW9uID0gc3RhY2sucmVnaW9uO1xuICAgIGNvbnN0IGFjY291bnQgPSBzdGFjay5hY2NvdW50O1xuICAgIGlmICghcmVnaW9uIHx8ICFhY2NvdW50KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wiYWNjb3VudFwiIGFuZCBcInJlZ2lvblwiIGFyZSByZXF1aXJlZCcpO1xuICAgIH1cblxuICAgIGlmICghc3RhY2sudGVtcGxhdGVVcmwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIGRldGVybWluZSB0ZW1wbGF0ZSBVUkwgZm9yIHN0YWNrICR7c3RhY2suc3RhY2tBcnRpZmFjdElkfWApO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc29sdmUgPSAoczogc3RyaW5nKTogc3RyaW5nID0+IHtcbiAgICAgIHJldHVybiBFbnZpcm9ubWVudFBsYWNlaG9sZGVycy5yZXBsYWNlKHMsIHtcbiAgICAgICAgYWNjb3VudElkOiBhY2NvdW50LFxuICAgICAgICByZWdpb246IHJlZ2lvbixcbiAgICAgICAgcGFydGl0aW9uOiAnYXdzJyxcbiAgICAgIH0pO1xuICAgIH07XG5cbiAgICBjb25zdCByZXBsYWNlQXNzZXRIYXNoID0gKHRlbXBsYXRlOiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IGhhc2ggPSBwYXRoLnBhcnNlKHRlbXBsYXRlLnNwbGl0KCcvJykucG9wKCkgPz8gJycpLm5hbWU7XG4gICAgICBpZiAodGhpcy5hc3NldEhhc2hNYXBbaGFzaF0gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRlbXBsYXRlIGFzc2V0IGhhc2ggJHtoYXNofSBub3QgZm91bmQuYCk7XG4gICAgICB9XG4gICAgICByZXR1cm4gdGVtcGxhdGUucmVwbGFjZShoYXNoLCBgXFwke3sgbmVlZHMuJHt0aGlzLmFzc2V0SGFzaE1hcFtoYXNoXX0ub3V0cHV0cy4ke0FTU0VUX0hBU0hfTkFNRX0gfX1gKTtcbiAgICB9O1xuXG4gICAgY29uc3QgcGFyYW1zOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge1xuICAgICAgJ25hbWUnOiBzdGFjay5zdGFja05hbWUsXG4gICAgICAndGVtcGxhdGUnOiByZXBsYWNlQXNzZXRIYXNoKHJlc29sdmUoc3RhY2sudGVtcGxhdGVVcmwpKSxcbiAgICAgICduby1mYWlsLW9uLWVtcHR5LWNoYW5nZXNldCc6ICcxJyxcbiAgICB9O1xuXG4gICAgY29uc3QgY2FwYWJpbGl0aWVzID0gdGhpcy5zdGFja1Byb3BlcnRpZXNbc3RhY2suc3RhY2tBcnRpZmFjdElkXT8uY2FwYWJpbGl0aWVzO1xuICAgIGlmIChjYXBhYmlsaXRpZXMpIHtcbiAgICAgIHBhcmFtcy5jYXBhYmlsaXRpZXMgPSBBcnJheShjYXBhYmlsaXRpZXMpLmpvaW4oJywnKTtcbiAgICB9XG5cbiAgICBpZiAoc3RhY2suZXhlY3V0aW9uUm9sZUFybikge1xuICAgICAgcGFyYW1zWydyb2xlLWFybiddID0gcmVzb2x2ZShzdGFjay5leGVjdXRpb25Sb2xlQXJuKTtcbiAgICB9XG4gICAgY29uc3QgYXNzdW1lUm9sZUFybiA9IHN0YWNrLmFzc3VtZVJvbGVBcm4gPyByZXNvbHZlKHN0YWNrLmFzc3VtZVJvbGVBcm4pIDogdW5kZWZpbmVkO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiBub2RlLnVuaXF1ZUlkLFxuICAgICAgZGVmaW5pdGlvbjoge1xuICAgICAgICBuYW1lOiBgRGVwbG95ICR7c3RhY2suc3RhY2tBcnRpZmFjdElkfWAsXG4gICAgICAgIC4uLnRoaXMuam9iU2V0dGluZ3MsXG4gICAgICAgIC4uLnRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0/LnNldHRpbmdzLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIGlkVG9rZW46IHRoaXMuYXdzQ3JlZGVudGlhbHMuam9iUGVybWlzc2lvbigpLFxuICAgICAgICB9LFxuICAgICAgICAuLi4odGhpcy5zdGFja1Byb3BlcnRpZXNbc3RhY2suc3RhY2tBcnRpZmFjdElkXT8uZW52aXJvbm1lbnQgPyB7XG4gICAgICAgICAgZW52aXJvbm1lbnQ6IHRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0uZW52aXJvbm1lbnQsXG4gICAgICAgIH0gOiB7fSksXG4gICAgICAgIG5lZWRzOiB0aGlzLnJlbmRlckRlcGVuZGVuY2llcyhub2RlKSxcbiAgICAgICAgcnVuc09uOiB0aGlzLnJ1bm5lci5ydW5zT24sXG4gICAgICAgIHN0ZXBzOiBbXG4gICAgICAgICAgLi4udGhpcy5zdGVwc1RvQ29uZmlndXJlQXdzKHJlZ2lvbiwgYXNzdW1lUm9sZUFybiksXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6ICdEZXBsb3knLFxuICAgICAgICAgICAgdXNlczogJ2F3cy1hY3Rpb25zL2F3cy1jbG91ZGZvcm1hdGlvbi1naXRodWItZGVwbG95QHYxLjEuMCcsXG4gICAgICAgICAgICB3aXRoOiBwYXJhbXMsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgam9iRm9yQnVpbGRTdGVwKG5vZGU6IEFHcmFwaE5vZGUsIHN0ZXA6IFN0ZXApOiBKb2Ige1xuICAgIGlmICghKHN0ZXAgaW5zdGFuY2VvZiBTaGVsbFN0ZXApKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3N5bnRoU3RlcCBtdXN0IGJlIGEgU2NyaXB0U3RlcCcpO1xuICAgIH1cblxuICAgIGlmIChzdGVwLmlucHV0cy5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3N5bnRoU3RlcCBjYW5ub3QgaGF2ZSBpbnB1dHMnKTtcbiAgICB9XG5cbiAgICBpZiAoc3RlcC5vdXRwdXRzLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignc3ludGhTdGVwIG11c3QgaGF2ZSBhIHNpbmdsZSBvdXRwdXQnKTtcbiAgICB9XG5cbiAgICBpZiAoIXN0ZXAucHJpbWFyeU91dHB1dCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdzeW50aFN0ZXAgcmVxdWlyZXMgYSBwcmltYXJ5T3V0cHV0IHdoaWNoIGNvbnRhaW5zIGNkay5vdXQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBjZGtPdXQgPSBzdGVwLm91dHB1dHNbMF07XG5cbiAgICBjb25zdCBpbnN0YWxsU3RlcHMgPSBzdGVwLmluc3RhbGxDb21tYW5kcy5sZW5ndGggPiAwID8gW3tcbiAgICAgIG5hbWU6ICdJbnN0YWxsJyxcbiAgICAgIHJ1bjogc3RlcC5pbnN0YWxsQ29tbWFuZHMuam9pbignXFxuJyksXG4gICAgfV0gOiBbXTtcblxuICAgIHJldHVybiB7XG4gICAgICBpZDogbm9kZS51bmlxdWVJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogJ1N5bnRoZXNpemUnLFxuICAgICAgICAuLi50aGlzLmpvYlNldHRpbmdzLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIC8vIFRoZSBTeW50aGVzaXplIGpvYiBkb2VzIG5vdCB1c2UgdGhlIEdpdEh1YiBBY3Rpb24gUm9sZSBvbiBpdHMgb3duLCBidXQgaXQncyBwb3NzaWJsZVxuICAgICAgICAgIC8vIHRoYXQgaXQgaXMgYmVpbmcgdXNlZCBpbiB0aGUgcHJlQnVpbGRTdGVwcy5cbiAgICAgICAgICBpZFRva2VuOiB0aGlzLmF3c0NyZWRlbnRpYWxzLmpvYlBlcm1pc3Npb24oKSxcbiAgICAgICAgfSxcbiAgICAgICAgcnVuc09uOiB0aGlzLnJ1bm5lci5ydW5zT24sXG4gICAgICAgIG5lZWRzOiB0aGlzLnJlbmRlckRlcGVuZGVuY2llcyhub2RlKSxcbiAgICAgICAgZW52OiBzdGVwLmVudixcbiAgICAgICAgY29udGFpbmVyOiB0aGlzLmJ1aWxkQ29udGFpbmVyLFxuICAgICAgICBzdGVwczogW1xuICAgICAgICAgIC4uLnRoaXMuc3RlcHNUb0NoZWNrb3V0KCksXG4gICAgICAgICAgLi4udGhpcy5wcmVCdWlsZFN0ZXBzLFxuICAgICAgICAgIC4uLmluc3RhbGxTdGVwcyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnQnVpbGQnLFxuICAgICAgICAgICAgcnVuOiBzdGVwLmNvbW1hbmRzLmpvaW4oJ1xcbicpLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgLi4udGhpcy5wb3N0QnVpbGRTdGVwcyxcbiAgICAgICAgICAuLi50aGlzLnN0ZXBzVG9VcGxvYWRBc3NlbWJseShjZGtPdXQuZGlyZWN0b3J5KSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZWFyY2hlcyBmb3IgdGhlIHN0YWNrIHRoYXQgcHJvZHVjZWQgdGhlIG91dHB1dCB2aWEgdGhlIGN1cnJlbnRcbiAgICogam9iJ3MgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBUaGlzIGZ1bmN0aW9uIHNob3VsZCBhbHdheXMgZmluZCBhIHN0YWNrLCBzaW5jZSBpdCBpcyBndWFyYW50ZWVkXG4gICAqIHRoYXQgYSBDZm5PdXRwdXQgY29tZXMgZnJvbSBhIHJlZmVyZW5jZWQgc3RhY2suXG4gICAqL1xuICBwcml2YXRlIGZpbmRTdGFja09mT3V0cHV0KHJlZjogU3RhY2tPdXRwdXRSZWZlcmVuY2UsIG5vZGU6IEFHcmFwaE5vZGUpIHtcbiAgICBmb3IgKGNvbnN0IGRlcCBvZiBub2RlLmFsbERlcHMpIHtcbiAgICAgIGlmIChkZXAuZGF0YT8udHlwZSA9PT0gJ2V4ZWN1dGUnICYmIHJlZi5pc1Byb2R1Y2VkQnkoZGVwLmRhdGEuc3RhY2spKSB7XG4gICAgICAgIHJldHVybiBkZXAudW5pcXVlSWQ7XG4gICAgICB9XG4gICAgfVxuICAgIC8vIFNob3VsZCBuZXZlciBoYXBwZW5cbiAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBvdXRwdXQgJHtyZWYub3V0cHV0TmFtZX0gaXMgbm90IHJlZmVyZW5jZWQgYnkgYW55IG9mIHRoZSBkZXBlbmRlbnQgc3RhY2tzIWApO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRKb2JPdXRwdXQoam9iSWQ6IHN0cmluZywgb3V0cHV0OiBnaXRodWIuSm9iU3RlcE91dHB1dCkge1xuICAgIGlmICh0aGlzLmpvYk91dHB1dHNbam9iSWRdID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuam9iT3V0cHV0c1tqb2JJZF0gPSBbb3V0cHV0XTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5qb2JPdXRwdXRzW2pvYklkXS5wdXNoKG91dHB1dCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBqb2JGb3JTY3JpcHRTdGVwKG5vZGU6IEFHcmFwaE5vZGUsIHN0ZXA6IFNoZWxsU3RlcCk6IEpvYiB7XG4gICAgY29uc3QgZW52VmFyaWFibGVzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge307XG4gICAgZm9yIChjb25zdCBbZW52TmFtZSwgcmVmXSBvZiBPYmplY3QuZW50cmllcyhzdGVwLmVudkZyb21DZm5PdXRwdXRzKSkge1xuICAgICAgY29uc3Qgam9iSWQgPSB0aGlzLmZpbmRTdGFja09mT3V0cHV0KHJlZiwgbm9kZSk7XG4gICAgICB0aGlzLmFkZEpvYk91dHB1dChqb2JJZCwge1xuICAgICAgICBvdXRwdXROYW1lOiByZWYub3V0cHV0TmFtZSxcbiAgICAgICAgc3RlcElkOiAnRGVwbG95JyxcbiAgICAgIH0pO1xuICAgICAgZW52VmFyaWFibGVzW2Vudk5hbWVdID0gYFxcJHt7IG5lZWRzLiR7am9iSWR9Lm91dHB1dHMuJHtyZWYub3V0cHV0TmFtZX0gfX1gO1xuICAgIH1cblxuICAgIGNvbnN0IGRvd25sb2FkSW5wdXRzID0gbmV3IEFycmF5PGdpdGh1Yi5Kb2JTdGVwPigpO1xuICAgIGNvbnN0IHVwbG9hZE91dHB1dHMgPSBuZXcgQXJyYXk8Z2l0aHViLkpvYlN0ZXA+KCk7XG5cbiAgICBmb3IgKGNvbnN0IGlucHV0IG9mIHN0ZXAuaW5wdXRzKSB7XG4gICAgICBkb3dubG9hZElucHV0cy5wdXNoKHtcbiAgICAgICAgdXNlczogJ2FjdGlvbnMvZG93bmxvYWQtYXJ0aWZhY3RAdjMnLFxuICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgbmFtZTogaW5wdXQuZmlsZVNldC5pZCxcbiAgICAgICAgICBwYXRoOiBpbnB1dC5kaXJlY3RvcnksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IG91dHB1dCBvZiBzdGVwLm91dHB1dHMpIHtcbiAgICAgIHVwbG9hZE91dHB1dHMucHVzaCh7XG4gICAgICAgIHVzZXM6ICdhY3Rpb25zL3VwbG9hZC1hcnRpZmFjdEB2MycsXG4gICAgICAgIHdpdGg6IHtcbiAgICAgICAgICBuYW1lOiBvdXRwdXQuZmlsZVNldC5pZCxcbiAgICAgICAgICBwYXRoOiBvdXRwdXQuZGlyZWN0b3J5LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgaW5zdGFsbFN0ZXBzID0gc3RlcC5pbnN0YWxsQ29tbWFuZHMubGVuZ3RoID4gMCA/IFt7XG4gICAgICBuYW1lOiAnSW5zdGFsbCcsXG4gICAgICBydW46IHN0ZXAuaW5zdGFsbENvbW1hbmRzLmpvaW4oJ1xcbicpLFxuICAgIH1dIDogW107XG5cbiAgICByZXR1cm4ge1xuICAgICAgaWQ6IG5vZGUudW5pcXVlSWQsXG4gICAgICBkZWZpbml0aW9uOiB7XG4gICAgICAgIG5hbWU6IHN0ZXAuaWQsXG4gICAgICAgIC4uLnRoaXMuam9iU2V0dGluZ3MsXG4gICAgICAgIHBlcm1pc3Npb25zOiB7XG4gICAgICAgICAgY29udGVudHM6IGdpdGh1Yi5Kb2JQZXJtaXNzaW9uLlJFQUQsXG4gICAgICAgIH0sXG4gICAgICAgIHJ1bnNPbjogdGhpcy5ydW5uZXIucnVuc09uLFxuICAgICAgICBuZWVkczogdGhpcy5yZW5kZXJEZXBlbmRlbmNpZXMobm9kZSksXG4gICAgICAgIGVudjoge1xuICAgICAgICAgIC4uLnN0ZXAuZW52LFxuICAgICAgICAgIC4uLmVudlZhcmlhYmxlcyxcbiAgICAgICAgfSxcbiAgICAgICAgc3RlcHM6IFtcbiAgICAgICAgICAuLi5kb3dubG9hZElucHV0cyxcbiAgICAgICAgICAuLi5pbnN0YWxsU3RlcHMsXG4gICAgICAgICAgeyBydW46IHN0ZXAuY29tbWFuZHMuam9pbignXFxuJykgfSxcbiAgICAgICAgICAuLi51cGxvYWRPdXRwdXRzLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBqb2JGb3JHaXRIdWJBY3Rpb25TdGVwKG5vZGU6IEFHcmFwaE5vZGUsIHN0ZXA6IEdpdEh1YkFjdGlvblN0ZXApOiBKb2Ige1xuICAgIHJldHVybiB7XG4gICAgICBpZDogbm9kZS51bmlxdWVJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogc3RlcC5pZCxcbiAgICAgICAgLi4udGhpcy5qb2JTZXR0aW5ncyxcbiAgICAgICAgcGVybWlzc2lvbnM6IHtcbiAgICAgICAgICBjb250ZW50czogZ2l0aHViLkpvYlBlcm1pc3Npb24uV1JJVEUsXG4gICAgICAgIH0sXG4gICAgICAgIHJ1bnNPbjogdGhpcy5ydW5uZXIucnVuc09uLFxuICAgICAgICBuZWVkczogdGhpcy5yZW5kZXJEZXBlbmRlbmNpZXMobm9kZSksXG4gICAgICAgIGVudjogc3RlcC5lbnYsXG4gICAgICAgIHN0ZXBzOiBzdGVwLmpvYlN0ZXBzLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBzdGVwc1RvQ29uZmlndXJlQXdzKHJlZ2lvbjogc3RyaW5nLCBhc3N1bWVSb2xlQXJuPzogc3RyaW5nKTogZ2l0aHViLkpvYlN0ZXBbXSB7XG4gICAgcmV0dXJuIHRoaXMuYXdzQ3JlZGVudGlhbHMuY3JlZGVudGlhbFN0ZXBzKHJlZ2lvbiwgYXNzdW1lUm9sZUFybik7XG4gIH1cblxuICBwcml2YXRlIHN0ZXBzVG9Db25maWd1cmVEb2NrZXIoZG9ja2VyQ3JlZGVudGlhbDogRG9ja2VyQ3JlZGVudGlhbCk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIGxldCBwYXJhbXM6IFJlY29yZDxzdHJpbmcsIGFueT47XG5cbiAgICBpZiAoZG9ja2VyQ3JlZGVudGlhbC5uYW1lID09PSAnZG9ja2VyJykge1xuICAgICAgcGFyYW1zID0ge1xuICAgICAgICB1c2VybmFtZTogYFxcJHt7IHNlY3JldHMuJHtkb2NrZXJDcmVkZW50aWFsLnVzZXJuYW1lS2V5fSB9fWAsXG4gICAgICAgIHBhc3N3b3JkOiBgXFwke3sgc2VjcmV0cy4ke2RvY2tlckNyZWRlbnRpYWwucGFzc3dvcmRLZXl9IH19YCxcbiAgICAgIH07XG4gICAgfSBlbHNlIGlmIChkb2NrZXJDcmVkZW50aWFsLm5hbWUgPT09ICdlY3InKSB7XG4gICAgICBwYXJhbXMgPSB7XG4gICAgICAgIHJlZ2lzdHJ5OiBkb2NrZXJDcmVkZW50aWFsLnJlZ2lzdHJ5LFxuICAgICAgfTtcbiAgICB9IGVsc2Uge1xuICAgICAgcGFyYW1zID0ge1xuICAgICAgICByZWdpc3RyeTogZG9ja2VyQ3JlZGVudGlhbC5yZWdpc3RyeSxcbiAgICAgICAgdXNlcm5hbWU6IGBcXCR7eyBzZWNyZXRzLiR7ZG9ja2VyQ3JlZGVudGlhbC51c2VybmFtZUtleX0gfX1gLFxuICAgICAgICBwYXNzd29yZDogYFxcJHt7IHNlY3JldHMuJHtkb2NrZXJDcmVkZW50aWFsLnBhc3N3b3JkS2V5fSB9fWAsXG4gICAgICB9O1xuICAgIH1cblxuICAgIHJldHVybiBbXG4gICAgICB7XG4gICAgICAgIHVzZXM6ICdkb2NrZXIvbG9naW4tYWN0aW9uQHYyJyxcbiAgICAgICAgd2l0aDogcGFyYW1zLFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGVwc1RvRG93bmxvYWRBc3NlbWJseSh0YXJnZXREaXI6IHN0cmluZyk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIGlmICh0aGlzLnByZVN5bnRoZWQpIHtcbiAgICAgIHJldHVybiB0aGlzLnN0ZXBzVG9DaGVja291dCgpO1xuICAgIH1cblxuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogYERvd25sb2FkICR7Q0RLT1VUX0FSVElGQUNUfWAsXG4gICAgICB1c2VzOiAnYWN0aW9ucy9kb3dubG9hZC1hcnRpZmFjdEB2MycsXG4gICAgICB3aXRoOiB7XG4gICAgICAgIG5hbWU6IENES09VVF9BUlRJRkFDVCxcbiAgICAgICAgcGF0aDogdGFyZ2V0RGlyLFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxuXG4gIHByaXZhdGUgc3RlcHNUb0NoZWNrb3V0KCk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogJ0NoZWNrb3V0JyxcbiAgICAgIHVzZXM6ICdhY3Rpb25zL2NoZWNrb3V0QHYzJyxcbiAgICB9XTtcbiAgfVxuXG4gIHByaXZhdGUgc3RlcHNUb1VwbG9hZEFzc2VtYmx5KGRpcjogc3RyaW5nKTogZ2l0aHViLkpvYlN0ZXBbXSB7XG4gICAgaWYgKHRoaXMucHJlU3ludGhlZCkge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cblxuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogYFVwbG9hZCAke0NES09VVF9BUlRJRkFDVH1gLFxuICAgICAgdXNlczogJ2FjdGlvbnMvdXBsb2FkLWFydGlmYWN0QHYzJyxcbiAgICAgIHdpdGg6IHtcbiAgICAgICAgbmFtZTogQ0RLT1VUX0FSVElGQUNULFxuICAgICAgICBwYXRoOiBkaXIsXG4gICAgICB9LFxuICAgIH1dO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJEZXBlbmRlbmNpZXMobm9kZTogQUdyYXBoTm9kZSkge1xuICAgIGNvbnN0IGRlcHMgPSBuZXcgQXJyYXk8QUdyYXBoTm9kZT4oKTtcblxuICAgIGZvciAoY29uc3QgZCBvZiBub2RlLmFsbERlcHMpIHtcbiAgICAgIGlmIChkIGluc3RhbmNlb2YgR3JhcGgpIHtcbiAgICAgICAgZGVwcy5wdXNoKC4uLmQuYWxsTGVhdmVzKCkubm9kZXMpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgZGVwcy5wdXNoKGQpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBkZXBzLm1hcCh4ID0+IHgudW5pcXVlSWQpO1xuICB9XG59XG5cbmludGVyZmFjZSBDb250ZXh0IHtcbiAgLyoqXG4gICAqIFRoZSBwaXBlbGluZSBncmFwaC5cbiAgICovXG4gIHJlYWRvbmx5IHN0cnVjdHVyZTogUGlwZWxpbmVHcmFwaDtcblxuICAvKipcbiAgICogTmFtZSBvZiBjbG91ZCBhc3NlbWJseSBkaXJlY3RvcnkuXG4gICAqL1xuICByZWFkb25seSBhc3NlbWJseURpcjogc3RyaW5nO1xufVxuXG5pbnRlcmZhY2UgSm9iIHtcbiAgcmVhZG9ubHkgaWQ6IHN0cmluZztcbiAgcmVhZG9ubHkgZGVmaW5pdGlvbjogZ2l0aHViLkpvYjtcbn1cblxuZnVuY3Rpb24gc25ha2VDYXNlS2V5czxUID0gdW5rbm93bj4ob2JqOiBULCBzZXAgPSAnLScpOiBUIHtcbiAgaWYgKHR5cGVvZiBvYmogIT09ICdvYmplY3QnIHx8IG9iaiA9PSBudWxsKSB7XG4gICAgcmV0dXJuIG9iajtcbiAgfVxuXG4gIGlmIChBcnJheS5pc0FycmF5KG9iaikpIHtcbiAgICByZXR1cm4gb2JqLm1hcChvID0+IHNuYWtlQ2FzZUtleXMobywgc2VwKSkgYXMgYW55O1xuICB9XG5cbiAgY29uc3QgcmVzdWx0OiBSZWNvcmQ8c3RyaW5nLCB1bmtub3duPiA9IHt9O1xuICBmb3IgKGxldCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMob2JqKSkge1xuICAgIC8vIHdlIGRvbid0IHdhbnQgdG8gc25ha2UgY2FzZSBlbnZpcm9ubWVudCB2YXJpYWJsZXNcbiAgICBpZiAoayAhPT0gJ2VudicgJiYgdHlwZW9mIHYgPT09ICdvYmplY3QnICYmIHYgIT0gbnVsbCkge1xuICAgICAgdiA9IHNuYWtlQ2FzZUtleXModik7XG4gICAgfVxuICAgIHJlc3VsdFtkZWNhbWVsaXplKGssIHsgc2VwYXJhdG9yOiBzZXAgfSldID0gdjtcbiAgfVxuICByZXR1cm4gcmVzdWx0IGFzIGFueTtcbn1cblxuLyoqXG4gKiBOYW1lcyBvZiBzZWNyZXRzIGZvciBBV1MgY3JlZGVudGlhbHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXdzQ3JlZGVudGlhbHNTZWNyZXRzIHtcbiAgLyoqXG4gICAqIEBkZWZhdWx0IFwiQVdTX0FDQ0VTU19LRVlfSURcIlxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzS2V5SWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBkZWZhdWx0IFwiQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZXCJcbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldEFjY2Vzc0tleT86IHN0cmluZztcblxuICAvKipcbiAgICogQGRlZmF1bHQgLSBubyBzZXNzaW9uIHRva2VuIGlzIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHNlc3Npb25Ub2tlbj86IHN0cmluZztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uKiBmbGF0dGVuPEE+KHhzOiBJdGVyYWJsZTxBW10+KTogSXRlcmFibGVJdGVyYXRvcjxBPiB7XG4gIGZvciAoY29uc3QgeCBvZiB4cykge1xuICAgIGZvciAoY29uc3QgeSBvZiB4KSB7XG4gICAgICB5aWVsZCB5O1xuICAgIH1cbiAgfVxufVxuIl19