"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.sslMethod !== 'undefined') {
            cfConfig.aliasConfiguration.sslMethod = config.sslMethod;
        }
        if (typeof config.securityPolicy !== 'undefined') {
            cfConfig.aliasConfiguration.securityPolicy = config.securityPolicy;
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website.
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     It will also setup error forwarding and unauth forwarding back to indexDoc.
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.distributionDomainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     configured by using the details in the hosted zone provided.
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: [`www.${config.zoneName}`],
            targetDomain: config.zoneName,
        });
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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