import json
from flask import Request, Response, request, jsonify
from logging import Logger
from typing import Any

from .iam_common import (
    IamServer, _iam_lock,
    _get_logger, _get_iam_server
)
from .iam_pomes import (
    user_login, user_logout,
    user_token, token_exchange, login_callback
)


# @flask_app.route(rule=<login_endpoint>,  # JUSBR_ENDPOINT_LOGIN
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,  # KEYCLOAK_ENDPOINT_LOGIN
#                  methods=["GET"])
def service_login() -> Response:
    """
    Entry point for the IAM server's login service.

    Return the URL for invoking the IAM server's authentication page, with the appropriate parameters.

    :return: *Response* with the URL for invoking the IAM server's authentication page, or *BAD REQUEST* if error
    """
    # declare the return variable
    result: Response | None = None

    # retrieve the operations's logger
    logger: Logger = _get_logger()
    if logger:
        # log the request
        logger.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=logger)
        if iam_server:
            # obtain the login URL
            login_data: dict[str,  str] = user_login(iam_server=iam_server,
                                                     args=request.args,
                                                     errors=errors,
                                                     logger=logger)
            if login_data:
                result = jsonify(login_data)

    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<logout_endpoint>,  # JUSBR_ENDPOINT_LOGOUT
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,   # KEYCLOAK_ENDPOINT_LOGOUT
#                  methods=["GET"])
def service_logout() -> Response:
    """
    Entry point for the JusBR logout service.

    Remove all data associating the user from the *IAM* server's registry.

    :return: *Response NO CONTENT*, or *BAD REQUEST* if error
    """
    # declare the return variable
    result: Response | None

    # retrieve the operations's logger
    logger: Logger = _get_logger()
    if logger:
        # log the request
        logger.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=logger)
        if iam_server:
            # logout the user
            user_logout(iam_server=iam_server,
                        args=request.args,
                        errors=errors,
                        logger=logger)
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = Response(status=204)

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<callback_endpoint>,  # JUSBR_ENDPOINT_CALLBACK
#                  methods=["GET", "POST"])
# @flask_app.route(rule=<callback_endpoint>,  # KEYCLOAK_ENDPOINT_CALLBACK
#                  methods=["POST"])
def service_callback() -> Response:
    """
    Entry point for the callback from JusBR on authentication operation.

    This callback is invoked from a front-end application after a successful login at the
    *IAM* server's login page, forwarding the data received. In a typical OAuth2 flow faction,
    this data is then used to effectively obtain the token from the *IAM* server.

    On success, the returned *Response* will contain the following JSON:
        {
            "user-id": <reference-user-identification>,
            "token": <token>
        }

    :return: *Response* containing the reference user identification and the token, or *BAD REQUEST*
    """
    # retrieve the operations's logger
    logger: Logger = _get_logger()
    if logger:
        # log the request
        logger.debug(msg=_log_init(request=request))

    errors: list[str] = []
    token_data: tuple[str, str] | None = None
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=logger)
        if iam_server:
            # process the callback operation
            token_data = login_callback(iam_server=iam_server,
                                        args=request.args,
                                        errors=errors,
                                        logger=logger)
    result: Response
    if errors:
        result = jsonify({"errors": "; ".join(errors)})
        result.status_code = 400
        if logger:
            logger.error(msg=json.dumps(obj=result))
    else:
        result = jsonify({"user-id": token_data[0],
                          "token": token_data[1]})
    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<token_endpoint>,  # JUSBR_ENDPOINT_TOKEN
#                  methods=["GET"])
# @flask_app.route(rule=<token_endpoint>,  # KEYCLOAK_ENDPOINT_TOKEN
#                  methods=["GET"])
def service_token() -> Response:
    """
    Entry point for retrieving a token from the *IAM* server.

    On success, the returned *Response* will contain the following JSON:
        {
            "user-id": <reference-user-identification>,
            "token": <token>
        }

    :return: *Response* containing the user reference identification and the token, or *BAD REQUEST*
    """
    # retrieve the operations's logger
    logger: Logger = _get_logger()
    if logger:
        # log the request
        logger.debug(msg=_log_init(request=request))

    # obtain the user's identification
    args: dict[str, Any] = request.args
    user_id: str = args.get("user-id") or args.get("user_id") or args.get("login")

    errors: list[str] = []
    token: str | None = None
    if user_id:
        with _iam_lock:
            # retrieve the IAM server
            iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                    errors=errors,
                                                    logger=logger)
            if iam_server:
                # retrieve the token
                errors: list[str] = []
                token: str = user_token(iam_server=iam_server,
                                        args=args,
                                        errors=errors,
                                        logger=logger)
    else:
        msg: str = "User identification not provided"
        errors.append(msg)
        if logger:
            logger.error(msg=msg)

    result: Response
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = jsonify({"user-id": user_id,
                          "token": token})
    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<callback_endpoint>,  # KEYCLOAK_ENDPOINT_EXCHANGE
#                  methods=["POST"])
def service_exchange() -> Response:
    """
    Entry point for requesting the *IAM* server to exchange the token.

    This is currently limited to the *KEYCLOAK* server. The token itself is stored in *KEYCLOAK*'s registry.
    The expected parameters in the request are:
        - client-id: identification for the reference user (aliases: 'client_id', 'login')
        - token: the token to be exchanged

    If the exchange is successful, the token data is stored in the *IAM* server's registry, and returned.
    Otherwise, *errors* will contain the appropriate error message.

    On success, the typical *Response* returned will contain the following attributes:
        {
            "token_type": "Bearer",
            "access_token": <str>,
            "expires_in": <number-of-seconds>,
            "refresh_token": <str>,
            "refesh_expires_in": <number-of-seconds>
        }

    :return: *Response* containing the token data, or *BAD REQUEST*
    """
    # retrieve the operations's logger
    logger: Logger = _get_logger()
    if logger:
        # log the request
        logger.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server (currently, only 'IAM_KEYCLOAK' is supported)
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=logger)
        # exchange the token
        token_data: dict[str, Any] | None = None
        if iam_server:
            errors: list[str] = []
            token_data = token_exchange(iam_server=iam_server,
                                        args=request.args,
                                        errors=errors,
                                        logger=logger)
    result: Response
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = jsonify(token_data)

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


def _log_init(request: Request) -> str:
    """
    Build the messages for logging the request entry.

    :param request: the Request object
    :return: the log message
    """

    params: str = json.dumps(obj=request.args,
                             ensure_ascii=False)
    return f"Request {request.method}:{request.path}, params {params}"
