"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path_1 = require("path");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const render_data_1 = require("./render-data");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *      Source.bucket(bucket, key)
 *      Source.asset('/local/path/to/directory')
 *      Source.asset('/local/path/to/a/file.zip')
 *      Source.data('hello/world/file.txt', 'Hello, world!')
 *      Source.data('config.json', { baz: topic.topicArn })
 *
 * @stability stable
 */
class Source {
    constructor() { }
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     *
     * Make sure you trust the producer of the archive.
     *
     * @param bucket The S3 Bucket.
     * @param zipObjectKey The S3 object key of the zip file with contents.
     * @stability stable
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     *
     * If the local asset is a .zip archive, make sure you trust the
     * producer of the archive.
     *
     * @param path The path to a local .zip file or a directory.
     * @stability stable
     */
    static asset(path, options) {
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified string contents into the bucket.
     *
     * The
     * content can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * To store a JSON object use `Source.jsonData()`.
     *
     * @param objectKey The destination S3 object key (relative to the root of the S3 deployment).
     * @param data The data to be stored in the object.
     * @stability stable
     */
    static data(objectKey, data) {
        return {
            bind: (scope, context) => {
                const workdir = core_1.FileSystem.mkdtemp('s3-deployment');
                const outputPath = path_1.join(workdir, objectKey);
                const rendered = render_data_1.renderData(scope, data);
                fs.mkdirSync(path_1.dirname(outputPath), { recursive: true });
                fs.writeFileSync(outputPath, rendered.text);
                const asset = this.asset(workdir).bind(scope, context);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.zipObjectKey,
                    markers: rendered.markers,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified JSON object into the bucket.
     *
     * The
     * object can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * @param objectKey The destination S3 object key (relative to the root of the S3 deployment).
     * @param obj A JSON object.
     * @stability stable
     */
    static jsonData(objectKey, obj) {
        return {
            bind: (scope, context) => {
                return Source.data(objectKey, core_1.Stack.of(scope).toJsonString(obj)).bind(scope, context);
            },
        };
    }
}
exports.Source = Source;
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "@aws-cdk/aws-s3-deployment.Source", version: "1.144.0" };
//# sourceMappingURL=data:application/json;base64,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