"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const asg = require("aws-cdk-lib/aws-autoscaling");
const ec2 = require("aws-cdk-lib/aws-ec2");
const gitlab_runner_autoscaling_1 = require("./gitlab-runner-autoscaling");
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, 'TestStackAutoscaling', { env });
const vpc = ec2.Vpc.fromLookup(stack, 'DefaultVpc', {
    isDefault: true,
});
const defaultCapacity = parseInt(stack.node.tryGetContext('CAPACITY'));
const runner = new gitlab_runner_autoscaling_1.GitlabRunnerAutoscaling(stack, 'TestRunnerAutoscaling', {
    gitlabToken: stack.node.tryGetContext('GITLAB_TOKEN'),
    ebsConfig: {
        volumeSize: 10,
    },
    vpc: vpc,
    dockerVolumes: [{
            hostPath: '/tmp/cache',
            containerPath: '/tmp/cache',
        }],
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
    spotInstance: true,
    alarms: [
        {
            AlarmName: 'GitlabRunnerDiskUsage',
            MetricName: 'disk_used_percent',
            Threshold: 50,
        },
    ],
    gitlabRunnerVersion: '15.10',
});
/**
 * Scheduled scaling
 * https://docs.aws.amazon.com/cdk/api/latest/docs/aws-autoscaling-readme.html#scheduled-scaling
 */
runner.autoscalingGroup.scaleOnSchedule('StopOnWeekends', {
    schedule: asg.Schedule.cron({ weekDay: 'fri', hour: '18', minute: '0' }),
    minCapacity: 0,
    maxCapacity: 0,
});
runner.autoscalingGroup.scaleOnSchedule('WorkOnWeekdays', {
    schedule: asg.Schedule.cron({ weekDay: 'mon', hour: '9', minute: '0' }),
    minCapacity: defaultCapacity,
    maxCapacity: defaultCapacity,
});
new aws_cdk_lib_1.CfnOutput(stack, 'role', { value: runner.instanceRole.roleArn });
//# sourceMappingURL=data:application/json;base64,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