"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const semver = require("semver");
const my_package_json_1 = require("./private/my-package-json");
const schema_helpers_1 = require("./private/schema-helpers");
const PACKAGE_VERSION = my_package_json_1.loadMyPackageJson().version;
/**
 * Static class with loader routines
 *
 * This class mostly exists to put the schema structs into input position
 * (taken into a function), so that the jsii-diff checker will make sure all
 * structs are only allowed to be weakened in future updates. For example,
 * it is now allowed to add new required fields, since old CDK frameworks
 * would not be emitting those fields yet.
 *
 * At the same time, we might as well validate the structure so code doesn't
 * barf on invalid disk input.
 */
class AssetManifestSchema {
    /**
     * Validate the given structured object as a valid ManifestFile schema
     */
    static validate(file) {
        const obj = file;
        if (typeof obj !== 'object' || obj === null) {
            throw new Error(`Expected object, got '${obj}`);
        }
        schema_helpers_1.expectKey(obj, 'version', schema_helpers_1.isString);
        // Current tool must be >= the version used to write the manifest
        // (disregarding MVs) which we can verify by ^-prefixing the file version.
        if (!semver.satisfies(AssetManifestSchema.currentVersion(), `^${obj.version}`)) {
            throw new Error(`Need CDK Tools >= '${obj.version}' to read this file (current is '${AssetManifestSchema.currentVersion()}')`);
        }
        schema_helpers_1.expectKey(obj, 'files', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isFileAsset)), true);
        schema_helpers_1.expectKey(obj, 'dockerImages', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isDockerImageAsset)), true);
    }
    /**
     * Take a ManifestFile as input
     *
     * The presence of this method makes sure the struct is only ever weakened
     * in future releases.
     */
    static input(file) {
        this.validate(file);
    }
    /**
     * Return the version of the schema module
     */
    static currentVersion() {
        return PACKAGE_VERSION;
    }
}
exports.AssetManifestSchema = AssetManifestSchema;
function isFileAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'path', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'packaging', schema_helpers_1.isFileAssetPackaging, true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'bucketName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'objectKey', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
function isDockerImageAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'directory', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'dockerFile', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildTarget', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildArgs', schema_helpers_1.isMapOf(schema_helpers_1.isString), true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'repositoryName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageTag', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
//# sourceMappingURL=data:application/json;base64,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