"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const file_asset_1 = require("../file-asset");
/**
 * Validate that a given key is of a given type in an object
 *
 * If not optional, the key is considered required.
 *
 * Uses predicate validators that return a 'witness', so we can use the return type of
 * the validator function to infer the actual type of the value and enrich the type information
 * of the given input field.
 *
 * In effect, validators should be written like this:
 *
 *   if (!valid(input)) { throw; }
 *   return input;
 */
function expectKey(obj, key, validate, optional) {
    if (typeof obj !== 'object' || obj === null || (!(key in obj) && !optional)) {
        throw new Error(`Expected key '${key}' missing: ${JSON.stringify(obj)}`);
    }
    if (key in obj) {
        try {
            validate(obj[key]);
        }
        catch (e) {
            throw new Error(`${key}: ${e.message}`);
        }
    }
}
exports.expectKey = expectKey;
function isString(x) {
    if (typeof x !== 'string') {
        throw new Error(`Expected a string, got '${x}'`);
    }
    return x;
}
exports.isString = isString;
function isMapOf(pred) {
    return x => {
        assertIsObject(x);
        Object.values(x).forEach(pred);
        return x;
    };
}
exports.isMapOf = isMapOf;
function isObjectAnd(p) {
    return x => {
        assertIsObject(x);
        return p(x);
    };
}
exports.isObjectAnd = isObjectAnd;
function assertIsObject(x) {
    if (typeof x !== 'object' || x === null) {
        throw new Error(`Expected a map, got '${x}'`);
    }
}
exports.assertIsObject = assertIsObject;
function isFileAssetPackaging(x) {
    const str = isString(x);
    const validValues = Object.values(file_asset_1.FileAssetPackaging); // Explicit cast needed because this is a string-valued enum
    if (!validValues.includes(str)) {
        throw new Error(`Expected a FileAssetPackaging (one of ${validValues.map(v => `'${v}'`).join(', ')}), got '${str}'`);
    }
    return x;
}
exports.isFileAssetPackaging = isFileAssetPackaging;
//# sourceMappingURL=data:application/json;base64,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