"""Level-based log filtering."""

import logging

from ..core.backend import LogRecord


class LevelFilter:
    """Filter logs based on level."""

    def __init__(
        self,
        min_level: int = logging.INFO,
        max_level: int = logging.CRITICAL,
        allowed_levels: set[int] | None = None
    ) -> None:
        """Initialize level filter.
        
        Args:
            min_level: Minimum log level to allow
            max_level: Maximum log level to allow  
            allowed_levels: Specific set of allowed levels (overrides min/max)
        """
        self.min_level = min_level
        self.max_level = max_level
        self.allowed_levels = allowed_levels

    def should_log(self, record: LogRecord) -> bool:
        """Check if record should be logged based on level.
        
        Args:
            record: The log record to check
            
        Returns:
            True if record should be logged, False otherwise
        """
        if self.allowed_levels is not None:
            return record.level in self.allowed_levels

        return self.min_level <= record.level <= self.max_level
