"""Custom exceptions for unified_logger."""

from typing import Any


class UnifiedLoggerError(Exception):
    """Base exception for all unified_logger errors."""

    def __init__(self, message: str, details: dict[str, Any] | None = None) -> None:
        """Initialize the exception.
        
        Args:
            message: The error message
            details: Optional additional error details
        """
        super().__init__(message)
        self.message = message
        self.details = details or {}


class BackendError(UnifiedLoggerError):
    """Exception raised when backend operations fail."""

    def __init__(
        self,
        message: str,
        backend_name: str,
        details: dict[str, Any] | None = None
    ) -> None:
        """Initialize the backend error.
        
        Args:
            message: The error message
            backend_name: Name of the backend that failed
            details: Optional additional error details
        """
        super().__init__(message, details)
        self.backend_name = backend_name


class ConfigurationError(UnifiedLoggerError):
    """Exception raised when configuration is invalid."""

    def __init__(
        self,
        message: str,
        config_path: str | None = None,
        details: dict[str, Any] | None = None
    ) -> None:
        """Initialize the configuration error.
        
        Args:
            message: The error message
            config_path: Path to the configuration file that failed
            details: Optional additional error details
        """
        super().__init__(message, details)
        self.config_path = config_path


class ValidationError(UnifiedLoggerError):
    """Exception raised when data validation fails."""

    def __init__(
        self,
        message: str,
        field_name: str | None = None,
        details: dict[str, Any] | None = None
    ) -> None:
        """Initialize the validation error.
        
        Args:
            message: The error message
            field_name: Name of the field that failed validation
            details: Optional additional error details
        """
        super().__init__(message, details)
        self.field_name = field_name
