"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * The log driver to use for the container.
 */
var LogDriver;
(function (LogDriver) {
    /**
     * Specifies the Amazon CloudWatch Logs logging driver.
     */
    LogDriver["AWSLOGS"] = "awslogs";
    /**
     * Specifies the Fluentd logging driver.
     */
    LogDriver["FLUENTD"] = "fluentd";
    /**
     * Specifies the Graylog Extended Format (GELF) logging driver.
     */
    LogDriver["GELF"] = "gelf";
    /**
     * Specifies the journald logging driver.
     */
    LogDriver["JOURNALD"] = "journald";
    /**
     * Specifies the logentries logging driver.
     */
    LogDriver["LOGENTRIES"] = "logentries";
    /**
     * Specifies the JSON file logging driver.
     */
    LogDriver["JSON_FILE"] = "json-file";
    /**
     * Specifies the Splunk logging driver.
     */
    LogDriver["SPLUNK"] = "splunk";
    /**
     * Specifies the syslog logging driver.
     */
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * Platform capabilities
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    /**
     * Specifies EC2 environment.
     */
    PlatformCapabilities["EC2"] = "EC2";
    /**
     * Specifies Fargate environment.
     */
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends aws_cdk_lib_1.Resource {
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefName = stack.splitArn(jobDefinitionArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: jobDefinitionName,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_JobDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, JobDefinition);
            }
            throw error;
        }
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!props.platformCapabilities?.includes(PlatformCapabilities.FARGATE);
        const jobDef = new aws_batch_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: props.platformCapabilities ?? [PlatformCapabilities.EC2],
            propagateTags: props.propagateTags,
        });
        // add read secrets permission to execution role
        if (props.container.secrets && props.container.executionRole) {
            const executionRole = props.container.executionRole;
            Object.values(props.container.secrets).forEach((secret) => {
                secret.grantRead(executionRole);
            });
        }
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return vars;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        // If the AWS_*** environment variables are not explicitly set to the container, infer them from the current environment.
        // This makes the usage of tools like AWS SDK inside the container frictionless
        const environment = this.deserializeEnvVariables(container.environment);
        if (!environment.find((x) => x.name === 'AWS_REGION')) {
            environment.push({
                name: 'AWS_REGION',
                value: aws_cdk_lib_1.Stack.of(this).region,
            });
        }
        if (!environment.find((x) => x.name === 'AWS_ACCOUNT')) {
            environment.push({
                name: 'AWS_ACCOUNT',
                value: aws_cdk_lib_1.Stack.of(this).account,
            });
        }
        return {
            command: container.command,
            environment,
            secrets: container.secrets
                ? Object.entries(container.secrets).map(([key, value]) => {
                    return {
                        name: key,
                        valueFrom: value.arn,
                    };
                })
                : undefined,
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch-alpha.JobDefinition", version: "2.72.0-alpha.0" };
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiam9iLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJqb2ItZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFHQSw2Q0FBOEU7QUFFOUUscURBQXlEO0FBRXpELCtFQUF5RTtBQUV6RTs7R0FFRztBQUNILElBQVksU0F3Q1g7QUF4Q0QsV0FBWSxTQUFTO0lBQ25COztPQUVHO0lBQ0gsZ0NBQW1CLENBQUE7SUFFbkI7O09BRUc7SUFDSCxnQ0FBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILDBCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILGtDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gsc0NBQXlCLENBQUE7SUFFekI7O09BRUc7SUFDSCxvQ0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILDhCQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsOEJBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQXhDVyxTQUFTLEdBQVQsaUJBQVMsS0FBVCxpQkFBUyxRQXdDcEI7QUFFRDs7R0FFRztBQUNILElBQVksb0JBVVg7QUFWRCxXQUFZLG9CQUFvQjtJQUM5Qjs7T0FFRztJQUNILG1DQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILDJDQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFWVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQVUvQjtBQWlVRDs7OztHQUlHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsc0JBQVE7SUFDekM7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGdCQUF3QjtRQUN2RixNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLGdCQUFnQixFQUFFLHVCQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxZQUFhLENBQUM7UUFFakcsTUFBTSxNQUFPLFNBQVEsc0JBQVE7WUFBN0I7O2dCQUNrQixxQkFBZ0IsR0FBRyxnQkFBZ0IsQ0FBQztnQkFDcEMsc0JBQWlCLEdBQUcsVUFBVSxDQUFDO1lBQ2pELENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxpQkFBeUI7UUFDekYsTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNoQyxPQUFPLEVBQUUsT0FBTztZQUNoQixRQUFRLEVBQUUsZ0JBQWdCO1lBQzFCLFNBQVMsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQjtZQUN4QyxZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0IscUJBQWdCLEdBQUcsU0FBUyxDQUFDO2dCQUM3QixzQkFBaUIsR0FBRyxpQkFBaUIsQ0FBQztZQUN4RCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQU1ELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtTQUN0QyxDQUFDLENBQUM7Ozs7OzsrQ0FwRE0sYUFBYTs7OztRQXNEdEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksc0RBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV2RSxNQUFNLFNBQVMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLG9CQUFvQixFQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV2RixNQUFNLE1BQU0sR0FBRyxJQUFJLDRCQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDdkUsSUFBSSxFQUFFLFdBQVc7WUFDakIsY0FBYyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUM3QixDQUFDLENBQUM7b0JBQ0EsUUFBUSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUTtvQkFDbEMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO29CQUN6RSxRQUFRLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLO2lCQUNoQztnQkFDRCxDQUFDLENBQUMsU0FBUztZQUNiLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixhQUFhLEVBQUU7Z0JBQ2IsUUFBUSxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksQ0FBQzthQUNuQztZQUNELE9BQU8sRUFBRTtnQkFDUCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQzlFO1lBQ0Qsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixJQUFJLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDO1lBQzlFLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtTQUNuQyxDQUFDLENBQUM7UUFFSCxnREFBZ0Q7UUFDaEQsSUFBSyxLQUFLLENBQUMsU0FBUyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsU0FBUyxDQUFDLGFBQWEsRUFBRztZQUM5RCxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQztZQUNwRCxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ3hELE1BQU0sQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDbEMsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUMvRCxPQUFPLEVBQUUsT0FBTztZQUNoQixRQUFRLEVBQUUsZ0JBQWdCO1lBQzFCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUNwRTtJQUVPLHVCQUF1QixDQUFDLEdBQWdDO1FBQzlELE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxFQUF3QyxDQUFDO1FBRS9ELElBQUksR0FBRyxLQUFLLFNBQVMsRUFBRTtZQUNyQixPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFZLEVBQUUsRUFBRTtZQUNwQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3hDLENBQUMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUVEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLEtBQXlCO1FBQzdDLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUN2QixPQUFPO1NBQ1I7UUFFRCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUM7ZUFDNUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM1RDtRQUVELElBQUksS0FBSyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQztlQUM1RyxLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7WUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1NBQzNEO1FBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztlQUMxRyxLQUFLLENBQUMsU0FBUyxDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDakQsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1NBQzdEO0tBQ0Y7SUFFTyxpQkFBaUIsQ0FBQyxTQUFpQyxFQUFFLFNBQWtCO1FBQzdFLElBQUksU0FBUyxLQUFLLFNBQVMsRUFBRTtZQUMzQixPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELHlIQUF5SDtRQUN6SCwrRUFBK0U7UUFFL0UsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUV4RSxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxZQUFZLENBQUMsRUFBRTtZQUNyRCxXQUFXLENBQUMsSUFBSSxDQUFDO2dCQUNmLElBQUksRUFBRSxZQUFZO2dCQUNsQixLQUFLLEVBQUUsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTTthQUM3QixDQUFDLENBQUM7U0FDSjtRQUNELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLGFBQWEsQ0FBQyxFQUFFO1lBQ3RELFdBQVcsQ0FBQyxJQUFJLENBQUM7Z0JBQ2YsSUFBSSxFQUFFLGFBQWE7Z0JBQ25CLEtBQUssRUFBRSxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPO2FBQzlCLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTztZQUNMLE9BQU8sRUFBRSxTQUFTLENBQUMsT0FBTztZQUMxQixXQUFXO1lBQ1gsT0FBTyxFQUFFLFNBQVMsQ0FBQyxPQUFPO2dCQUN4QixDQUFDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRTtvQkFDdkQsT0FBTzt3QkFDTCxJQUFJLEVBQUUsR0FBRzt3QkFDVCxTQUFTLEVBQUUsS0FBSyxDQUFDLEdBQUc7cUJBQ3JCLENBQUM7Z0JBQ0osQ0FBQyxDQUFDO2dCQUNGLENBQUMsQ0FBQyxTQUFTO1lBQ2IsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUztZQUNqQyxZQUFZLEVBQUUsU0FBUyxDQUFDLFlBQVksSUFBSSxTQUFTLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtZQUN6RSxVQUFVLEVBQUUsU0FBUyxDQUFDLE9BQU8sSUFBSSxTQUFTLENBQUMsT0FBTyxDQUFDLE9BQU87WUFDMUQsZ0JBQWdCLEVBQUUsU0FBUyxDQUFDLGFBQWEsSUFBSSxTQUFTLENBQUMsYUFBYSxDQUFDLE9BQU87WUFDNUUsZUFBZSxFQUFFLFNBQVMsQ0FBQyxXQUFXO2dCQUNwQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsU0FBUyxDQUFDLFdBQVcsQ0FBQyxxQkFBcUIsRUFBRSxDQUFDLE9BQU8sRUFBRTtnQkFDcEUsQ0FBQyxDQUFDLFNBQVM7WUFDYixnQkFBZ0IsRUFBRSxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO2dCQUM3QyxTQUFTLEVBQUUsU0FBUyxDQUFDLGdCQUFnQixDQUFDLFNBQVM7Z0JBQy9DLE9BQU8sRUFBRSxTQUFTLENBQUMsZ0JBQWdCLENBQUMsT0FBTztnQkFDM0MsYUFBYSxFQUFFLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhO29CQUNyRCxDQUFDLENBQUMsSUFBSSxDQUFDLGtDQUFrQyxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUM7b0JBQ25GLENBQUMsQ0FBQyxTQUFTO2FBQ2QsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNiLFdBQVcsRUFBRSxTQUFTLENBQUMsV0FBVztZQUNsQyxVQUFVLEVBQUUsU0FBUyxDQUFDLFVBQVUsSUFBSSxLQUFLO1lBQ3pDLG9CQUFvQixFQUFFLFNBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO2dCQUMvQyxjQUFjLEVBQUUsU0FBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVO2FBQ2xFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixzQkFBc0IsRUFBRSxTQUFTLENBQUMsUUFBUSxJQUFJLEtBQUs7WUFDbkQsT0FBTyxFQUFFLFNBQVMsQ0FBQyxPQUFPO1lBQzFCLElBQUksRUFBRSxTQUFTLENBQUMsSUFBSTtZQUNwQixPQUFPLEVBQUUsU0FBUyxDQUFDLE9BQU87WUFDMUIsNEJBQTRCLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUM7Z0JBQ3hELGVBQWUsRUFBRSxTQUFTLENBQUMsZUFBZTthQUMzQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2Isb0JBQW9CLEVBQUU7Z0JBQ3BCLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDMUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDLGNBQWMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO2FBQ3JGLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1NBQ3pGLENBQUM7S0FDSDtJQUVPLG1CQUFtQixDQUFDLGNBQStCLEVBQUUsU0FBa0I7UUFDN0UsTUFBTSxVQUFVLEdBQUcsSUFBSSxLQUFLLEVBQThDLENBQUM7UUFFM0UsS0FBSyxNQUFNLElBQUksSUFBSSxjQUFjLENBQUMsVUFBVSxFQUFFO1lBQzVDLFVBQVUsQ0FBQyxJQUFJLENBQUM7Z0JBQ2QsU0FBUyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQztnQkFDNUQsV0FBVyxFQUFFLEdBQUcsSUFBSSxDQUFDLGFBQWEsSUFBSSxDQUFDLElBQUksSUFBSSxDQUFDLFdBQVcsSUFBSSxjQUFjLENBQUMsS0FBSyxFQUFFO2FBQ3RGLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxVQUFVLENBQUM7S0FDbkI7SUFFTyxrQ0FBa0MsQ0FBQyxhQUE4QjtRQUN2RSxPQUFPLGFBQWEsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDdEMsT0FBTztnQkFDTCxJQUFJLEVBQUUsWUFBWSxDQUFDLFVBQVU7Z0JBQzdCLFNBQVMsRUFBRSxZQUFZLENBQUMsU0FBUzthQUNsQyxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7S0FDSjs7OztBQS9OVSxzQ0FBYSIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGVjcyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCwgRHVyYXRpb24sIElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5Kb2JEZWZpbml0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWJhdGNoJztcbmltcG9ydCB7IEV4cG9zZWRTZWNyZXQgfSBmcm9tICcuL2V4cG9zZWQtc2VjcmV0JztcbmltcG9ydCB7IEpvYkRlZmluaXRpb25JbWFnZUNvbmZpZyB9IGZyb20gJy4vam9iLWRlZmluaXRpb24taW1hZ2UtY29uZmlnJztcblxuLyoqXG4gKiBUaGUgbG9nIGRyaXZlciB0byB1c2UgZm9yIHRoZSBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBlbnVtIExvZ0RyaXZlciB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEFtYXpvbiBDbG91ZFdhdGNoIExvZ3MgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBBV1NMT0dTID0gJ2F3c2xvZ3MnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEZsdWVudGQgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBGTFVFTlREID0gJ2ZsdWVudGQnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEdyYXlsb2cgRXh0ZW5kZWQgRm9ybWF0IChHRUxGKSBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIEdFTEYgPSAnZ2VsZicsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgam91cm5hbGQgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBKT1VSTkFMRCA9ICdqb3VybmFsZCcsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgbG9nZW50cmllcyBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIExPR0VOVFJJRVMgPSAnbG9nZW50cmllcycsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgSlNPTiBmaWxlIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgSlNPTl9GSUxFID0gJ2pzb24tZmlsZScsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgU3BsdW5rIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgU1BMVU5LID0gJ3NwbHVuaycsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgc3lzbG9nIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgU1lTTE9HID0gJ3N5c2xvZydcbn1cblxuLyoqXG4gKiBQbGF0Zm9ybSBjYXBhYmlsaXRpZXNcbiAqL1xuZXhwb3J0IGVudW0gUGxhdGZvcm1DYXBhYmlsaXRpZXMge1xuICAvKipcbiAgICogU3BlY2lmaWVzIEVDMiBlbnZpcm9ubWVudC5cbiAgICovXG4gIEVDMiA9ICdFQzInLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgRmFyZ2F0ZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIEZBUkdBVEUgPSAnRkFSR0FURSdcbn1cblxuLyoqXG4gKiBMb2cgY29uZmlndXJhdGlvbiBvcHRpb25zIHRvIHNlbmQgdG8gYSBjdXN0b20gbG9nIGRyaXZlciBmb3IgdGhlIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBsb2cgZHJpdmVyIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ0RyaXZlcjogTG9nRHJpdmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgY29uZmlndXJhdGlvbiBvcHRpb25zIHRvIHNlbmQgdG8gdGhlIGxvZyBkcml2ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY29uZmlndXJhdGlvbiBvcHRpb25zIGFyZSBzZW50XG4gICAqL1xuICByZWFkb25seSBvcHRpb25zPzogYW55O1xuXG4gIC8qKlxuICAgKiBUaGUgc2VjcmV0cyB0byBwYXNzIHRvIHRoZSBsb2cgY29uZmlndXJhdGlvbiBhcyBvcHRpb25zLlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL3NwZWNpZnlpbmctc2Vuc2l0aXZlLWRhdGEtc2VjcmV0cy5odG1sI3NlY3JldHMtbG9nY29uZmlnXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc2VjcmV0cyBhcmUgcGFzc2VkXG4gICAqL1xuICByZWFkb25seSBzZWNyZXRPcHRpb25zPzogRXhwb3NlZFNlY3JldFtdO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgb2YgYSBqb2IgZGVmaW5pdGlvbiBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iRGVmaW5pdGlvbkNvbnRhaW5lciB7XG4gIC8qKlxuICAgKiBUaGUgY29tbWFuZCB0aGF0IGlzIHBhc3NlZCB0byB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBJZiB5b3UgcHJvdmlkZSBhIHNoZWxsIGNvbW1hbmQgYXMgYSBzaW5nbGUgc3RyaW5nLCB5b3UgaGF2ZSB0byBxdW90ZSBjb21tYW5kLWxpbmUgYXJndW1lbnRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENNRCB2YWx1ZSBidWlsdCBpbnRvIGNvbnRhaW5lciBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbW1hbmQ/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB0byBwYXNzIHRvIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyBmcm9tIHNlY3JldHMgbWFuYWdlciBvciBzc20gcGFyYW1ldGVyIHN0b3JlXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldHM/OiB7IFtrZXk6IHN0cmluZ106IGVjcy5TZWNyZXQgfTtcblxuICAvKipcbiAgICogVGhlIGltYWdlIHVzZWQgdG8gc3RhcnQgYSBjb250YWluZXIuXG4gICAqL1xuICByZWFkb25seSBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlO1xuXG4gIC8qKlxuICAgKiBUaGUgaW5zdGFuY2UgdHlwZSB0byB1c2UgZm9yIGEgbXVsdGktbm9kZSBwYXJhbGxlbCBqb2IuIEN1cnJlbnRseSBhbGwgbm9kZSBncm91cHMgaW4gYVxuICAgKiBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYiBtdXN0IHVzZSB0aGUgc2FtZSBpbnN0YW5jZSB0eXBlLiBUaGlzIHBhcmFtZXRlciBpcyBub3QgdmFsaWRcbiAgICogZm9yIHNpbmdsZS1ub2RlIGNvbnRhaW5lciBqb2JzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB0aGF0IHRoZSBjb250YWluZXIgY2FuIGFzc3VtZSBmb3IgQVdTIHBlcm1pc3Npb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFuIElBTSByb2xlIHdpbGwgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGpvYlJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIExpbnV4LXNwZWNpZmljIG1vZGlmaWNhdGlvbnMgdGhhdCBhcmUgYXBwbGllZCB0byB0aGUgY29udGFpbmVyLCBzdWNoIGFzIGRldGFpbHMgZm9yIGRldmljZSBtYXBwaW5ncy5cbiAgICogRm9yIG5vdywgb25seSB0aGUgYGRldmljZXNgIHByb3BlcnR5IGlzIHN1cHBvcnRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGxpbnV4UGFyYW1zPzogZWNzLkxpbnV4UGFyYW1ldGVycztcblxuICAvKipcbiAgICogVGhlIGxvZyBjb25maWd1cmF0aW9uIHNwZWNpZmljYXRpb24gZm9yIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gY29udGFpbmVycyB1c2UgdGhlIHNhbWUgbG9nZ2luZyBkcml2ZXIgdGhhdCB0aGUgRG9ja2VyIGRhZW1vbiB1c2VzXG4gICAqL1xuICByZWFkb25seSBsb2dDb25maWd1cmF0aW9uPzogTG9nQ29uZmlndXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGhhcmQgbGltaXQgKGluIE1pQikgb2YgbWVtb3J5IHRvIHByZXNlbnQgdG8gdGhlIGNvbnRhaW5lci4gSWYgeW91ciBjb250YWluZXIgYXR0ZW1wdHMgdG8gZXhjZWVkXG4gICAqIHRoZSBtZW1vcnkgc3BlY2lmaWVkIGhlcmUsIHRoZSBjb250YWluZXIgaXMga2lsbGVkLiBZb3UgbXVzdCBzcGVjaWZ5IGF0IGxlYXN0IDQgTWlCIG9mIG1lbW9yeSBmb3IgRUMyIGFuZCA1MTIgTWlCIGZvciBGYXJnYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDQgZm9yIEVDMiwgNTEyIGZvciBGYXJnYXRlXG4gICAqL1xuICByZWFkb25seSBtZW1vcnlMaW1pdE1pQj86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1vdW50IHBvaW50cyBmb3IgZGF0YSB2b2x1bWVzIGluIHlvdXIgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIG1vdW50IHBvaW50cyB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBtb3VudFBvaW50cz86IGVjcy5Nb3VudFBvaW50W107XG5cbiAgLyoqXG4gICAqIFdoZW4gdGhpcyBwYXJhbWV0ZXIgaXMgdHJ1ZSwgdGhlIGNvbnRhaW5lciBpcyBnaXZlbiBlbGV2YXRlZCBwcml2aWxlZ2VzIG9uIHRoZSBob3N0IGNvbnRhaW5lciBpbnN0YW5jZSAoc2ltaWxhciB0byB0aGUgcm9vdCB1c2VyKS5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByaXZpbGVnZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGVuIHRoaXMgcGFyYW1ldGVyIGlzIHRydWUsIHRoZSBjb250YWluZXIgaXMgZ2l2ZW4gcmVhZC1vbmx5IGFjY2VzcyB0byBpdHMgcm9vdCBmaWxlIHN5c3RlbS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlYWRPbmx5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBwaHlzaWNhbCBHUFVzIHRvIHJlc2VydmUgZm9yIHRoZSBjb250YWluZXIuIFRoZSBudW1iZXIgb2YgR1BVcyByZXNlcnZlZCBmb3IgYWxsXG4gICAqIGNvbnRhaW5lcnMgaW4gYSBqb2Igc2hvdWxkIG5vdCBleGNlZWQgdGhlIG51bWJlciBvZiBhdmFpbGFibGUgR1BVcyBvbiB0aGUgY29tcHV0ZSByZXNvdXJjZSB0aGF0IHRoZSBqb2IgaXMgbGF1bmNoZWQgb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gR1BVIHJlc2VydmF0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgZ3B1Q291bnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiB1bGltaXRzIHRvIHNldCBpbiB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGxpbWl0cy5cbiAgICovXG4gIHJlYWRvbmx5IHVsaW1pdHM/OiBlY3MuVWxpbWl0W107XG5cbiAgLyoqXG4gICAqIFRoZSB1c2VyIG5hbWUgdG8gdXNlIGluc2lkZSB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgdXNlcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB2Q1BVcyByZXNlcnZlZCBmb3IgdGhlIGNvbnRhaW5lci4gRWFjaCB2Q1BVIGlzIGVxdWl2YWxlbnQgdG9cbiAgICogMSwwMjQgQ1BVIHNoYXJlcy4gWW91IG11c3Qgc3BlY2lmeSBhdCBsZWFzdCBvbmUgdkNQVSBmb3IgRUMyIGFuZCAwLjI1IGZvciBGYXJnYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEgZm9yIEVDMiwgMC4yNSBmb3IgRmFyZ2F0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdmNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBkYXRhIHZvbHVtZXMgdXNlZCBpbiBhIGpvYi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBkYXRhIHZvbHVtZXMgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lcz86IGVjcy5Wb2x1bWVbXTtcblxuICAvKipcbiAgICogRmFyZ2F0ZSBwbGF0Zm9ybSB2ZXJzaW9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTEFURVNUIHBsYXRmb3JtIHZlcnNpb24gd2lsbCBiZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBwbGF0Zm9ybVZlcnNpb24/OiBlY3MuRmFyZ2F0ZVBsYXRmb3JtVmVyc2lvblxuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdGhhdCBBV1MgQmF0Y2ggY2FuIGFzc3VtZS5cbiAgICogUmVxdWlyZWQgd2hlbiB1c2luZyBGYXJnYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4ZWN1dGlvblJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRvIGFzc2lnbiBhIHB1YmxpYyBJUCB0byB0aGUgam9iXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGFzc2lnblB1YmxpY0lwPzogYm9vbGVhblxufVxuXG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSBgSm9iRGVmaW5pdGlvbmAgY29uc3RydWN0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYkRlZmluaXRpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgam9iIGRlZmluaXRpb24uXG4gICAqXG4gICAqIFVwIHRvIDEyOCBsZXR0ZXJzICh1cHBlcmNhc2UgYW5kIGxvd2VyY2FzZSksIG51bWJlcnMsIGh5cGhlbnMsIGFuZCB1bmRlcnNjb3JlcyBhcmUgYWxsb3dlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgQ2xvdWRmb3JtYXRpb24tZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbiBvYmplY3Qgd2l0aCB2YXJpb3VzIHByb3BlcnRpZXMgc3BlY2lmaWMgdG8gY29udGFpbmVyLWJhc2VkIGpvYnMuXG4gICAqL1xuICByZWFkb25seSBjb250YWluZXI6IEpvYkRlZmluaXRpb25Db250YWluZXI7XG5cbiAgLyoqXG4gICAqIEFuIG9iamVjdCB3aXRoIHZhcmlvdXMgcHJvcGVydGllcyBzcGVjaWZpYyB0byBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBub2RlUHJvcHM/OiBJTXVsdGlOb2RlUHJvcHM7XG5cbiAgLyoqXG4gICAqIFdoZW4geW91IHN1Ym1pdCBhIGpvYiwgeW91IGNhbiBzcGVjaWZ5IHBhcmFtZXRlcnMgdGhhdCBzaG91bGQgcmVwbGFjZSB0aGVcbiAgICogcGxhY2Vob2xkZXJzIG9yIG92ZXJyaWRlIHRoZSBkZWZhdWx0IGpvYiBkZWZpbml0aW9uIHBhcmFtZXRlcnMuIFBhcmFtZXRlcnNcbiAgICogaW4gam9iIHN1Ym1pc3Npb24gcmVxdWVzdHMgdGFrZSBwcmVjZWRlbmNlIG92ZXIgdGhlIGRlZmF1bHRzIGluIGEgam9iIGRlZmluaXRpb24uXG4gICAqIFRoaXMgYWxsb3dzIHlvdSB0byB1c2UgdGhlIHNhbWUgam9iIGRlZmluaXRpb24gZm9yIG11bHRpcGxlIGpvYnMgdGhhdCB1c2UgdGhlIHNhbWVcbiAgICogZm9ybWF0LCBhbmQgcHJvZ3JhbW1hdGljYWxseSBjaGFuZ2UgdmFsdWVzIGluIHRoZSBjb21tYW5kIGF0IHN1Ym1pc3Npb24gdGltZS5cbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvam9iX2RlZmluaXRpb25fcGFyYW1ldGVycy5odG1sXG4gICAqIEBkZWZhdWx0IC0gdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBwYXJhbWV0ZXJzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB0aW1lcyB0byBtb3ZlIGEgam9iIHRvIHRoZSBSVU5OQUJMRSBzdGF0dXMuIFlvdSBtYXkgc3BlY2lmeSBiZXR3ZWVuIDEgYW5kXG4gICAqIDEwIGF0dGVtcHRzLiBJZiB0aGUgdmFsdWUgb2YgYXR0ZW1wdHMgaXMgZ3JlYXRlciB0aGFuIG9uZSwgdGhlIGpvYiBpcyByZXRyaWVkIG9uIGZhaWx1cmVcbiAgICogdGhlIHNhbWUgbnVtYmVyIG9mIGF0dGVtcHRzIGFzIHRoZSB2YWx1ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0cnlBdHRlbXB0cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHRpbWVvdXQgY29uZmlndXJhdGlvbiBmb3Igam9icyB0aGF0IGFyZSBzdWJtaXR0ZWQgd2l0aCB0aGlzIGpvYiBkZWZpbml0aW9uLiBZb3UgY2FuIHNwZWNpZnlcbiAgICogYSB0aW1lb3V0IGR1cmF0aW9uIGFmdGVyIHdoaWNoIEFXUyBCYXRjaCB0ZXJtaW5hdGVzIHlvdXIgam9icyBpZiB0aGV5IGhhdmUgbm90IGZpbmlzaGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgdGltZW91dD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgcGxhdGZvcm0gY2FwYWJpbGl0aWVzIHJlcXVpcmVkIGJ5IHRoZSBqb2IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBFQzJcbiAgICovXG4gIHJlYWRvbmx5IHBsYXRmb3JtQ2FwYWJpbGl0aWVzPzogUGxhdGZvcm1DYXBhYmlsaXRpZXNbXTtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdG8gcHJvcGFnYXRlIHRoZSB0YWdzIGZyb20gdGhlIGpvYiBvciBqb2IgZGVmaW5pdGlvbiB0byB0aGUgY29ycmVzcG9uZGluZyBBbWF6b24gRUNTIHRhc2suXG4gICAqIElmIG5vIHZhbHVlIGlzIHNwZWNpZmllZCwgdGhlIHRhZ3MgYXJlbid0IHByb3BhZ2F0ZWQuXG4gICAqIFRhZ3MgY2FuIG9ubHkgYmUgcHJvcGFnYXRlZCB0byB0aGUgdGFza3MgZHVyaW5nIHRhc2sgY3JlYXRpb24uIEZvciB0YWdzIHdpdGggdGhlIHNhbWUgbmFtZSxcbiAgICogam9iIHRhZ3MgYXJlIGdpdmVuIHByaW9yaXR5IG92ZXIgam9iIGRlZmluaXRpb25zIHRhZ3MuXG4gICAqIElmIHRoZSB0b3RhbCBudW1iZXIgb2YgY29tYmluZWQgdGFncyBmcm9tIHRoZSBqb2IgYW5kIGpvYiBkZWZpbml0aW9uIGlzIG92ZXIgNTAsIHRoZSBqb2IgaXMgbW92ZWQgdG8gdGhlIGBGQUlMRURgIHN0YXRlLlxuICAgKlxuICAgKiBAbGluayBodHRwOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1iYXRjaC1qb2JkZWZpbml0aW9uLmh0bWwjY2ZuLWJhdGNoLWpvYmRlZmluaXRpb24tcHJvcGFnYXRldGFnc1xuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvcGFnYXRlVGFncz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3Igc3BlY2lmeWluZyBtdWx0aS1ub2RlIHByb3BlcnRpZXMgZm9yIGNvbXB1dGUgcmVzb3VyY2VzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElNdWx0aU5vZGVQcm9wcyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIG5vZGUgaW5kZXggZm9yIHRoZSBtYWluIG5vZGUgb2YgYSBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYi4gVGhpcyBub2RlIGluZGV4IHZhbHVlIG11c3QgYmUgZmV3ZXIgdGhhbiB0aGUgbnVtYmVyIG9mIG5vZGVzLlxuICAgKi9cbiAgbWFpbk5vZGU6IG51bWJlcjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG5vZGUgcmFuZ2VzIGFuZCB0aGVpciBwcm9wZXJ0aWVzIGFzc29jaWF0ZWQgd2l0aCBhIG11bHRpLW5vZGUgcGFyYWxsZWwgam9iLlxuICAgKi9cbiAgcmFuZ2VQcm9wczogSU5vZGVSYW5nZVByb3BzW107XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygbm9kZXMgYXNzb2NpYXRlZCB3aXRoIGEgbXVsdGktbm9kZSBwYXJhbGxlbCBqb2IuXG4gICAqL1xuICBjb3VudDogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbXVsdGktbm9kZSBiYXRjaCBqb2IuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU5vZGVSYW5nZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBjb250YWluZXIgZGV0YWlscyBmb3IgdGhlIG5vZGUgcmFuZ2UuXG4gICAqL1xuICBjb250YWluZXI6IEpvYkRlZmluaXRpb25Db250YWluZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG5vZGUgaW5kZXggdmFsdWUgdG8gYXBwbHkgdGhpcyBjb250YWluZXIgZGVmaW5pdGlvbiBhZ2FpbnN0LlxuICAgKlxuICAgKiBZb3UgbWF5IG5lc3Qgbm9kZSByYW5nZXMsIGZvciBleGFtcGxlIDA6MTAgYW5kIDQ6NSwgaW4gd2hpY2ggY2FzZSB0aGUgNDo1IHJhbmdlIHByb3BlcnRpZXMgb3ZlcnJpZGUgdGhlIDA6MTAgcHJvcGVydGllcy5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgZnJvbU5vZGVJbmRleD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbm9kZSBpbmRleCB2YWx1ZSB0byBhcHBseSB0aGlzIGNvbnRhaW5lciBkZWZpbml0aW9uIGFnYWluc3QuIElmIG9taXR0ZWQsIHRoZSBoaWdoZXN0IHZhbHVlIGlzIHVzZWQgcmVsYXRpdmUuXG4gICAqXG4gICAqIHRvIHRoZSBudW1iZXIgb2Ygbm9kZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBqb2IuIFlvdSBtYXkgbmVzdCBub2RlIHJhbmdlcywgZm9yIGV4YW1wbGUgMDoxMCBhbmQgNDo1LFxuICAgKiBpbiB3aGljaCBjYXNlIHRoZSA0OjUgcmFuZ2UgcHJvcGVydGllcyBvdmVycmlkZSB0aGUgMDoxMCBwcm9wZXJ0aWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBgSU11bHRpTm9kZXByb3BzLmNvdW50YFxuICAgKi9cbiAgdG9Ob2RlSW5kZXg/OiBudW1iZXI7XG59XG5cbi8qKlxuICogQW4gaW50ZXJmYWNlIHJlcHJlc2VudGluZyBhIGpvYiBkZWZpbml0aW9uIC0gZWl0aGVyIGEgbmV3IG9uZSwgY3JlYXRlZCB3aXRoIHRoZSBDREssICp1c2luZyB0aGVcbiAqIGBKb2JEZWZpbml0aW9uYCBjbGFzcywgb3IgZXhpc3Rpbmcgb25lcywgcmVmZXJlbmNlZCB1c2luZyB0aGUgYEpvYkRlZmluaXRpb24uZnJvbUpvYkRlZmluaXRpb25Bcm5gIG1ldGhvZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJSm9iRGVmaW5pdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgYmF0Y2ggam9iIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGJhdGNoIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhdGNoIEpvYiBEZWZpbml0aW9uLlxuICpcbiAqIERlZmluZXMgYSBiYXRjaCBqb2IgZGVmaW5pdGlvbiB0byBleGVjdXRlIGEgc3BlY2lmaWMgYmF0Y2ggam9iLlxuICovXG5leHBvcnQgY2xhc3MgSm9iRGVmaW5pdGlvbiBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAvKipcbiAgICogSW1wb3J0cyBhbiBleGlzdGluZyBiYXRjaCBqb2IgZGVmaW5pdGlvbiBieSBpdHMgYW1hem9uIHJlc291cmNlIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZVxuICAgKiBAcGFyYW0gaWRcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25Bcm5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUpvYkRlZmluaXRpb25Bcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgam9iRGVmaW5pdGlvbkFybjogc3RyaW5nKTogSUpvYkRlZmluaXRpb24ge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGpvYkRlZk5hbWUgPSBzdGFjay5zcGxpdEFybihqb2JEZWZpbml0aW9uQXJuLCBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSkucmVzb3VyY2VOYW1lITtcblxuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm4gPSBqb2JEZWZpbml0aW9uQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lID0gam9iRGVmTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydHMgYW4gZXhpc3RpbmcgYmF0Y2ggam9iIGRlZmluaXRpb24gYnkgaXRzIG5hbWUuXG4gICAqIElmIG5hbWUgaXMgc3BlY2lmaWVkIHdpdGhvdXQgYSByZXZpc2lvbiB0aGVuIHRoZSBsYXRlc3QgYWN0aXZlIHJldmlzaW9uIGlzIHVzZWQuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZVxuICAgKiBAcGFyYW0gaWRcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25OYW1lXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Kb2JEZWZpbml0aW9uTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nKTogSUpvYkRlZmluaXRpb24ge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGpvYkRlZkFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnYmF0Y2gnLFxuICAgICAgcmVzb3VyY2U6ICdqb2ItZGVmaW5pdGlvbicsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FLFxuICAgICAgcmVzb3VyY2VOYW1lOiBqb2JEZWZpbml0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm4gPSBqb2JEZWZBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgam9iRGVmaW5pdGlvbk5hbWUgPSBqb2JEZWZpbml0aW9uTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgaW1hZ2VDb25maWc6IEpvYkRlZmluaXRpb25JbWFnZUNvbmZpZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSm9iRGVmaW5pdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmpvYkRlZmluaXRpb25OYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy52YWxpZGF0ZVByb3BzKHByb3BzKTtcblxuICAgIHRoaXMuaW1hZ2VDb25maWcgPSBuZXcgSm9iRGVmaW5pdGlvbkltYWdlQ29uZmlnKHRoaXMsIHByb3BzLmNvbnRhaW5lcik7XG5cbiAgICBjb25zdCBpc0ZhcmdhdGUgPSAhIXByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzPy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKTtcblxuICAgIGNvbnN0IGpvYkRlZiA9IG5ldyBDZm5Kb2JEZWZpbml0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGpvYkRlZmluaXRpb25OYW1lOiBwcm9wcy5qb2JEZWZpbml0aW9uTmFtZSxcbiAgICAgIGNvbnRhaW5lclByb3BlcnRpZXM6IHRoaXMuYnVpbGRKb2JDb250YWluZXIocHJvcHMuY29udGFpbmVyLCBpc0ZhcmdhdGUpLFxuICAgICAgdHlwZTogJ2NvbnRhaW5lcicsXG4gICAgICBub2RlUHJvcGVydGllczogcHJvcHMubm9kZVByb3BzXG4gICAgICAgID8ge1xuICAgICAgICAgIG1haW5Ob2RlOiBwcm9wcy5ub2RlUHJvcHMubWFpbk5vZGUsXG4gICAgICAgICAgbm9kZVJhbmdlUHJvcGVydGllczogdGhpcy5idWlsZE5vZGVSYW5nZVByb3BzKHByb3BzLm5vZGVQcm9wcywgaXNGYXJnYXRlKSxcbiAgICAgICAgICBudW1Ob2RlczogcHJvcHMubm9kZVByb3BzLmNvdW50LFxuICAgICAgICB9XG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgcGFyYW1ldGVyczogcHJvcHMucGFyYW1ldGVycyxcbiAgICAgIHJldHJ5U3RyYXRlZ3k6IHtcbiAgICAgICAgYXR0ZW1wdHM6IHByb3BzLnJldHJ5QXR0ZW1wdHMgfHwgMSxcbiAgICAgIH0sXG4gICAgICB0aW1lb3V0OiB7XG4gICAgICAgIGF0dGVtcHREdXJhdGlvblNlY29uZHM6IHByb3BzLnRpbWVvdXQgPyBwcm9wcy50aW1lb3V0LnRvU2Vjb25kcygpIDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICAgIHBsYXRmb3JtQ2FwYWJpbGl0aWVzOiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcyA/PyBbUGxhdGZvcm1DYXBhYmlsaXRpZXMuRUMyXSxcbiAgICAgIHByb3BhZ2F0ZVRhZ3M6IHByb3BzLnByb3BhZ2F0ZVRhZ3MsXG4gICAgfSk7XG5cbiAgICAvLyBhZGQgcmVhZCBzZWNyZXRzIHBlcm1pc3Npb24gdG8gZXhlY3V0aW9uIHJvbGVcbiAgICBpZiAoIHByb3BzLmNvbnRhaW5lci5zZWNyZXRzICYmIHByb3BzLmNvbnRhaW5lci5leGVjdXRpb25Sb2xlICkge1xuICAgICAgY29uc3QgZXhlY3V0aW9uUm9sZSA9IHByb3BzLmNvbnRhaW5lci5leGVjdXRpb25Sb2xlO1xuICAgICAgT2JqZWN0LnZhbHVlcyhwcm9wcy5jb250YWluZXIuc2VjcmV0cykuZm9yRWFjaCgoc2VjcmV0KSA9PiB7XG4gICAgICAgIHNlY3JldC5ncmFudFJlYWQoZXhlY3V0aW9uUm9sZSk7XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLmpvYkRlZmluaXRpb25Bcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGpvYkRlZi5yZWYsIHtcbiAgICAgIHNlcnZpY2U6ICdiYXRjaCcsXG4gICAgICByZXNvdXJjZTogJ2pvYi1kZWZpbml0aW9uJyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gICAgdGhpcy5qb2JEZWZpbml0aW9uTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKGpvYkRlZi5yZWYpO1xuICB9XG5cbiAgcHJpdmF0ZSBkZXNlcmlhbGl6ZUVudlZhcmlhYmxlcyhlbnY/OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSk6IENmbkpvYkRlZmluaXRpb24uRW52aXJvbm1lbnRQcm9wZXJ0eVtdIHtcbiAgICBjb25zdCB2YXJzID0gbmV3IEFycmF5PENmbkpvYkRlZmluaXRpb24uRW52aXJvbm1lbnRQcm9wZXJ0eT4oKTtcblxuICAgIGlmIChlbnYgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHZhcnM7XG4gICAgfVxuXG4gICAgT2JqZWN0LmtleXMoZW52KS5tYXAoKG5hbWU6IHN0cmluZykgPT4ge1xuICAgICAgdmFycy5wdXNoKHsgbmFtZSwgdmFsdWU6IGVudltuYW1lXSB9KTtcbiAgICB9KTtcblxuICAgIHJldHVybiB2YXJzO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgcHJvcGVydGllcyBwcm92aWRlZCBmb3IgYSBuZXcgam9iIGRlZmluaXRpb24uXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlUHJvcHMocHJvcHM6IEpvYkRlZmluaXRpb25Qcm9wcykge1xuICAgIGlmIChwcm9wcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzICE9PSB1bmRlZmluZWQgJiYgcHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMuaW5jbHVkZXMoUGxhdGZvcm1DYXBhYmlsaXRpZXMuRkFSR0FURSlcbiAgICAgICYmIHByb3BzLmNvbnRhaW5lci5leGVjdXRpb25Sb2xlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRmFyZ2F0ZSBqb2IgbXVzdCBoYXZlIGV4ZWN1dGlvblJvbGUgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzICE9PSB1bmRlZmluZWQgJiYgcHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMuaW5jbHVkZXMoUGxhdGZvcm1DYXBhYmlsaXRpZXMuRkFSR0FURSlcbiAgICAgICYmIHByb3BzLmNvbnRhaW5lci5ncHVDb3VudCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZhcmdhdGUgam9iIG11c3Qgbm90IGhhdmUgZ3B1Q291bnQgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKChwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcyA9PT0gdW5kZWZpbmVkIHx8IHByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzLmluY2x1ZGVzKFBsYXRmb3JtQ2FwYWJpbGl0aWVzLkVDMikpXG4gICAgICAmJiBwcm9wcy5jb250YWluZXIuYXNzaWduUHVibGljSXAgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdFQzIgam9iIG11c3Qgbm90IGhhdmUgYXNzaWduUHVibGljSXAgc2V0Jyk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBidWlsZEpvYkNvbnRhaW5lcihjb250YWluZXI6IEpvYkRlZmluaXRpb25Db250YWluZXIsIGlzRmFyZ2F0ZTogYm9vbGVhbik6IENmbkpvYkRlZmluaXRpb24uQ29udGFpbmVyUHJvcGVydGllc1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoY29udGFpbmVyID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgLy8gSWYgdGhlIEFXU18qKiogZW52aXJvbm1lbnQgdmFyaWFibGVzIGFyZSBub3QgZXhwbGljaXRseSBzZXQgdG8gdGhlIGNvbnRhaW5lciwgaW5mZXIgdGhlbSBmcm9tIHRoZSBjdXJyZW50IGVudmlyb25tZW50LlxuICAgIC8vIFRoaXMgbWFrZXMgdGhlIHVzYWdlIG9mIHRvb2xzIGxpa2UgQVdTIFNESyBpbnNpZGUgdGhlIGNvbnRhaW5lciBmcmljdGlvbmxlc3NcblxuICAgIGNvbnN0IGVudmlyb25tZW50ID0gdGhpcy5kZXNlcmlhbGl6ZUVudlZhcmlhYmxlcyhjb250YWluZXIuZW52aXJvbm1lbnQpO1xuXG4gICAgaWYgKCFlbnZpcm9ubWVudC5maW5kKCh4KSA9PiB4Lm5hbWUgPT09ICdBV1NfUkVHSU9OJykpIHtcbiAgICAgIGVudmlyb25tZW50LnB1c2goe1xuICAgICAgICBuYW1lOiAnQVdTX1JFR0lPTicsXG4gICAgICAgIHZhbHVlOiBTdGFjay5vZih0aGlzKS5yZWdpb24sXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKCFlbnZpcm9ubWVudC5maW5kKCh4KSA9PiB4Lm5hbWUgPT09ICdBV1NfQUNDT1VOVCcpKSB7XG4gICAgICBlbnZpcm9ubWVudC5wdXNoKHtcbiAgICAgICAgbmFtZTogJ0FXU19BQ0NPVU5UJyxcbiAgICAgICAgdmFsdWU6IFN0YWNrLm9mKHRoaXMpLmFjY291bnQsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29tbWFuZDogY29udGFpbmVyLmNvbW1hbmQsXG4gICAgICBlbnZpcm9ubWVudCxcbiAgICAgIHNlY3JldHM6IGNvbnRhaW5lci5zZWNyZXRzXG4gICAgICAgID8gT2JqZWN0LmVudHJpZXMoY29udGFpbmVyLnNlY3JldHMpLm1hcCgoW2tleSwgdmFsdWVdKSA9PiB7XG4gICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIG5hbWU6IGtleSxcbiAgICAgICAgICAgIHZhbHVlRnJvbTogdmFsdWUuYXJuLFxuICAgICAgICAgIH07XG4gICAgICAgIH0pXG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgaW1hZ2U6IHRoaXMuaW1hZ2VDb25maWcuaW1hZ2VOYW1lLFxuICAgICAgaW5zdGFuY2VUeXBlOiBjb250YWluZXIuaW5zdGFuY2VUeXBlICYmIGNvbnRhaW5lci5pbnN0YW5jZVR5cGUudG9TdHJpbmcoKSxcbiAgICAgIGpvYlJvbGVBcm46IGNvbnRhaW5lci5qb2JSb2xlICYmIGNvbnRhaW5lci5qb2JSb2xlLnJvbGVBcm4sXG4gICAgICBleGVjdXRpb25Sb2xlQXJuOiBjb250YWluZXIuZXhlY3V0aW9uUm9sZSAmJiBjb250YWluZXIuZXhlY3V0aW9uUm9sZS5yb2xlQXJuLFxuICAgICAgbGludXhQYXJhbWV0ZXJzOiBjb250YWluZXIubGludXhQYXJhbXNcbiAgICAgICAgPyB7IGRldmljZXM6IGNvbnRhaW5lci5saW51eFBhcmFtcy5yZW5kZXJMaW51eFBhcmFtZXRlcnMoKS5kZXZpY2VzIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBsb2dDb25maWd1cmF0aW9uOiBjb250YWluZXIubG9nQ29uZmlndXJhdGlvbiA/IHtcbiAgICAgICAgbG9nRHJpdmVyOiBjb250YWluZXIubG9nQ29uZmlndXJhdGlvbi5sb2dEcml2ZXIsXG4gICAgICAgIG9wdGlvbnM6IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLm9wdGlvbnMsXG4gICAgICAgIHNlY3JldE9wdGlvbnM6IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLnNlY3JldE9wdGlvbnNcbiAgICAgICAgICA/IHRoaXMuYnVpbGRMb2dDb25maWd1cmF0aW9uU2VjcmV0T3B0aW9ucyhjb250YWluZXIubG9nQ29uZmlndXJhdGlvbi5zZWNyZXRPcHRpb25zKVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIG1vdW50UG9pbnRzOiBjb250YWluZXIubW91bnRQb2ludHMsXG4gICAgICBwcml2aWxlZ2VkOiBjb250YWluZXIucHJpdmlsZWdlZCB8fCBmYWxzZSxcbiAgICAgIG5ldHdvcmtDb25maWd1cmF0aW9uOiBjb250YWluZXIuYXNzaWduUHVibGljSXAgPyB7XG4gICAgICAgIGFzc2lnblB1YmxpY0lwOiBjb250YWluZXIuYXNzaWduUHVibGljSXAgPyAnRU5BQkxFRCcgOiAnRElTQUJMRUQnLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlYWRvbmx5Um9vdEZpbGVzeXN0ZW06IGNvbnRhaW5lci5yZWFkT25seSB8fCBmYWxzZSxcbiAgICAgIHVsaW1pdHM6IGNvbnRhaW5lci51bGltaXRzLFxuICAgICAgdXNlcjogY29udGFpbmVyLnVzZXIsXG4gICAgICB2b2x1bWVzOiBjb250YWluZXIudm9sdW1lcyxcbiAgICAgIGZhcmdhdGVQbGF0Zm9ybUNvbmZpZ3VyYXRpb246IGNvbnRhaW5lci5wbGF0Zm9ybVZlcnNpb24gPyB7XG4gICAgICAgIHBsYXRmb3JtVmVyc2lvbjogY29udGFpbmVyLnBsYXRmb3JtVmVyc2lvbixcbiAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICByZXNvdXJjZVJlcXVpcmVtZW50czogW1xuICAgICAgICB7IHR5cGU6ICdWQ1BVJywgdmFsdWU6IFN0cmluZyhjb250YWluZXIudmNwdXMgfHwgKGlzRmFyZ2F0ZSA/IDAuMjUgOiAxKSkgfSxcbiAgICAgICAgeyB0eXBlOiAnTUVNT1JZJywgdmFsdWU6IFN0cmluZyhjb250YWluZXIubWVtb3J5TGltaXRNaUIgfHwgKGlzRmFyZ2F0ZSA/IDUxMiA6IDQpKSB9LFxuICAgICAgXS5jb25jYXQoY29udGFpbmVyLmdwdUNvdW50ID8gW3sgdHlwZTogJ0dQVScsIHZhbHVlOiBTdHJpbmcoY29udGFpbmVyLmdwdUNvdW50KSB9XSA6IFtdKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZE5vZGVSYW5nZVByb3BzKG11bHRpTm9kZVByb3BzOiBJTXVsdGlOb2RlUHJvcHMsIGlzRmFyZ2F0ZTogYm9vbGVhbik6IENmbkpvYkRlZmluaXRpb24uTm9kZVJhbmdlUHJvcGVydHlQcm9wZXJ0eVtdIHtcbiAgICBjb25zdCByYW5nZVByb3BzID0gbmV3IEFycmF5PENmbkpvYkRlZmluaXRpb24uTm9kZVJhbmdlUHJvcGVydHlQcm9wZXJ0eT4oKTtcblxuICAgIGZvciAoY29uc3QgcHJvcCBvZiBtdWx0aU5vZGVQcm9wcy5yYW5nZVByb3BzKSB7XG4gICAgICByYW5nZVByb3BzLnB1c2goe1xuICAgICAgICBjb250YWluZXI6IHRoaXMuYnVpbGRKb2JDb250YWluZXIocHJvcC5jb250YWluZXIsIGlzRmFyZ2F0ZSksXG4gICAgICAgIHRhcmdldE5vZGVzOiBgJHtwcm9wLmZyb21Ob2RlSW5kZXggfHwgMH06JHtwcm9wLnRvTm9kZUluZGV4IHx8IG11bHRpTm9kZVByb3BzLmNvdW50fWAsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmFuZ2VQcm9wcztcbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRMb2dDb25maWd1cmF0aW9uU2VjcmV0T3B0aW9ucyhzZWNyZXRPcHRpb25zOiBFeHBvc2VkU2VjcmV0W10pOiBDZm5Kb2JEZWZpbml0aW9uLlNlY3JldFByb3BlcnR5W10ge1xuICAgIHJldHVybiBzZWNyZXRPcHRpb25zLm1hcChzZWNyZXRPcHRpb24gPT4ge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgbmFtZTogc2VjcmV0T3B0aW9uLm9wdGlvbk5hbWUsXG4gICAgICAgIHZhbHVlRnJvbTogc2VjcmV0T3B0aW9uLnNlY3JldEFybixcbiAgICAgIH07XG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==