# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_core.ipynb (unless otherwise specified).

__all__ = ['XLAOptimProxy', 'DeviceMoverTransform', 'isAffineCoordTfm', 'isDeviceMoverTransform', 'has_affinecoord_tfm',
           'has_devicemover_tfm', 'get_last_affinecoord_tfm_idx', 'insert_batch_tfm', 'XLAOptCallback']

# Internal Cell
from .utils import xla_imported

# Internal Cell
try:
    import torch_xla
except ImportError:
    pass

# Internal Cell
if xla_imported():
    import torch_xla.core.xla_model as xm

from fastcore.foundation import GetAttr, patch
from fastcore.transform import Transform,DisplayedTransform
from fastcore.basics import store_attr
from torch import Tensor
import torch
from fastai.vision.augment import AffineCoordTfm, RandomResizedCropGPU
from fastai.data.core import DataLoaders
from fastai.data.load import DataLoader
from fastai.learner import Learner
from fastai.callback.core import Callback, TrainEvalCallback
from fastai.learner import Recorder

# Cell

class XLAOptimProxy(GetAttr):
    "Proxy optimizer to override `opt.step` with Pytorch XLA sync method `xm.optimizer_step` "
    _default='opt'
    def __init__(self,opt, barrier):
        self.opt = opt
        self._barrier = barrier

    def step(self):
        xm.optimizer_step(self.opt,barrier=self._barrier)

    @property
    def barrier(self): return self._barrier
    @barrier.setter
    def barrier(self,v): self._barrier = v

# Cell
class DeviceMoverTransform(DisplayedTransform):
    "Transform to move input to new device and reverse to cpu"
    def __init__(self, device_to, device_from=torch.device('cpu')):
        store_attr('device_to,device_from')
    def encodes(self, o:Tensor):
        return o.to(self.device_to)
    def decodes(self, o:Tensor):
        return o.to(self.device_from)

# Cell

def isAffineCoordTfm(o:Transform):
    "check whether the transform is either an AffineCoordTfm or RandomResizedCropGPU"
    return isinstance(o,(AffineCoordTfm,RandomResizedCropGPU))

def isDeviceMoverTransform(o:Transform):
    "check whether the transform is a DeviceMoverTransform"
    return isinstance(o,DeviceMoverTransform)

def has_affinecoord_tfm(dls: DataLoaders) -> bool:
    "returns true if train dataloader has an AffineCoordTfm in the batch_tfms"
    if not hasattr(dls.train,'after_batch'): return False
    if not hasattr(dls.train.after_batch,'fs'): return False
    idxs = dls.train.after_batch.fs.argwhere(isAffineCoordTfm)
    return len(idxs) > 0
def has_devicemover_tfm(dl: DataLoader) -> bool:
    "returns true if train dataloader has a DeviceMoverTransform in the batch_tfms"
    if not hasattr(dl,'after_batch'): return False
    if not hasattr(dl.after_batch,'fs'): return False
    idxs = dl.after_batch.fs.argwhere(isDeviceMoverTransform)
    return len(idxs) > 0

def get_last_affinecoord_tfm_idx(dl:DataLoader)-> int: # -1 if none
    "returns index of last AffineCoordTfm if it exists, otherwise returns -1"
    idxs = dl.after_batch.fs.argwhere(isAffineCoordTfm)
    return -1 if len(idxs) == 0 else idxs[-1]

# Cell
def insert_batch_tfm(dl:DataLoader, batch_tfm:Transform, idx:int):
    "adds a batch_tfm in the batch_tfms for the dataloader at idx location"
    dl.after_batch.fs.insert(idx, batch_tfm)

# Cell
@patch
def setup_input_device_mover(self: Learner, new_device):
    "setup batch_tfms to use cpu if dataloader batch_tfms has AffineCoordTfms"
    if not has_affinecoord_tfm(self.dls):
        self.dls.device = new_device
        return
    self.dls.device = None
    if has_devicemover_tfm(self.dls.train):
        return # skip adding device mover if already added
    dm_tfm = DeviceMoverTransform(new_device)
    for dl in self.dls.loaders:
        if not has_devicemover_tfm(dl):
            idx = get_last_affinecoord_tfm_idx(dl)
            if idx != -1:
                insert_batch_tfm(dl, dm_tfm, idx+1)

# Cell

class XLAOptCallback(Callback):
    'Callback to replace `opt.step` with `xm.optimizer_step(opt)` as required to run on TPU'
    run_after,run_before = TrainEvalCallback,Recorder
    def __init__(self, barrier=True):
        self._barrier = barrier

    def before_fit(self):
        'replace opt with proxy which calls `xm.optimizer_step` instead of `opt.step` and set `dls.device` and model to `xla_device`'
        # set dls device to none so prevent trigger of moving to batch input to XLA device
        # as this move will be done by the DeviceMoverTransform which has been added to the dls after_batch tfms
        if has_affinecoord_tfm(self.dls):
            self.dls.device = None

        if self.learn.opt is not None:
            if not isinstance(self.learn.opt,XLAOptimProxy):
                # force opt to reinitialize its parameters and make sure its parameters
                opt = self.learn.opt
                self.learn.opt = XLAOptimProxy(opt, barrier=self._barrier)

    def after_fit(self):
        'restore original opt '
        if isinstance(self.learn.opt, XLAOptimProxy):
            opt = self.learn.opt.opt
            self.learn.opt = opt
    @property
    def barrier(self): return self._barrier
    @barrier.setter
    def barrier(self,v): self._barrier = v

# Cell

@patch
def to_xla(self:Learner, new_device=None):
    "Setup learner for single tpu core training"
    self.add_cb(XLAOptCallback())
    if new_device is None:
        new_device = xm.xla_device()
    self.model.to(new_device)
    self.setup_input_device_mover(new_device)
    self.opt = None
    return self

# Cell

@patch
def detach_xla(self:Learner):
    "reset TPU single core setup and move model and dls back to cpu "
    self.remove_cb(XLAOptCallback)
    self.dls.device = torch.device('cpu')
    self.model = self.model.to(self.dls.device)
    self.opt = None
    return self