# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['typeddfs']

package_data = \
{'': ['*']}

install_requires = \
['natsort>=7',
 'pandas>=1.2,<2.0',
 'regex>=2021.8.28',
 'tabulate>=0.8,<0.9',
 'wcwidth>=0.2']

extras_require = \
{'all': ['pyarrow>=4',
         'lxml>=4',
         'openpyxl>=3',
         'pyxlsb>=1',
         'html5lib>=1',
         'beautifulsoup4>=4'],
 'excel': ['openpyxl>=3'],
 'feather': ['pyarrow>=4'],
 'html': ['html5lib>=1', 'beautifulsoup4>=4'],
 'main': ['pyarrow>=4', 'lxml>=4', 'openpyxl>=3'],
 'parquet': ['pyarrow>=4'],
 'toml': ['tomlkit>=0.7'],
 'xlsb': ['pyxlsb>=1'],
 'xml': ['lxml>=4']}

setup_kwargs = {
    'name': 'typeddfs',
    'version': '0.13.2',
    'description': 'Pandas DataFrame subclasses that enforce structure and can self-organize.',
    'long_description': '# Typed DataFrames\n\n[![Version status](https://img.shields.io/pypi/status/typeddfs?label=status)](https://pypi.org/project/typeddfs)\n[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Python version compatibility](https://img.shields.io/pypi/pyversions/typeddfs?label=Python)](https://pypi.org/project/typeddfs)\n[![Version on Github](https://img.shields.io/github/v/release/dmyersturnbull/typed-dfs?include_prereleases&label=GitHub)](https://github.com/dmyersturnbull/typed-dfs/releases)\n[![Version on PyPi](https://img.shields.io/pypi/v/typeddfs?label=PyPi)](https://pypi.org/project/typeddfs)  \n[![Build (Actions)](https://img.shields.io/github/workflow/status/dmyersturnbull/typed-dfs/Build%20&%20test?label=Tests)](https://github.com/dmyersturnbull/typed-dfs/actions)\n[![Coverage (coveralls)](https://coveralls.io/repos/github/dmyersturnbull/typed-dfs/badge.svg?branch=main&service=github)](https://coveralls.io/github/dmyersturnbull/typed-dfs?branch=main)\n[![Documentation status](https://readthedocs.org/projects/typed-dfs/badge)](https://typed-dfs.readthedocs.io/en/stable/)\n[![Maintainability](https://api.codeclimate.com/v1/badges/6b804351b6ba5e7694af/maintainability)](https://codeclimate.com/github/dmyersturnbull/typed-dfs/maintainability)\n[![Scrutinizer Code Quality](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/badges/quality-score.png?b=main)](https://scrutinizer-ci.com/g/dmyersturnbull/typed-dfs/?branch=main)\n[![Created with Tyrannosaurus](https://img.shields.io/badge/Created_with-Tyrannosaurus-0000ff.svg)](https://github.com/dmyersturnbull/tyrannosaurus)\n\nPandas DataFrame subclasses that self-organize and read/write correctly.\n\n```python\nFilm = TypedDfs.typed("Film").require("name", "studio", "year").build()\ndf = Film.read_csv("file.csv")\nassert df.columns.tolist() == ["name", "studio", "year"]\n```\n\nYour types will remember how they’re supposed to be read,\nincluding dtypes, columns for set_index, and custom requirements.\nThen you can stop passing index_cols=, header=, set_index, and astype each time you read.\nInstead, `read_csv`, `read_parquet`, ..., will just work.\n\nYou can also document your functions clearly,\nand read and write _any_ format in a single file.\n\n```python\ndef hello(df: Film):\n    print("read!")\n\n\ndf = Film.read_file("input file? [.csv/.tsv/.tab/.feather/.snappy/.json.gz/.h5/...]")\nhello(df)\n```\n\nYou can read/write TOML, INI, .properties, fixed-width format, and any compressed variants.\nNeed dataclasses?\n\n```python\ninstances = df.to_dataclass_instances()\nFilm.from_dataclass_instances(instances)\n```\n\n### 🐛 Pandas serialization bugs fixed\n\nPandas has several issues with serialization.\nDepending on the format and columns, these issues occur:\n\n- columns being silently added or dropped,\n- errors on either read or write of empty DataFrames,\n- the inability to use DataFrames with indices in Feather,\n- writing to Parquet failing with half-precision,\n- lingering partially written files on error,\n- the buggy xlrd being preferred by read_excel,\n- the buggy odfpy also being preferred,\n- writing a file and reading it back results in a different DataFrame,\n- you can’t write fixed-width format,\n- and the platform text encoding being used rather than utf-8.\n\n### 🎁️ New methods, etc.\n\nDocs coming soon...\n\n\n### 🎨 Simple example\n\n```python\nfrom typeddfs import TypedDfs\n\nMyDfType = (\n    TypedDfs.typed("MyDfType")\n    .require("name", index=True)  # always keep in index\n    .require("value", dtype=float)  # require a column and type\n    .drop("_temp")  # auto-drop a column\n    .verify(lambda ddf: len(ddf) == 12)  # require exactly 12 rows\n).build()\n\ndf = MyDfType.read_file(input("filename? [.feather/.csv.gz/.tsv.xz/etc.]"))\ndf = df.sort_natural()\ndf.write_file("myfile.feather", mkdirs=True)\n# want to write to a sha1sum-like (.sha256) file?\ndf.write_file("myfile.feather", file_hash=True)\n# verify it?\nMyDfType.read_file("myfile.feather", check_hash="file")\n```\n\n### 📉 A matrix-style DataFrame\n\n```python\nimport numpy as np\nfrom typeddfs import TypedDfs\n\nSymmetric64 = (\n    TypedDfs.matrix("Symmetric64", doc="A symmetric float64 matrix")\n    .dtype(np.float64)\n    .verify(lambda df: df.values.sum().sum() == 1.0)\n    .add_methods(product=lambda df: df.flatten().product())\n).build()\n\nmx = Symmetric64.read_file("input.tab")\nprint(mx.product())  # defined above\nif mx.is_symmetric():\n    mx = mx.triangle()  # it\'s symmetric, so we only need half\n    long = mx.drop_na().long_form()  # columns: "row", \'column", and "value"\n    long.write_file("long-form.xml")\n```\n\n### 🔍 More complex example\n\nFor a CSV like this:\n\n| key | value | note |\n| --- | ----- | ---- |\n| abc | 123   | ?    |\n\n```python\nfrom typeddfs import TypedDfs\n\n# Build me a Key-Value-Note class!\nKeyValue = (\n    TypedDfs.typed("KeyValue")  # With enforced reqs / typing\n    .require("key", dtype=str, index=True)  # automagically add to index\n    .require("value")  # required\n    .reserve("note")  # permitted but not required\n    .strict()  # disallow other columns\n).build()\n\n# This will self-organize and use "key" as the index:\ndf = KeyValue.read_csv("example.csv")\n\n# For fun, let"s write it and read it back:\ndf.to_csv("remake.csv")\ndf = KeyValue.read_csv("remake.csv")\nprint(df.index_names(), df.column_names())  # ["key"], ["value", "note"]\n\n# And now, we can type a function to require a KeyValue,\n# and let it raise an `InvalidDfError` (here, a `MissingColumnError`):\ndef my_special_function(df: KeyValue) -> float:\n    return KeyValue(df)["value"].sum()\n```\n\nAll of the normal DataFrame methods are available.\nUse `.untyped()` or `.vanilla()` to make a detyped copy that doesn’t enforce requirements.\nUse `.of(df)` to convert a DataFrame to your type.\n\n### 🗨️ Q & A\n\n**What is the difference between `__init__`, `convert`, and `of`?**\n\nThese three methods in `TypedDf` (and its superclasses) are a bit different.\n`__init__` does NOT attempt to reorganize or validate your DataFrame,\nwhile `convert` and `of` do.\n`of` is simply more flexible than `convert`: `convert` only accepts a DataFrame,\nwhile `of` will take anything that `DataFrame.__init__` will.\n\n**When do typed DFs "detype" during chained invocations?**\n\nMost DataFrame-level functions that ordinarily return DataFrames themselves\ntry to keep the same type.\nThis includes `reindex`, `drop_duplicates`, `sort_values`, and `set_index`.\nThis is to allow for easy chained invocation, but it’s important to note\nthat the returned DataFrame might not conform to your requirements.\nCall `retype` at the end to reorganize and verify.\n\n```python\nfrom typeddfs import TypedDfs\n\nMyDf = TypedDfs.typed("MyDf").require("valid").build()\nmy_df = MyDf.read_csv("x.csv")\nmy_df_2 = my_df.drop_duplicates().rename_cols(valid="ok")\nprint(type(my_df_2))  # type(MyDf)\n# but this fails!\nmy_df_3 = my_df.drop_duplicates().rename_cols(valid="ok").retype()\n# MissingColumnError "valid"\n```\n\nYou can call `.detype()` to remove any typing rules\nand `.vanilla()` if you need a plain DataFrame.\n\n**How does one get the typing info?**\n\nCall `.get_typing()`\n\n```python\nfrom typeddfs import TypedDfs\n\nMyDf = TypedDfs.typed("MyDf").require("valid").build()\nMyDf.get_typing().required_columns  # ["valid"]\n```\n\n**How are toml documents read and written?**\n\nThese are limited to a single array of tables (AOT).\nThe AOT is named `row` by default (set with `aot=`).\nOn read, you can pass `aot=None` to have it use the unique outermost key.\n\n**How are INI files read and written?**\n\nThese require exactly 2 columns after `reset_index()`.\nParsing is purposefully minimal because these formats are flexible.\nTrailing whitespace and whitespace surrounding `=` is ignored.\nValues are not escaped, and keys may not contain `=`.\nLine continuation with `\\` is not allowed.\nQuotation marks surrounding values are not dropped,\nunless `drop_quotes=True` is passed.\nComments begin with `;`, along with `#` if `hash_sign=True` is passed.\n\nOn read, section names are prepended to the keys.\nFor example, the key name will be `section.key` in this example:\n\n```ini\n[section]\nkey = value\n```\n\nOn write, the inverse happens.\n\n**What about .properties?**\n\nThese are similar to INI files.\nOnly hash signs are allowed for comments, and reserved chars\n*are* escaped in keys.\nThis includes `\\\\`,`\\ `, `\\=`, and `\\:` These are not escaped in values.\n\n**What is "flex-width format"?**\n\nThis is a format that shows up a lot in the wild, but doesn’t seem to have a name.\nIt’s just a text format like TSV or CSV, but where columns are preferred to line up\nin a fixed-width font. Whitespace is ignored on read, but on write the columns are made\nto line up neatly. These files are easy to view.\nBy default, the delimiter is three vertical bars (`|||`).\n\n**When are read and write guaranteed to be inverses?**\n\nIn principle, this invariant holds when you call `.strict()` to disallow\nadditional columns and specify `dtype=` in all calls to `.require` and `.reserve`.\nIn practice, this might break down for certain combinations of\nDataFrame structure, dtypes, and serialization format.\nIt seems pretty solid for Feather, Parquet, and CSV/TSV-like variants,\nespecially if the dtypes are limited to bools, real values, int values, and strings.\nThere may be corner cases for XML, TOML, INI, Excel, OpenDocument, and HDF5,\nas well as for categorical and miscellaneous `object` dtypes.\n\n**How do I include another filename suffix?**\n\nUse `.suffix()` to register a suffix or remap it to another format.\n\n```python\nfrom typeddfs import TypedDfs, FileFormat\n\nMyDf = TypedDfs.typed("MyDf").suffix(tabbed="tsv").build()\n# or:\nMyDf = TypedDfs.typed("MyDf").suffix(**{".tabbed": FileFormat.tsv}).build()\n```\n\n**How do the checksums work?**\n\nThere are simple convenience flags to write sha1sum-like files while\nwriting files, and to verify them when reading.\n\n\n```python\nfrom pathlib import Path\nfrom typeddfs import TypedDfs\n\nMyDf = TypedDfs.typed("MyDf").build()\ndf = MyDf()\ndf.write_file("here.csv", file_hash=True)\n# a hex-encoded hash and filename\nPath("here.csv.sha256").read_text(encoding="utf8")\nMyDf.read_file("here.csv", file_hash=True)  # verifies that it matches\n```\n\nYou can change the hash algorithm with `.hash()`.\nThe second variant is `dir_hash`.\n\n```python\nfrom pathlib import Path\nfrom typeddfs import TypedDfs, Checksums\n\nMyDf = TypedDfs.typed("MyDf").build()\ndf = MyDf()\npath = Path("dir", "here.csv")\ndf.write_file(path, dir_hash=True, mkdirs=True)\n# potentially many hex-encoded hashes and filenames; always appended to\nMyDf.read_file(path, dir_hash=True)  # verifies that it matches\n\n# read it\nsums = Checksums.parse_hash_file_resolved(Path("my_dir", "my_dir.sha256"))\nsums[path]  # return the hex hash\n```\n\n\n### 💔 Limitations\n\n- Multi-level columns are not yet supported.\n- Columns and index levels cannot share names.\n- Duplicate column names are not supported. (These are strange anyway.)\n- A typed DF cannot have columns "level_0", "index", or "Unnamed: 0".\n- `inplace` is forbidden in some functions; avoid it or use `.vanilla()`.\n\n### 🔌 Serialization support\n\nLike Pandas, TypedDfs can read and write to various formats.\nIt provides the methods `read_file` and `write_file`, which guess the format from the\nfilename extension. For example, `df.write_file("myfile.snappy)` writes Parquet files,\nand `df.write_file("myfile.tab.gz")` writes a gzipped, tab-delimited file.\nThe `read_file` method works the same way: `MyDf.read_file("myfile.feather")` will\nread an Apache Arrow Feather file, and `MyDf.read_file("myfile.json.gzip")`reads\na gzipped JSON file. You can pass keyword arguments to those functions.\n\nSerialization is provided through Pandas, and some formats require additional packages.\nPandas does not specify compatible versions, so typed-dfs specifies\n[extras](https://python-poetry.org/docs/pyproject/#extras) are provided in typed-dfs\nto ensure that those packages are installed with compatible versions.\n\n- To install with [Feather](https://arrow.apache.org/docs/python/feather.html) support,\n  use `pip install typeddfs[feather]`.\n- To install with support for all formats,\n  use `pip install typeddfs[all]`.\n\nFeather offers massively better performance over CSV, gzipped CSV, and HDF5\nin read speed, write speed, memory overhead, and compression ratios.\nParquet typically results in smaller file sizes than Feather at some cost in speed.\nFeather is the preferred format for most cases.\n\n### 📊 Serialization in-depth\n\n**⚠ Note:** The `hdf5` extra is currently disabled.\n\n| format   | packages                     | extra     | sanity | speed | file sizes |\n| -------- | ---------------------------- | --------- | ------ | ----- | ---------- |\n| Feather  | `pyarrow`                    | `feather` | +++    | ++++  | +++        |\n| Parquet  | `pyarrow` or `fastparquet` † | `parquet` | ++     | +++   | ++++       |\n| csv/tsv  | none                         | none      | ++     | −−    | −−         |\n| flexwf ‡ | none                         | none      | ++     | −−    | −−         |\n| .fwf     | none                         | none      | +      | −−    | −−         |\n| json     | none                         | none      | −−     | −−−   | −−−        |\n| xml      | `lxml`                       | `xml`     | −      | −−−   | −−−        |\n| .properties | none                      | none      | −−     | −−    | −−         |\n| toml     | `tomlkit`                    | `toml`    | −−     | −−    | −−         |\n| INI      | none                         | none      | −−−   | −−    | −−         |\n| .lines   | none                         | none      | ++     | −−    | −−         |\n| .npy     | none                         | none      | −      | +     | +++        |\n| .npz     | none                         | none      | −      | +     | +++        |\n| .html    | `html5lib,beautifulsoup4`    | `html`    | −−     | −−−   | −−−        |\n| pickle   | none                         | none      | −−     | −−−   | −−−        |\n| XLSX     | `openpyxl,defusedxml`        | `excel`   | +      | −−    | +          |\n| ODS      | `openpyxl,defusedxml`        | `excel`   | +      | −−    | +          |\n| XLS      | `openpyxl,defusedxml`        | `excel`   | −−     | −−    | +          |\n| XLSB     | `pyxlsb`                     | `xlsb`    | −−     | −−    | ++         |\n| HDF5     | `tables`                     | `hdf5`    | −−     | −     | ++         |\n\n**Notes:**\n\n- † `fastparquet` can be used instead. It is slower but much smaller.\n- Parquet only supports str, float64, float32, int64, int32, and bool.\n  Other numeric types are automatically converted during write.\n- ‡ `.flexwf` is fixed-width with optional delimiters.\n- JSON has inconsistent handling of `None`. ([orjson](https://github.com/ijl/orjson) is more consistent).\n- XML requires Pandas 1.3+.\n- Not all JSON, XML, TOML, and HDF5 files can be read.\n- .ini and .properties can only be written with exactly 2 columns + index levels:\n  a key and a value. INI keys are in the form `section.name`.\n- .lines can only be written with exactly 1 column or index level.\n- .npy and .npz only serialize numpy objects.\n  They are not supported in `read_file` and `write_file`.\n- .html is not supported in `read_file` and `write_file`.\n- Pickle is insecure and not recommended.\n- Pandas supports odfpy for ODS and xlrd for XLS. In fact, it prefers those.\n  However, they are very buggy; openpyxl is much better.\n- XLSM, XLTX, XLTM, XLS, and XLSB files can contain macros, which Microsoft Excel will ingest.\n- XLS is a deprecated format.\n- XLSB is not fully supported in Pandas.\n- HDF may not work on all platforms yet due to a\n  [tables issue](https://github.com/PyTables/PyTables/issues/854).\n\n### 🔒 Security\n\nRefer to the [security policy](https://github.com/dmyersturnbull/typed-dfs/blob/main/SECURITY.md).\n\n### 📝 Extra notes\n\nDependencies in the extras are only restricted to minimum version numbers;\nlibraries that use them can set their own version ranges.\nFor example, typed-dfs only requires tables >= 0.4, but Pandas can further restrict it.\n[natsort](https://github.com/SethMMorton/natsort) is also only assigned a minimum version number;\nthis is because it receives frequent major version bumps.\nThis means that the result of typed-df’s `sort_natural` could change.\nTo fix this, pin natsort to a specific major version;\ne.g. `natsort = "^7"` with [Poetry](https://python-poetry.org/) or `natsort>=7,<8` with pip.\n\n### 🍁 Contributing\n\nTyped-Dfs is licensed under the [Apache License, version 2.0](https://www.apache.org/licenses/LICENSE-2.0).\n[New issues](https://github.com/dmyersturnbull/typed-dfs/issues) and pull requests are welcome.\nPlease refer to the [contributing guide](https://github.com/dmyersturnbull/typed-dfs/blob/main/CONTRIBUTING.md).\nGenerated with [Tyrannosaurus](https://github.com/dmyersturnbull/tyrannosaurus).\n',
    'author': 'Douglas Myers-Turnbull',
    'author_email': None,
    'maintainer': 'dmyersturnbull',
    'maintainer_email': None,
    'url': 'https://github.com/dmyersturnbull/typed-dfs',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
