"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoScaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
class AutoScaler extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const launchTemplate = this.getLT(props.templateProps, props.asgName);
        this.loadBalancerProperties = this.getTG(props.networkProps, props.templateProps.vpc.vpcName, props.appName);
        new aws_autoscaling_1.CfnAutoScalingGroup(this, props.asgName, {
            maxSize: props.maxSize,
            minSize: props.minSize,
            autoScalingGroupName: props.asgName,
            launchTemplate: {
                version: launchTemplate.versionNumber,
                launchTemplateId: launchTemplate.launchTemplateId,
                launchTemplateName: launchTemplate.launchTemplateName,
            },
            targetGroupArns: this.loadBalancerProperties.map((lb) => { return lb.targetGroupArn; }),
            tags: props.tags,
            availabilityZones: this.getZones(props.subnets),
            vpcZoneIdentifier: props.subnets,
            healthCheckGracePeriod: 300,
        });
    }
    getVPC(props) {
        const stackVPC = aws_ec2_1.Vpc.fromLookup(this, props.vpcName, {
            isDefault: false,
            vpcId: props.vpcName,
        });
        return stackVPC;
    }
    getRole(props, asgName) {
        var _b;
        if (props.type == 'existing') {
            const role = aws_iam_1.Role.fromRoleArn(this, asgName + '-stackRole', props.roleArn);
            return role;
        }
        else {
            const role = new aws_iam_1.Role(this, asgName + '-stackRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                roleName: asgName + '-role',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2SpotFleetTaggingRole',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/ReadOnlyAccess',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/AmazonEC2FullAccess',
            });
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'ec2:DescribeTags',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
                sid: 'VisualEditor0',
            }));
            (_b = props.additionalPolicies) === null || _b === void 0 ? void 0 : _b.forEach(policyDoc => {
                role.addToPolicy(aws_iam_1.PolicyStatement.fromJson(policyDoc));
            });
            return role;
        }
    }
    getSG(props, vpc, asgName) {
        var _b, _c, _d;
        if (props.type == 'existing') {
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, 'stack-sg', props.sgGroupId);
            return securityGroup;
        }
        else {
            const sgProps = {
                securityGroupName: (_b = props.securityGroupName) !== null && _b !== void 0 ? _b : 'stack-sg-group',
                vpc: vpc,
                allowAllOutbound: (_c = props.allowAllOutbound) !== null && _c !== void 0 ? _c : true,
                disableInlineRules: (_d = props.disableInlineRules) !== null && _d !== void 0 ? _d : false,
            };
            const securityGroup = new aws_ec2_1.SecurityGroup(this, asgName + '-stack-sg', sgProps);
            props.ingressRules.forEach(ingress => {
                var _b;
                const ingressSG = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, ingress.sourceSG + ingress.port.toString(), ingress.sourceSG);
                securityGroup.connections.allowFrom(ingressSG, aws_ec2_1.Port.tcp(ingress.port), (_b = ingress.description) !== null && _b !== void 0 ? _b : 'Application port');
            });
            return securityGroup;
        }
    }
    getBD(props) {
        const bd = {
            deviceName: props.name,
            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(props.size, {
                volumeType: props.type,
                deleteOnTermination: true,
            }),
        };
        return bd;
    }
    getLT(props, asgName) {
        if (props.type == 'existing') {
            const launchTemplate = aws_ec2_1.LaunchTemplate.fromLaunchTemplateAttributes(this, props.templateName, props.existingAttributes);
            return launchTemplate;
        }
        else {
            const launchTemplate = new aws_ec2_1.LaunchTemplate(this, props.templateName, {
                launchTemplateName: props.templateName,
                instanceType: new aws_ec2_1.InstanceType(props.instanceType),
                machineImage: aws_ec2_1.MachineImage.lookup({
                    name: props.amiImageId,
                }),
                securityGroup: this.getSG(props.securityGroup, this.getVPC(props.vpc), asgName),
                role: this.getRole(props.role, asgName),
                detailedMonitoring: false,
                blockDevices: [this.getBD(props.blockDevice)],
                keyName: props.sshKey,
            });
            return launchTemplate;
        }
    }
    getTG(props, vpcId, appName) {
        let lbProps = [];
        props.forEach(t => {
            var _b;
            const tg = new aws_elasticloadbalancingv2_1.CfnTargetGroup(this, appName + t.port.toString(), {
                name: appName + ((_b = t.port) === null || _b === void 0 ? void 0 : _b.toString()),
                healthCheckEnabled: true,
                healthCheckPath: t.healthCheckPath,
                ...((t.protocol == 'GRPC') ? { protocol: 'HTTP' } : { protocol: t.protocol }),
                ...((t.protocol == 'GRPC') ? { protocolVersion: 'GRPC' } : {}),
                healthCheckTimeoutSeconds: 5,
                healthCheckPort: String(t.port),
                port: t.port,
                vpcId: vpcId,
            });
            lbProps.push({
                appName: appName,
                hostHeader: t.host,
                lbArn: t.lbArn,
                sslEnabled: t.sslEnabled,
                targetGroupArn: tg.ref,
                zoneName: t.zoneName,
            });
        });
        return lbProps;
    }
    getZones(subnets) {
        var availabilityZones = [];
        subnets.forEach(subnet => {
            const net = aws_ec2_1.Subnet.fromSubnetAttributes(this, subnet, {
                availabilityZone: 'dummy',
                subnetId: subnet,
            });
            availabilityZones.push(net.availabilityZone);
        });
        return availabilityZones;
    }
}
exports.AutoScaler = AutoScaler;
_a = JSII_RTTI_SYMBOL_1;
AutoScaler[_a] = { fqn: "@smallcase/aws-cdk-microservice.AutoScaler", version: "0.0.17" };
//# sourceMappingURL=data:application/json;base64,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