"""Support Peewee ORM for Muffin framework."""

import typing as t

import muffin
from muffin.plugins import BasePlugin
from peewee_migrate import Router
from peewee_aio import Manager

from .fields import JSONField, Choices


__version__ = "0.0.0"
__project__ = "muffin-peewee-aio"
__author__ = "Kirill Klenov <horneds@gmail.com>"
__license__ = "MIT"


__all__ = 'Plugin', 'JSONField', 'Choices'


assert JSONField and Choices


class Plugin(BasePlugin):

    """Muffin Peewee Plugin."""

    name = "peewee"
    defaults = {

        # Connection params
        'connection': 'sqlite+async:///db.sqlite',
        'connection_params': {},

        # Manage connections automatically
        'auto_connection': True,
        'auto_transaction': True,

        # Setup migration engine
        'migrations_enabled': True,
        'migrations_path': 'migrations',
    }

    router: t.Optional[Router] = None
    manager: t.Optional[Manager] = None

    def setup(self, app: muffin.Application, **options):
        """Init the plugin."""
        super().setup(app, **options)
        self.manager = Manager(self.cfg.connection, **self.cfg.connection_params)

        if self.cfg.migrations_enabled:
            self.router = Router(self.manager.pw_database, migrate_dir=self.cfg.migrations_path)

            # Register migration commands
            @app.manage
            def pw_migrate(name: str = None, fake: bool = False):
                """Run application's migrations.

                :param name: Choose a migration' name
                :param fake: Run as fake. Update migration history and don't touch the database
                """
                with self.manager.allow_sync():
                    self.router.run(name, fake=fake)

            @app.manage
            def pw_create(name: str = 'auto', auto: bool = False):
                """Create a migration.

                :param name: Set name of migration [auto]
                :param auto: Track changes and setup migrations automatically
                """
                with self.manager.allow_sync():
                    self.router.create(name, auto and [m for m in self.models.values()])

            @app.manage
            def pw_rollback(name: str = None):
                """Rollback a migration.

                :param name: Migration name (actually it always should be a last one)
                """
                with self.manager.allow_sync():
                    self.router.rollback(name)

            @app.manage
            def pw_list():
                """List migrations."""
                with self.manager.allow_sync():
                    self.router.logger.info('Migrations are done:')
                    self.router.logger.info('\n'.join(self.router.done))
                    self.router.logger.info('')
                    self.router.logger.info('Migrations are undone:')
                    self.router.logger.info('\n'.join(self.router.diff))

        if self.cfg.auto_connection:
            app.middleware(self.get_middleware(), insert_first=True)

    async def startup(self):
        """Connect to the database (initialize a pool and etc)."""
        await self.manager.connect()

    async def shutdown(self):
        """Disconnect from the database (close a pool and etc.)."""
        await self.manager.disconnect()

    def __getattr__(self, name: str) -> t.Any:
        """Proxy attrs to self database."""
        return getattr(self.manager, name)

    def get_middleware(self):
        """Generate a middleware to manage connection/transaction."""

        async def middleware(handler, request, receive, send):
            async with self.manager.connection(True):
                return await handler(request, receive, send)

        if self.cfg.auto_transaction:

            async def middleware(handler, request, receive, send):  # noqa
                async with self.manager.connection(True):
                    async with self.manager.transaction():
                        return await handler(request, receive, send)

        return middleware

    def conftest(self) -> t.Awaitable:
        """Prepare database tables for tests."""
        return self.manager.create_tables(*self.manager)
