"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.clusterDns,
                            props.endpoints.managerDns,
                            props.endpoints.telemetryDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                // ingressController: {
                //   enabled: true,
                //   installCRDs: false,
                //   image: {
                //     repository: 'kong/kubernetes-ingress-controller',
                //     tag: '1.3.1-alpine',
                //   },
                // },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi9zcmMvcmVzb3VyY2VzL2Vrcy9oZWxtLWNoYXJ0cy9rb25nL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUF5RTtBQUN6RSwyQ0FBdUM7QUF1QnZDLE1BQWEsbUJBQW9CLFNBQVEsc0JBQVM7SUFDaEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQjtRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQztRQUNsQyxNQUFNLDBCQUEwQixHQUFHLDRCQUE0QixDQUFDO1FBRWhFLE1BQU0sY0FBYyxHQUFHLElBQUkscUJBQU8sQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQzNFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixTQUFTLEVBQUUsSUFBSTtZQUNmLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxVQUFVLEVBQUUsSUFBSTtvQkFDaEIsSUFBSSxFQUFFLFdBQVc7b0JBQ2pCLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsU0FBUyxFQUFFO2lCQUNwQzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLHFCQUFPLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDM0UsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLElBQUksRUFBRSx5QkFBeUI7WUFDL0IsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzNCLENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUN6QyxNQUFNLFdBQVcsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFFM0MsdUJBQXVCLENBQUMsb0JBQW9CLENBQUMsSUFBSSxxQkFBTyxDQUFDLGVBQWUsQ0FBQztZQUN2RSxTQUFTLEVBQUU7Z0JBQ1QsMEJBQTBCLFVBQVUsSUFBSSxXQUFXLFdBQVcsS0FBSyxDQUFDLG1CQUFtQixTQUFTO2dCQUNoRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRTthQUNyQztZQUNELE9BQU8sRUFBRTtnQkFDUCwrQkFBK0I7Z0JBQy9CLCtCQUErQjthQUNoQztTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosdUJBQXVCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzRCxNQUFNLGdCQUFnQixHQUFHLElBQUkscUJBQU8sQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3hFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixTQUFTLEVBQUUsSUFBSTtZQUNmLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxVQUFVLEVBQUUsb0JBQW9CO29CQUNoQyxJQUFJLEVBQUUsYUFBYTtvQkFDbkIsUUFBUSxFQUFFO3dCQUNSLElBQUksRUFBRSxLQUFLLENBQUMsVUFBVTt3QkFDdEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO3FCQUMzQjtvQkFDRCxJQUFJLEVBQUU7d0JBQ0osVUFBVSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsY0FBYzt3QkFDMUMsUUFBUSxFQUFFOzRCQUNSLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUTs0QkFDeEIsS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVOzRCQUMxQixLQUFLLENBQUMsU0FBUyxDQUFDLFVBQVU7NEJBQzFCLEtBQUssQ0FBQyxTQUFTLENBQUMsWUFBWTs0QkFDNUIsS0FBSyxDQUFDLFNBQVMsQ0FBQyxjQUFjO3lCQUMvQjt3QkFDRCxRQUFRLEVBQUUsV0FBVzt3QkFDckIsU0FBUyxFQUFFOzRCQUNULEtBQUssRUFBRSx3QkFBd0I7NEJBQy9CLElBQUksRUFBRSxxQkFBcUI7NEJBQzNCLElBQUksRUFBRSxLQUFLLENBQUMsaUJBQWlCO3lCQUM5Qjt3QkFDRCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsVUFBVSxFQUFFLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixTQUFTO3dCQUMvQyxNQUFNLEVBQUU7NEJBQ04sYUFBYTs0QkFDYixhQUFhO3lCQUNkO3dCQUNELFVBQVUsRUFBRTs0QkFDVixTQUFTLEVBQUUsS0FBSzs0QkFDaEIsSUFBSSxFQUFFLElBQUk7eUJBQ1g7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILGdCQUFnQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFcEQsd0VBQXdFO1FBQ3hFLGdIQUFnSDtRQUVoSCxNQUFNLGlCQUFpQixHQUFHLElBQUkscUJBQU8sQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzdFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixTQUFTLEVBQUUsSUFBSTtZQUNmLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxVQUFVLEVBQUUsK0JBQStCO29CQUMzQyxJQUFJLEVBQUUscUJBQXFCO29CQUMzQixRQUFRLEVBQUU7d0JBQ1IsSUFBSSxFQUFFLDBCQUEwQjt3QkFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO3FCQUMzQjtvQkFDRCxJQUFJLEVBQUU7d0JBQ0osUUFBUSxFQUFFLEtBQUs7d0JBQ2YsVUFBVSxFQUFFOzRCQUNWLE9BQU8sRUFBRSxrQkFBa0IsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsYUFBYSx3TkFBd047eUJBQ25SO3dCQUNELGFBQWEsRUFBRTs0QkFDYjtnQ0FDRSxVQUFVLEVBQUUsV0FBVztnQ0FDdkIsSUFBSSxFQUFFLFFBQVE7Z0NBQ2QsSUFBSSxFQUFFO29DQUNKO3dDQUNFLFVBQVUsRUFBRSxZQUFZO3dDQUN4QixHQUFHLEVBQUUsWUFBWTtxQ0FDbEI7b0NBQ0Q7d0NBQ0UsVUFBVSxFQUFFLFlBQVk7d0NBQ3hCLEdBQUcsRUFBRSxZQUFZO3FDQUNsQjtvQ0FDRDt3Q0FDRSxVQUFVLEVBQUUsUUFBUTt3Q0FDcEIsR0FBRyxFQUFFLFFBQVE7cUNBQ2Q7b0NBQ0Q7d0NBQ0UsVUFBVSxFQUFFLFFBQVE7d0NBQ3BCLEdBQUcsRUFBRSxRQUFRO3FDQUNkO2lDQU1GOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLFVBQVUsRUFBRSwrQkFBK0I7b0JBQzNDLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLFFBQVEsRUFBRTt3QkFDUixJQUFJLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjt3QkFDL0IsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO3FCQUMzQjtvQkFDRCxJQUFJLEVBQUU7d0JBQ0osUUFBUSxFQUFFLEtBQUs7d0JBQ2YsYUFBYSxFQUFFOzRCQUNiO2dDQUNFLFVBQVUsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2dDQUNyQyxJQUFJLEVBQUUsUUFBUTtnQ0FDZCxJQUFJLEVBQUU7b0NBQ0o7d0NBQ0UsVUFBVSxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7d0NBQ3JDLEdBQUcsRUFBRSxTQUFTO3FDQUNmO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNELFVBQVUsRUFBRTs0QkFDVixPQUFPLEVBQUUsa0JBQWtCLEtBQUssQ0FBQyxtQkFBbUIscUNBQXFDO3lCQU8xRjtxQkFFRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsaUJBQWlCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUVyRCxNQUFNLG1CQUFtQixHQUFHLElBQUkscUJBQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUNwRSxHQUFHLEtBQUssQ0FBQyxXQUFXO1lBQ3BCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixVQUFVLEVBQUUsMkJBQTJCO1lBQ3ZDLEtBQUssRUFBRSxNQUFNO1lBQ2IsT0FBTyxFQUFFLE1BQU07WUFDZixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDMUIsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLEVBQUUsSUFBSTtZQUNWLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLE1BQU0sRUFBRTtnQkFDTix1QkFBdUI7Z0JBQ3ZCLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2dCQUN4QixhQUFhO2dCQUNiLHdEQUF3RDtnQkFDeEQsMkJBQTJCO2dCQUMzQixPQUFPO2dCQUNQLEtBQUs7Z0JBQ0wsV0FBVztnQkFDWCxxQ0FBcUM7Z0JBQ3JDLDJCQUEyQjtnQkFDM0IsS0FBSztnQkFDTCxHQUFHLEVBQUU7b0JBQ0gsUUFBUSxFQUFFLFVBQVU7b0JBQ3BCLElBQUksRUFBRSxlQUFlO29CQUNyQixZQUFZLEVBQUUsS0FBSztvQkFDbkIsWUFBWSxFQUFFLG9EQUFvRDtvQkFDbEUsZ0JBQWdCLEVBQUUsb0RBQW9EO29CQUN0RSxlQUFlLEVBQUUsbURBQW1EO29CQUNwRSxPQUFPLEVBQUU7d0JBQ1AsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixJQUFJLEVBQUUsV0FBVztnQ0FDakIsR0FBRyxFQUFFLFlBQVk7NkJBQ2xCO3lCQUNGO3FCQUNGO29CQUNELFdBQVcsRUFBRTt3QkFDWCxTQUFTLEVBQUU7NEJBQ1QsWUFBWSxFQUFFO2dDQUNaLElBQUksRUFBRSxXQUFXO2dDQUNqQixHQUFHLEVBQUUsWUFBWTs2QkFDbEI7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLEdBQUcsRUFBRSxRQUFROzZCQUNkO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRTt3QkFDUCxTQUFTLEVBQUU7NEJBQ1QsWUFBWSxFQUFFO2dDQUNaLElBQUksRUFBRSxXQUFXO2dDQUNqQixHQUFHLEVBQUUsUUFBUTs2QkFDZDt5QkFDRjtxQkFDRjtpQkFTRjtnQkFDRCxPQUFPLEVBQUU7b0JBQ1AsT0FBTyxFQUFFLE1BQU07b0JBQ2YsSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLEdBQUcsRUFBRTt3QkFDSCxPQUFPLEVBQUUsTUFBTTt3QkFDZixXQUFXLEVBQUUsR0FBRztxQkFDakI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLG1EQUFtRCxFQUFFLEtBQUs7d0JBQzFELG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLGNBQWM7d0JBQy9GLDJDQUEyQyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVTt3QkFDdkUsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsZ0JBQWdCLEVBQUU7b0JBQ2hCLE9BQU8sRUFBRSxJQUFJO29CQUNiLElBQUksRUFBRSxjQUFjO29CQUNwQixHQUFHLEVBQUU7d0JBQ0gsT0FBTyxFQUFFLE1BQU07d0JBQ2YsV0FBVyxFQUFFLEdBQUc7cUJBQ2pCO29CQUNELFdBQVcsRUFBRTt3QkFDWCxtREFBbUQsRUFBRSxLQUFLO3dCQUMxRCwyQ0FBMkMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLFlBQVk7d0JBQ3pFLG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLGdCQUFnQjt3QkFDakcsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLE9BQU8sRUFBRSxLQUFLO2lCQUNmO2dCQUNELEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsSUFBSTtvQkFDYixJQUFJLEVBQUUsY0FBYztvQkFDcEIsTUFBTSxFQUFFO3dCQUNOLGdCQUFnQixFQUFFLElBQUk7cUJBQ3ZCO29CQUNELEdBQUcsRUFBRTt3QkFDSCxXQUFXLEVBQUUsR0FBRztxQkFDakI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLG1EQUFtRCxFQUFFLEtBQUs7d0JBQzFELHVFQUF1RSxFQUFFLFlBQVk7d0JBQ3JGLDJDQUEyQyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUTt3QkFDckUsbURBQW1ELEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsWUFBWTt3QkFDN0YsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUNWO29CQUNFLGNBQWMsRUFBRTt3QkFDZCxNQUFNLEVBQUUsS0FBSzt3QkFDYixJQUFJLEVBQUUsdUJBQXVCLENBQUMsa0JBQWtCO3FCQUNqRDtvQkFDRCxrQkFBa0IsRUFDbEI7d0JBQ0U7NEJBQ0UsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7NEJBQy9CLEdBQUcsRUFBRTtnQ0FDSCxNQUFNLEVBQUUsMEJBQTBCO2dDQUNsQyxRQUFRLEVBQUUsSUFBSTtnQ0FDZCxnQkFBZ0IsRUFBRTtvQ0FDaEIsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtpQ0FDL0M7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLFdBQVc7NEJBQ2pCLEdBQUcsRUFBRTtnQ0FDSCxNQUFNLEVBQUUsMEJBQTBCO2dDQUNsQyxRQUFRLEVBQUUsSUFBSTtnQ0FDZCxnQkFBZ0IsRUFBRTtvQ0FDaEIsbUJBQW1CLEVBQUUsMEJBQTBCO2lDQUNoRDs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCx1QkFBdUIsRUFBRTt3QkFDdkI7NEJBQ0UsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7NEJBQy9CLFNBQVMsRUFBRSxzQkFBc0I7NEJBQ2pDLFFBQVEsRUFBRSxJQUFJO3lCQUNmO3dCQUNEOzRCQUNFLElBQUksRUFBRSxXQUFXOzRCQUNqQixTQUFTLEVBQUUsa0JBQWtCOzRCQUM3QixRQUFRLEVBQUUsSUFBSTt5QkFDZjtxQkFDRjtpQkFDRjtnQkFDRCxVQUFVLEVBQUU7b0JBQ1YsT0FBTyxFQUFFLElBQUk7b0JBQ2IsY0FBYyxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7aUJBQzFDO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxPQUFPLEVBQUUsSUFBSTtpQkFDZDtnQkFDRCxPQUFPLEVBQUU7b0JBQ1AsT0FBTyxFQUFFLElBQUk7b0JBQ2IsSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLFdBQVcsRUFBRTt3QkFDWCxtREFBbUQsRUFBRSxLQUFLO3dCQUMxRCxtREFBbUQsRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxjQUFjO3dCQUMvRiwyQ0FBMkMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLFVBQVU7d0JBQ3ZFLGdGQUFnRixFQUFFLElBQUk7cUJBQ3ZGO2lCQUNGO2dCQUNELFVBQVUsRUFBRTtvQkFDVixPQUFPLEVBQUUsS0FBSztpQkFDZjtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsK0JBQStCO2lCQUNoQzthQUNGO1NBRUYsQ0FBQyxDQUFDO1FBRUgsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsRUFBRSxjQUFjLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztJQUNyRyxDQUFDO0NBQ0Y7QUExV0Qsa0RBMFdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24sIGF3c19pYW0sIGF3c19la3MsIGF3c19yZHMsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBEbnNQcm9wcyB9IGZyb20gJy4uLy4uL2ludGVyZmFjZXMvZG5zLXByb3BzJztcbi8vIGltcG9ydCB7IEtvbmdDb250cm9sUGxhbmUgfSBmcm9tICcuLi8uLi8uLi9Lb25nQ29udHJvbFBsYW5lJztcblxuLy8gZXhwb3J0IGludGVyZmFjZSBEbnNQcm9wcyB7XG4vLyAgIHJlYWRvbmx5IGFkbWluRG5zIDogc3RyaW5nO1xuLy8gICByZWFkb25seSBjbHVzdGVyRG5zIDogc3RyaW5nO1xuLy8gICByZWFkb25seSB0ZWxlbWV0cnlEbnMgOiBzdHJpbmc7XG4vLyAgIHJlYWRvbmx5IG1hbmFnZXJEbnMgOiBzdHJpbmc7XG4vLyB9XG5leHBvcnQgaW50ZXJmYWNlIEtvbmdDUFByb3BzIHtcbiAgcmVhZG9ubHkgY2x1c3RlciA6IGF3c19la3MuQ2x1c3RlcjtcbiAgcmVhZG9ubHkgcmRzOiBhd3NfcmRzLkRhdGFiYXNlSW5zdGFuY2U7XG4gIHJlYWRvbmx5IG5hbWVzcGFjZTogc3RyaW5nO1xuICAvLyByZWFkb25seSBub2RlZ3JvdXAgOiBhd3NfZWtzLk5vZGVncm91cDtcbiAgcmVhZG9ubHkgbGljZW5zZV9zZWNyZXRfbmFtZSA6IHN0cmluZztcbiAgcmVhZG9ubHkgY2FjZXJ0bmFtZSA6IFN0cmluZztcbiAgcmVhZG9ubHkgZW5kcG9pbnRzOiBEbnNQcm9wcztcbiAgLy8gcmVhZG9ubHkgaG9zdGVkWm9uZU5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgY2x1c3Rlcklzc3Vlck5hbWU6IFN0cmluZztcbiAgcmVhZG9ubHkgSGVsbU9wdGlvbnM/OiBhd3NfZWtzLkhlbG1DaGFydE9wdGlvbnM7XG59XG5cbmV4cG9ydCBjbGFzcyBLb25nQ29udHJvbFBsYW5lRUtTIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEtvbmdDUFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHJkc19zZWNyZXRzID0gJ3Jkcy1zZWNyZXRzJztcbiAgICBjb25zdCByZHNfc2VjcmV0c19wcm92aWRlcl9jbGFzcyA9ICdyZHMtc2VjcmV0cy1wcm92aWRlci1jbGFzcyc7XG5cbiAgICBjb25zdCBrb25nX25hbWVzcGFjZSA9IG5ldyBhd3NfZWtzLkt1YmVybmV0ZXNNYW5pZmVzdCh0aGlzLCAnS29uZ05hbWVzcGFjZScsIHtcbiAgICAgIGNsdXN0ZXI6IHByb3BzLmNsdXN0ZXIsXG4gICAgICBvdmVyd3JpdGU6IHRydWUsXG4gICAgICBtYW5pZmVzdDogW1xuICAgICAgICB7XG4gICAgICAgICAgYXBpVmVyc2lvbjogJ3YxJyxcbiAgICAgICAgICBraW5kOiAnTmFtZXNwYWNlJyxcbiAgICAgICAgICBtZXRhZGF0YTogeyBuYW1lOiBwcm9wcy5uYW1lc3BhY2UgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBrb25nX2NwX3NlcnZpY2VfYWNjb3VudCA9IG5ldyBhd3NfZWtzLlNlcnZpY2VBY2NvdW50KHRoaXMsICdLb25nQ3BTQScsIHtcbiAgICAgIGNsdXN0ZXI6IHByb3BzLmNsdXN0ZXIsXG4gICAgICBuYW1lOiAna29uZy1jcC1zZXJ2aWNlLWFjY291bnQnLFxuICAgICAgbmFtZXNwYWNlOiBwcm9wcy5uYW1lc3BhY2UsXG4gICAgfSk7XG5cbiAgICBjb25zdCBhd3NfcmVnaW9uID0gU3RhY2sub2YodGhpcykucmVnaW9uO1xuICAgIGNvbnN0IGF3c19hY2NvdW50ID0gU3RhY2sub2YodGhpcykuYWNjb3VudDtcblxuICAgIGtvbmdfY3Bfc2VydmljZV9hY2NvdW50LmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBhd3NfaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgYGFybjphd3M6c2VjcmV0c21hbmFnZXI6JHthd3NfcmVnaW9ufToke2F3c19hY2NvdW50fTpzZWNyZXQ6JHtwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lfS0/Pz8/Pz9gLFxuICAgICAgICBgJHtwcm9wcy5yZHMuc2VjcmV0Py5zZWNyZXRGdWxsQXJufWAsXG4gICAgICBdLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnc2VjcmV0c21hbmFnZXI6R2V0U2VjcmV0VmFsdWUnLFxuICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBrb25nX2NwX3NlcnZpY2VfYWNjb3VudC5ub2RlLmFkZERlcGVuZGVuY3koa29uZ19uYW1lc3BhY2UpO1xuXG4gICAgY29uc3Qga29uZ19jZXJ0aWZpY2F0ZSA9IG5ldyBhd3NfZWtzLkt1YmVybmV0ZXNNYW5pZmVzdCh0aGlzLCAnS29uZ0NlcnQnLCB7XG4gICAgICBjbHVzdGVyOiBwcm9wcy5jbHVzdGVyLFxuICAgICAgb3ZlcndyaXRlOiB0cnVlLFxuICAgICAgbWFuaWZlc3Q6IFtcbiAgICAgICAge1xuICAgICAgICAgIGFwaVZlcnNpb246ICdjZXJ0LW1hbmFnZXIuaW8vdjEnLFxuICAgICAgICAgIGtpbmQ6ICdDZXJ0aWZpY2F0ZScsXG4gICAgICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAgICAgIG5hbWU6IHByb3BzLmNhY2VydG5hbWUsXG4gICAgICAgICAgICBuYW1lc3BhY2U6IHByb3BzLm5hbWVzcGFjZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHNwZWM6IHtcbiAgICAgICAgICAgIGNvbW1vbk5hbWU6IHByb3BzLmVuZHBvaW50cy5ob3N0ZWRab25lTmFtZSxcbiAgICAgICAgICAgIGRuc05hbWVzOiBbXG4gICAgICAgICAgICAgIHByb3BzLmVuZHBvaW50cy5hZG1pbkRucyxcbiAgICAgICAgICAgICAgcHJvcHMuZW5kcG9pbnRzLmNsdXN0ZXJEbnMsXG4gICAgICAgICAgICAgIHByb3BzLmVuZHBvaW50cy5tYW5hZ2VyRG5zLFxuICAgICAgICAgICAgICBwcm9wcy5lbmRwb2ludHMudGVsZW1ldHJ5RG5zLFxuICAgICAgICAgICAgICBwcm9wcy5lbmRwb2ludHMuaG9zdGVkWm9uZU5hbWUsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgZHVyYXRpb246ICcyMTYwaDBtMHMnLFxuICAgICAgICAgICAgaXNzdWVyUmVmOiB7XG4gICAgICAgICAgICAgIGdyb3VwOiAnYXdzcGNhLmNlcnQtbWFuYWdlci5pbycsXG4gICAgICAgICAgICAgIGtpbmQ6ICdBV1NQQ0FDbHVzdGVySXNzdWVyJyxcbiAgICAgICAgICAgICAgbmFtZTogcHJvcHMuY2x1c3Rlcklzc3Vlck5hbWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgcmVuZXdCZWZvcmU6ICczNjBoMG0wcycsXG4gICAgICAgICAgICBzZWNyZXROYW1lOiBgJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0YCxcbiAgICAgICAgICAgIHVzYWdlczogW1xuICAgICAgICAgICAgICAnc2VydmVyIGF1dGgnLFxuICAgICAgICAgICAgICAnY2xpZW50IGF1dGgnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHByaXZhdGVLZXk6IHtcbiAgICAgICAgICAgICAgYWxnb3JpdGhtOiAnUlNBJyxcbiAgICAgICAgICAgICAgc2l6ZTogMjA0OCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBrb25nX2NlcnRpZmljYXRlLm5vZGUuYWRkRGVwZW5kZW5jeShrb25nX25hbWVzcGFjZSk7XG5cbiAgICAvLyBjb25zdCBzZWNyZXROYW1lUGFydHMgPSBGbi5zcGxpdCgnLScsIHByb3BzLnJkcy5zZWNyZXQ/LnNlY3JldE5hbWUhKTtcbiAgICAvLyBjb25zdCBzZWNyZXROYW1lV2l0aG91dFN1ZmZpeCA9IEZuLmpvaW4oJy0nLCBbRm4uc2VsZWN0KDAsIHNlY3JldE5hbWVQYXJ0cyksIEZuLnNlbGVjdCgxLCBzZWNyZXROYW1lUGFydHMpXSk7XG5cbiAgICBjb25zdCBzZWNyZXRzX2NyZF9tb3VudCA9IG5ldyBhd3NfZWtzLkt1YmVybmV0ZXNNYW5pZmVzdCh0aGlzLCAnU2VjcmV0c01vdW50Jywge1xuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIG92ZXJ3cml0ZTogdHJ1ZSxcbiAgICAgIG1hbmlmZXN0OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAnc2VjcmV0cy1zdG9yZS5jc2kueC1rOHMuaW8vdjEnLFxuICAgICAgICAgIGtpbmQ6ICdTZWNyZXRQcm92aWRlckNsYXNzJyxcbiAgICAgICAgICBtZXRhZGF0YToge1xuICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHNfcHJvdmlkZXJfY2xhc3MsXG4gICAgICAgICAgICBuYW1lc3BhY2U6IHByb3BzLm5hbWVzcGFjZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHNwZWM6IHtcbiAgICAgICAgICAgIHByb3ZpZGVyOiAnYXdzJyxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgb2JqZWN0czogYC0gb2JqZWN0TmFtZTogXCIke3Byb3BzLnJkcy5zZWNyZXQ/LnNlY3JldEZ1bGxBcm59XCJcXG4gIGptZXNQYXRoOlxcbiAgICAtIHBhdGg6IHVzZXJuYW1lXFxuICAgICAgb2JqZWN0QWxpYXM6IGRidXNlcm5hbWVcXG4gICAgLSBwYXRoOiBwYXNzd29yZFxcbiAgICAgIG9iamVjdEFsaWFzOiBkYnBhc3N3b3JkXFxuICAgIC0gcGF0aDogZGJuYW1lXFxuICAgICAgb2JqZWN0QWxpYXM6IGRibmFtZVxcbiAgICAtIHBhdGg6IGhvc3RcXG4gICAgICBvYmplY3RBbGlhczogZGJob3N0XFxuYCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBzZWNyZXRPYmplY3RzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBzZWNyZXROYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgICB0eXBlOiAnT3BhcXVlJyxcbiAgICAgICAgICAgICAgICBkYXRhOiBbXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6ICdkYnVzZXJuYW1lJyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJ1c2VybmFtZScsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBvYmplY3ROYW1lOiAnZGJwYXNzd29yZCcsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2RicGFzc3dvcmQnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgb2JqZWN0TmFtZTogJ2RibmFtZScsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2RibmFtZScsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBvYmplY3ROYW1lOiAnZGJob3N0JyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJob3N0JyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAvLyBLb25nIGRvZXMgbm90IHN1cHBvcnRzIHBvcnQgb2JmdXNjYXRpb25cbiAgICAgICAgICAgICAgICAgIC8vIHtcbiAgICAgICAgICAgICAgICAgIC8vICAgb2JqZWN0TmFtZTogJ2RicG9ydCcsXG4gICAgICAgICAgICAgICAgICAvLyAgIGtleTogJ2RicG9ydCcsXG4gICAgICAgICAgICAgICAgICAvLyB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAnc2VjcmV0cy1zdG9yZS5jc2kueC1rOHMuaW8vdjEnLFxuICAgICAgICAgIGtpbmQ6ICdTZWNyZXRQcm92aWRlckNsYXNzJyxcbiAgICAgICAgICBtZXRhZGF0YToge1xuICAgICAgICAgICAgbmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgIG5hbWVzcGFjZTogcHJvcHMubmFtZXNwYWNlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3BlYzoge1xuICAgICAgICAgICAgcHJvdmlkZXI6ICdhd3MnLFxuICAgICAgICAgICAgc2VjcmV0T2JqZWN0czogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgc2VjcmV0TmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgICB0eXBlOiAnb3BhcXVlJyxcbiAgICAgICAgICAgICAgICBkYXRhOiBbXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgICAgICAgIGtleTogJ2xpY2Vuc2UnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgb2JqZWN0czogYC0gb2JqZWN0TmFtZTogXCIke3Byb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWV9XCJcXG4gIG9iamVjdFR5cGU6IFwic2VjcmV0c21hbmFnZXJcIlxcbmAsXG4gICAgICAgICAgICAgIC8vIFtcbiAgICAgICAgICAgICAgLy8gICB7XG4gICAgICAgICAgICAgIC8vICAgICBvYmplY3ROYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICAvLyAgICAgb2JqZWN0VHlwZTogJ3NlY3JldHNtYW5hZ2VyJyxcbiAgICAgICAgICAgICAgLy8gICB9LFxuICAgICAgICAgICAgICAvLyBdLFxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgc2VjcmV0c19jcmRfbW91bnQubm9kZS5hZGREZXBlbmRlbmN5KGtvbmdfbmFtZXNwYWNlKTtcblxuICAgIGNvbnN0IGRlcGxveV9rb25nX2NwX2hlbG0gPSBuZXcgYXdzX2Vrcy5IZWxtQ2hhcnQodGhpcywgJ0tvbmdDcEhlbG0nLCB7XG4gICAgICAuLi5wcm9wcy5IZWxtT3B0aW9ucyxcbiAgICAgIGNsdXN0ZXI6IHByb3BzLmNsdXN0ZXIsXG4gICAgICByZXBvc2l0b3J5OiAnaHR0cHM6Ly9jaGFydHMua29uZ2hxLmNvbScsXG4gICAgICBjaGFydDogJ2tvbmcnLFxuICAgICAgcmVsZWFzZTogJ2tvbmcnLFxuICAgICAgbmFtZXNwYWNlOiBwcm9wcy5uYW1lc3BhY2UsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHdhaXQ6IHRydWUsXG4gICAgICBjcmVhdGVOYW1lc3BhY2U6IHRydWUsXG4gICAgICB2YWx1ZXM6IHtcbiAgICAgICAgLy8gaW5ncmVzc0NvbnRyb2xsZXI6IHtcbiAgICAgICAgLy8gICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAvLyAgIGluc3RhbGxDUkRzOiBmYWxzZSxcbiAgICAgICAgLy8gICBpbWFnZToge1xuICAgICAgICAvLyAgICAgcmVwb3NpdG9yeTogJ2tvbmcva3ViZXJuZXRlcy1pbmdyZXNzLWNvbnRyb2xsZXInLFxuICAgICAgICAvLyAgICAgdGFnOiAnMS4zLjEtYWxwaW5lJyxcbiAgICAgICAgLy8gICB9LFxuICAgICAgICAvLyB9LFxuICAgICAgICAvLyBpbWFnZToge1xuICAgICAgICAvLyAgIHJlcG9zaXRvcnk6ICdrb25nL2tvbmctZ2F0ZXdheScsXG4gICAgICAgIC8vICAgdGFnOiAnMi41LjAuMC1hbHBpbmUnLFxuICAgICAgICAvLyB9LFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBkYXRhYmFzZTogJ3Bvc3RncmVzJyxcbiAgICAgICAgICByb2xlOiAnY29udHJvbF9wbGFuZScsXG4gICAgICAgICAgY2x1c3Rlcl9tdGxzOiAncGtpJyxcbiAgICAgICAgICBjbHVzdGVyX2NlcnQ6ICcvZXRjL3NlY3JldHMva29uZy1jcC1jbHVzdGVyLWlzc3Vlci1zZWNyZXQvdGxzLmNydCcsXG4gICAgICAgICAgY2x1c3Rlcl9jZXJ0X2tleTogJy9ldGMvc2VjcmV0cy9rb25nLWNwLWNsdXN0ZXItaXNzdWVyLXNlY3JldC90bHMua2V5JyxcbiAgICAgICAgICBjbHVzdGVyX2NhX2NlcnQ6ICcvZXRjL3NlY3JldHMva29uZy1jcC1jbHVzdGVyLWlzc3Vlci1zZWNyZXQvY2EuY3J0JyxcbiAgICAgICAgICBwZ191c2VyOiB7XG4gICAgICAgICAgICB2YWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgc2VjcmV0S2V5UmVmOiB7XG4gICAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgICAga2V5OiAnZGJ1c2VybmFtZScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgcGdfcGFzc3dvcmQ6IHtcbiAgICAgICAgICAgIHZhbHVlRnJvbToge1xuICAgICAgICAgICAgICBzZWNyZXRLZXlSZWY6IHtcbiAgICAgICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgICBrZXk6ICdkYnBhc3N3b3JkJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwZ19kYXRhYmFzZToge1xuICAgICAgICAgICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIGtleTogJ2RibmFtZScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgcGdfaG9zdDoge1xuICAgICAgICAgICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIGtleTogJ2RiaG9zdCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgLy8gcGdfcG9ydDoge1xuICAgICAgICAgIC8vICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgLy8gICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgIC8vICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgIC8vICAgICAgIGtleTogJ2RicG9ydCcsXG4gICAgICAgICAgLy8gICAgIH0sXG4gICAgICAgICAgLy8gICB9LFxuICAgICAgICAgIC8vIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGNsdXN0ZXI6IHtcbiAgICAgICAgICBlbmFibGVkOiAndHJ1ZScsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBlbmFibGVkOiAndHJ1ZScsXG4gICAgICAgICAgICBzZXJ2aWNlUG9ydDogNDQzLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYW5ub3RhdGlvbnM6IHtcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci10eXBlJzogJ25sYicsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmFtZSc6IGAke3Byb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWV9LWVrcy1jbHVzdGVyYCxcbiAgICAgICAgICAgICdleHRlcm5hbC1kbnMuYWxwaGEua3ViZXJuZXRlcy5pby9ob3N0bmFtZSc6IHByb3BzLmVuZHBvaW50cy5jbHVzdGVyRG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLWNyb3NzLXpvbmUtbG9hZC1iYWxhbmNpbmctZW5hYmxlZCc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgY2x1c3RlcnRlbGVtZXRyeToge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBlbmFibGVkOiAndHJ1ZScsXG4gICAgICAgICAgICBzZXJ2aWNlUG9ydDogNDQzLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYW5ub3RhdGlvbnM6IHtcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci10eXBlJzogJ25sYicsXG4gICAgICAgICAgICAnZXh0ZXJuYWwtZG5zLmFscGhhLmt1YmVybmV0ZXMuaW8vaG9zdG5hbWUnOiBwcm9wcy5lbmRwb2ludHMudGVsZW1ldHJ5RG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5hbWUnOiBgJHtwcm9wcy5jbHVzdGVyLmNsdXN0ZXJOYW1lfS1la3MtdGVsZW1ldHJ5YCxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1jcm9zcy16b25lLWxvYWQtYmFsYW5jaW5nLWVuYWJsZWQnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHByb3h5OiB7XG4gICAgICAgICAgZW5hYmxlZDogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICAgIGFkbWluOiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICB0eXBlOiAnTG9hZEJhbGFuY2VyJyxcbiAgICAgICAgICBsYWJlbHM6IHtcbiAgICAgICAgICAgICdlbmFibGUtbWV0cmljcyc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB0bHM6IHtcbiAgICAgICAgICAgIHNlcnZpY2VQb3J0OiA0NDMsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXR5cGUnOiAnbmxiJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1hZGRpdGlvbmFsLXJlc291cmNlLXRhZ3MnOiAnVHlwZT1hZG1pbicsXG4gICAgICAgICAgICAnZXh0ZXJuYWwtZG5zLmFscGhhLmt1YmVybmV0ZXMuaW8vaG9zdG5hbWUnOiBwcm9wcy5lbmRwb2ludHMuYWRtaW5EbnMsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmFtZSc6IGAke3Byb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWV9LWVrcy1hZG1pbmAsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItY3Jvc3Mtem9uZS1sb2FkLWJhbGFuY2luZy1lbmFibGVkJzogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBkZXBsb3ltZW50OlxuICAgICAgICB7XG4gICAgICAgICAgc2VydmljZUFjY291bnQ6IHtcbiAgICAgICAgICAgIGNyZWF0ZTogZmFsc2UsXG4gICAgICAgICAgICBuYW1lOiBrb25nX2NwX3NlcnZpY2VfYWNjb3VudC5zZXJ2aWNlQWNjb3VudE5hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB1c2VyRGVmaW5lZFZvbHVtZXM6XG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBuYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICBjc2k6IHtcbiAgICAgICAgICAgICAgICBkcml2ZXI6ICdzZWNyZXRzLXN0b3JlLmNzaS5rOHMuaW8nLFxuICAgICAgICAgICAgICAgIHJlYWRPbmx5OiB0cnVlLFxuICAgICAgICAgICAgICAgIHZvbHVtZUF0dHJpYnV0ZXM6IHtcbiAgICAgICAgICAgICAgICAgIHNlY3JldFByb3ZpZGVyQ2xhc3M6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICBjc2k6IHtcbiAgICAgICAgICAgICAgICBkcml2ZXI6ICdzZWNyZXRzLXN0b3JlLmNzaS5rOHMuaW8nLFxuICAgICAgICAgICAgICAgIHJlYWRPbmx5OiB0cnVlLFxuICAgICAgICAgICAgICAgIHZvbHVtZUF0dHJpYnV0ZXM6IHtcbiAgICAgICAgICAgICAgICAgIHNlY3JldFByb3ZpZGVyQ2xhc3M6IHJkc19zZWNyZXRzX3Byb3ZpZGVyX2NsYXNzLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgdXNlckRlZmluZWRWb2x1bWVNb3VudHM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgbW91bnRQYXRoOiAnL21udC9saWNlbnNlX3NlY3JldHMnLFxuICAgICAgICAgICAgICByZWFkT25seTogdHJ1ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICBtb3VudFBhdGg6ICcvbW50L3Jkc19zZWNyZXRzJyxcbiAgICAgICAgICAgICAgcmVhZE9ubHk6IHRydWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIGVudGVycHJpc2U6IHtcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgIGxpY2Vuc2Vfc2VjcmV0OiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICB9LFxuICAgICAgICBzZXJ2aWNlTW9uaXRvcjoge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIG1hbmFnZXI6IHtcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgIHR5cGU6ICdMb2FkQmFsYW5jZXInLFxuICAgICAgICAgIGFubm90YXRpb25zOiB7XG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItdHlwZSc6ICdubGInLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5hbWUnOiBgJHtwcm9wcy5jbHVzdGVyLmNsdXN0ZXJOYW1lfS1la3MtbWFuYWdlcmAsXG4gICAgICAgICAgICAnZXh0ZXJuYWwtZG5zLmFscGhhLmt1YmVybmV0ZXMuaW8vaG9zdG5hbWUnOiBwcm9wcy5lbmRwb2ludHMubWFuYWdlckRucyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1jcm9zcy16b25lLWxvYWQtYmFsYW5jaW5nLWVuYWJsZWQnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHBvc3RncmVzcWw6IHtcbiAgICAgICAgICBlbmFibGVkOiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2VjcmV0Vm9sdW1lczogW1xuICAgICAgICAgICdrb25nLWNwLWNsdXN0ZXItaXNzdWVyLXNlY3JldCcsXG4gICAgICAgIF0sXG4gICAgICB9LFxuXG4gICAgfSk7XG5cbiAgICBkZXBsb3lfa29uZ19jcF9oZWxtLm5vZGUuYWRkRGVwZW5kZW5jeShzZWNyZXRzX2NyZF9tb3VudCwga29uZ19uYW1lc3BhY2UsIGtvbmdfY3Bfc2VydmljZV9hY2NvdW50KTtcbiAgfVxufSJdfQ==