# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['benchling_api_client',
 'benchling_api_client.api',
 'benchling_api_client.api.aa_sequences',
 'benchling_api_client.api.assay_results',
 'benchling_api_client.api.assay_runs',
 'benchling_api_client.api.batches',
 'benchling_api_client.api.blobs',
 'benchling_api_client.api.boxes',
 'benchling_api_client.api.containers',
 'benchling_api_client.api.custom_entities',
 'benchling_api_client.api.dna_alignments',
 'benchling_api_client.api.dna_oligos',
 'benchling_api_client.api.dna_sequences',
 'benchling_api_client.api.dropdowns',
 'benchling_api_client.api.entries',
 'benchling_api_client.api.events',
 'benchling_api_client.api.exports',
 'benchling_api_client.api.feature_libraries',
 'benchling_api_client.api.folders',
 'benchling_api_client.api.inventory',
 'benchling_api_client.api.lab_automation',
 'benchling_api_client.api.label_templates',
 'benchling_api_client.api.legacy_workflows',
 'benchling_api_client.api.locations',
 'benchling_api_client.api.mixtures',
 'benchling_api_client.api.oligos',
 'benchling_api_client.api.plates',
 'benchling_api_client.api.printers',
 'benchling_api_client.api.projects',
 'benchling_api_client.api.registry',
 'benchling_api_client.api.requests',
 'benchling_api_client.api.rna_oligos',
 'benchling_api_client.api.schemas',
 'benchling_api_client.api.tasks',
 'benchling_api_client.api.warehouse',
 'benchling_api_client.api.workflows',
 'benchling_api_client.models',
 'benchling_sdk',
 'benchling_sdk.auth',
 'benchling_sdk.benchbots',
 'benchling_sdk.benchbots.codegen',
 'benchling_sdk.benchbots.commands',
 'benchling_sdk.benchbots.dependencies',
 'benchling_sdk.benchbots.helpers',
 'benchling_sdk.benchbots.types',
 'benchling_sdk.helpers',
 'benchling_sdk.models',
 'benchling_sdk.services']

package_data = \
{'': ['*'],
 'benchling_sdk': ['codegen/templates/*',
                   'codegen/templates/property_templates/*'],
 'benchling_sdk.benchbots.codegen': ['templates/*']}

install_requires = \
['Jinja2>=2.11.3,<3.0.0',
 'PyYAML>=5.4.1,<6.0.0',
 'attrs>=20.1.0,<21.0.0',
 'backoff>=1.10.0,<2.0.0',
 'black==20.8b1',
 'dataclasses-json>=0.5.2,<0.6.0',
 'dataclasses-jsonschema>=2.14.1,<3.0.0',
 'httpx>=0.15.0,<0.16.0',
 'jsonpath-ng>=1.5.2,<2.0.0',
 'keyring==22.0.1',
 'python-dateutil>=2.8.0,<3.0.0',
 'typer>=0.3.2,<0.4.0',
 'typing-extensions>=3.7.4,<4.0.0']

entry_points = \
{'console_scripts': ['benchling-sdk = benchling_sdk.cli:cli']}

setup_kwargs = {
    'name': 'ce-bnch-sdk',
    'version': '1.0.0b0',
    'description': 'SDK for interacting with the Benchling Platform.',
    'long_description': '# Benchling SDK\n\nA Python 3.7+ SDK for the [Benchling](https://www.benchling.com/) platform designed to provide typed, fluent\ninteractions with [Benchling APIs](https://docs.benchling.com/reference).\n\n*Important!* This is a pre-release alpha. Changes to this project may not be backwards compatible.\n\n## Getting Started\n\n### Installation\n\nInstall the dependency via [Poetry](https://python-poetry.org/) (if applicable):\n\n```bash\npoetry add benchling-sdk\n```\n \nOr [Pip](https://pypi.org/project/pip/):\n \n```bash\npip install benchling-sdk\n```\n\n### Using the SDK\n\nObtain a valid API key from your Benchling account and provide it to the SDK, along with the URL for the server.\nExample:\n\n```python\nfrom benchling_sdk.benchling import Benchling\nbenchling = Benchling(url="https://my.benchling.com", api_key="api_key")\n```\n\nWith `Benchling` now instantiated, make a sample call to get a custom entity with the ID `"custom_id"`.\n\n```python\nentity = benchling.custom_entities.get_by_id(entity_id="custom_id")\n```\n\nAPI calls made by the SDK are synchronous and blocking.\n\n### Generators and nextToken\n\nPaginated API endpoints listing objects and supporting the `nextToken` parameter will produce a `PageIterator`, which\nis a [Python generator](https://wiki.python.org/moin/Generators). Example:\n\n```python\nrequests_generator = benchling.requests.list(schema_id="assaych_test")\nnext_request = next(requests_generator)\n```\n\nIn this example, `requests_generator` is a generator. Each iteration will return a `List` of `Request`s, not an \nindividual `Request`.\n\nThe `PageIterator` object has an `estimated_count()` which will return the value of the `Result-Count` header from\nthe API, if applicable for the endpoint. If the endpoint in question does not support the header, will raise a \n`NotImplementedError` instead.\n\n`PageIterator` also supports `first()` which will return an optional first element in the list. Using DNA Sequences as\nan example, if at least one DNA sequence was returned by the results list, `first()` will return the first DNA sequence.\n\n`first()` operates independent of iteration, meaning calls to `first()` after starting iteration will still \nreturn the first item from the first page and not the current page. \n\nIf no results were available, `first()` will return `None`.\n\n```python\nfrom typing import Optional\nfrom benchling_sdk.models import DnaSequence\n\ndna_sequences_generator = benchling.dna_sequences.list()\n\n# Typing is optional to illustrate the expected return\nfirst_sequence: Optional[DnaSequence] = dna_sequences_generator.first()\nif first_sequence:\n   print(f"The first sequence was {first_sequence.id}")\nelse:\n   print("No sequence found")\n```\n\n### Working with Benchling Fields\n\nMany objects in Benchling have the concept of `fields`. They are represented in the SDK via the \n`benchling_sdk.models.Fields` class.\n\nTo conveniently construct `Fields` from a dictionary, we have provided a `fields` method \nin the `serialization_helper` module:\n\n```python\nfrom benchling_sdk.helpers.serialization_helpers import fields\nfrom benchling_sdk.models import CustomEntity\n\nentity = CustomEntity(\n    name="My Entity",\n    fields=fields({\n    "a_field": {"value": "First value"},\n    "second_field": {"value": "Second value"},\n    })\n)\n```\n\n### Async Tasks\n\nMany Benchling endpoints that perform expensive operations launch [Tasks](https://docs.benchling.com/reference#tasks).\nThese are modeled in the SDK as `benchling_sdk.models.AsyncTask`.\n\nTo simply retrieve the status of a task given its id:\n\n```python\nasync_task = benchling.tasks.get_by_id("task_id")\n```\n\nThis will return the `AsyncTask` object, which may still be in progress. More commonly, it may be desirable to delay\nfurther execution until a task is completed.\n\nIn this case, you may block further processing until the task is no longer `RUNNING`:\n\n```python\ncompleted_task = benchling.tasks.wait_for_task("task_id")\n```\n\nThe `wait_for_task` method will return the task once its status is no longer `RUNNING`. This does not guarantee\nthe task executed successfully (see `benchling_sdk.models.AsyncTaskStatus`), only that \nBenchling considers it complete.\n\n`wait_for_task` can be configured by optionally specifying `interval_wait_seconds` for the time to wait between calls and \n`max_wait_seconds` which is the maximum number of seconds before `wait_for_task` will give up and raise \n`benchling_sdk.errors.WaitForTaskExpiredError`.\n\n```python\n# Check the task status once every 2 seconds for up to 60 seconds\ncompleted_task = benchling.tasks.wait_for_task(task_id="task_id", interval_wait_seconds=2, max_wait_seconds=60)\n```\n\n### Unset\n\nThe Benchling SDK uses the type `benchling_api_client.types.Unset` and the constant value \n`benchling_api_client.types.UNSET` to represent values that were not present in an interaction with the API. This is to\ndistinguish from values that were explicitly set to `None` from those that were simply unspecified.\n\nA common example might be updating only specific properties of an object:\n\n```python\nfrom benchling_sdk.models import CustomEntityUpdate\n\nupdate = CustomEntityUpdate(name="New name")\n\nupdated_entity = benchling.custom_entities.update(\n    entity_id="entity_id", entity=update\n)\n```\n\nAll other properties of `CustomEntityUpdate` besides `name` will default to `UNSET` and not be sent with the update. Setting any\noptional property to `None` will send a `null` JSON value. In general, you should not need to set `UNSET` directly.\n\nWhen receiving objects from the API, some of their fields may be `Unset`. The SDK will raise a\n`benchling_api_client.extensions.NotPresentError` if a field which is Unset is accessed, so that type hinting always reflects the\ntype of the field without needing to Union with the Unset type.  When constructing objects, if you need to set a field to Unset\nafter its construction, properties which are optional support the python `del` keyword, e.g.:\n\n```python\nfrom benchling_sdk.models import CustomEntityUpdate\n\nupdate = CustomEntityUpdate(name="New name", folder_id="folder_id")\ndel update.folder_id\n```\n\nIf the property cannot be Unset but `del` is used on it, Python will raise with `AttributeError: can\'t delete attribute`.\n\nIf you happen to have an instance of Unset that you\'d like to treat equivalent to `Optional[T]`, you can use the convenience function `unset_as_none()`:\n\n```python\nfrom benchling_sdk.helpers.serialization_helpers import unset_as_none\n\nsample_value: Union[Unset, None, int] = UNSET\n\noptional_value = unset_as_none(sample_value)\n# optional_value will be None\n```\n\n### Error Handling\n\nFailed API interactions will generally return a `BenchlingError`, which will contain some underlying\ninformation on the HTTP response such as the status. Example:\n\n```python\nfrom benchling_sdk.errors import BenchlingError\n\ntry:\n    requests = benchling.requests.get_by_id("request_id")\nexcept BenchlingError as error:\n    print(error.status_code)\n```\n\nIf an HTTP error code is not returned to the SDK or deserialization fails, an unbounded `Exception` \ncould be raised instead.\n\n### Advanced Use Cases\n\nBy default, the Benchling SDK is designed to be opinionated towards most common usage. There is some more \nadvanced configuration available for use cases which require it.\n\n### Retries\n\nThe SDK will automatically retry certain HTTP calls when the calls fail and certain conditions are met.\n\nThe default strategy is to retry calls failing with HTTP status codes `429`, `502`, `503`, and `504`. The rationale for\nthese status codes being retried is that many times they are indicative of a temporary network failure or exceeding the\nrate limit and may be successful upon retry.\n\nRetries will be attempted up to 5 times, with an exponential time delay backoff between calls.\n\nTo disable retries, specify `None` for `retry_strategy` when constructing `Benchling`:\n\n```python\nbenchling = Benchling(url="https://my.benchling.com", api_key="api_key", retry_strategy=None)\n```\n\nAlternatively, instantiate your own `benchling_sdk.retry_helpers.RetryStrategy` to further customize retry behavior.\n\n### BenchlingApiClient Customization (e.g., HTTP Timeout Settings)\n\nWhile the SDK abstracts most of the HTTP transport layer, access can still be granted via the `BenchlingApiClient`. A\ncommon use case might be extending HTTP timeouts for all calls.\n\nThis can be achieved by specifying a function which accepts a default configured instance of `BenchlingApiClient` and\nreturns a mutated instance with the desired changes.\n\nFor example, to set the HTTP timeout to 180 seconds:\n\n```python\nfrom benchling_api_client.benchling_client import BenchlingApiClient\n\ndef higher_timeout_client(client: BenchlingApiClient) -> BenchlingApiClient:\n    return client.with_timeout(180)\n\n\nbenchling = Benchling(\n    url="https://my.benchling.com",\n    api_key="api_key",\n    client_decorator=higher_timeout_client,\n)\n```\n\nTo fully customize the `BenchlingApiClient` and ignore default settings, construct your own instance in lieu of \nmodifying the `client` argument.\n\n#### Changing the Base URL\n\nWhen instantiating `Benchling`, the path `/api/v2` will automatically be appended to the server information provided.\n\nFor example, if creating `Benchling` like this:\n\n```python\nbenchling = Benchling(url="https://my.benchling.com", api_key="api_key")\n```\n\nAPI calls will be made to `https://my.benchling.com/api/v2`.\n\nTo specify, an alternative path, set the `base_path` when creating `Benchling`:\n\n```python\nbenchling = Benchling(url="https://my.benchling.com", api_key="api_key", base_path="/api/custom")\n```\n\nIn this case, API calls will be made to `https://my.benchling.com/api/custom`.\n\n### Custom API Calls\n\nFor making customized API calls to Benchling, the SDK supports an open-ended `Benchling.api` namespace that exposes\nvarying levels of interaction for HTTP `GET`, `POST`, `PATCH`, and `DELETE` methods. This is useful for API endpoints\nwhich the SDK may not support yet or more granular control at the HTTP layer.\n\nFor each verb, there are two related methods. Using `GET` as an example:\n\n1. `get_response()` - Returns a `benchling_api_client.types.Response` which has been parsed to a JSON `dict` \n   and is slightly more structured.\n2. `get_modeled()` - Returns any custom model which extends `benchling_sdk.helpers.serialization_helpers.DeserializableModel`\n   and must be a Python `@dataclass`.\n\nBoth will automatically retry failures according to `RetryStrategy` and will marshall errors to `BenchlingError`.\n\nWhen calling any of the methods in `Benchling.api`, specify the **full path** to the URL except for the scheme and server. \nThis differs from other API services, which will prepend the URL with a `base_path`.\n\nFor example, if wishing to call an endpoint `https://my.benchling.com/api/v2/custom-entities?some=param`, \npass `api/v2/custom-entities?some=param` for the `url`.\n\nHere\'s an example of making a custom call with `post_modeled()`:\n\n```python\nfrom dataclasses import dataclass, field\nfrom typing import Any, Dict\nfrom dataclasses_json import config\nfrom benchling_sdk.helpers.serialization_helpers import DeserializableModel, SerializableModel\n\n@dataclass\nclass ModeledCustomEntityPost(SerializableModel):\n    name: str\n    fields: Dict[str, Any]\n    # If the property name in the API JSON payload does not match the Python attribute, use\n    # field and config to specify the appropriate name for serializing/deserializing\n    folder_id: str = field(metadata=config(field_name="folderId"))\n    schema_id: str = field(metadata=config(field_name="schemaId"))\n\n\n@dataclass\nclass ModeledCustomEntityGet(DeserializableModel):\n    id: str\n    name: str\n    fields: Dict[str, Any]\n    folder_id: str = field(metadata=config(field_name="folderId"))\n\n# Assumes `benchling` is already configured and instantiated as `Benchling`\nbody = ModeledCustomEntityPost(\n    name="My Custom Entity Model",\n    folder_id="folder_id",\n    schema_id="schema_id",\n    fields={"My Field": {"value": "Modeled Entity"}},\n)\n\ncreated_entity = benchling.api.post_modeled(\n    url="api/v2/custom-entities", body=body, target_type=ModeledCustomEntityGet\n)\n```\n\nThe returned `created_entity` will be of type `ModeledCustomEntityGet`. Classes extending `SerializableModel` and\n`DeserializableModel` will inherit `serialize()` and `deserialize()` methods respectively which will act on Python \nclass attributes by default. These can be overridden for more customized serialization needs.\n',
    'author': 'Benchling Customer Engineering',
    'author_email': 'ce-team@benchling.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
