// Copyright (c) Mito
import React, { useState } from 'react';
import { ModalEnum } from './modals';
import BlueMitoFolk from '../icons/mitofolks/BlueMitoFolk';
import PinkMitoFolk from '../icons/mitofolks/PinkMitoFolk';
import YellowMitoFolk from '../icons/mitofolks/YellowMitoFolk';
import '../../../css/signup-modal.css';
import TextButton from '../elements/TextButton';
import Input from '../elements/Input';
/*
    This file contains all the screens used in the signup modal. As these
    are only used in this one file, we keep them together for cleanlyness.
*/
// If the user signs up for pro, this is the access code they must put in
const ACCESS_CODE = 'mito-pro-access-code-UEKXPTTQECAULCMW';
// The first question we ask on the signup page
const FirstQuestion = 'Your Company/Organization';
const FirstPlaceholder = 'AppleSoftBook';
/* Step one requires an email input */
const StepOne = (props) => {
    const onSubmit = async (e) => {
        e.preventDefault();
        await props.mitoAPI.updateSignUp(props.email);
        props.next();
    };
    return (React.createElement("div", { className: 'signup-modal-left-column' },
        React.createElement("div", null,
            React.createElement("h1", { className: 'mt-0' }, "Sign Up for Mito"),
            React.createElement("p", { className: 'signup-modal-text' }, "We\u2019ll send you periodic product updates and welcome any feedback. And no spam. Duh.")),
        React.createElement("form", { className: 'signup-modal-email-form', onSubmit: onSubmit },
            React.createElement("h3", { className: 'mb-5px' }, "Your Email"),
            React.createElement(Input, { value: props.email, onChange: (event) => { props.setEmail(event.target.value); }, type: 'email', width: 'large', placeholder: 'example@gmail.com', required: true, autoFocus: true }),
            React.createElement("label", null,
                React.createElement("h3", { className: 'mt-10px mb-5px' }, FirstQuestion)),
            React.createElement(Input, { value: props.firstResponse, onChange: (e) => { props.setFirstResponse(e.target.value); }, placeholder: FirstPlaceholder, required: true }),
            React.createElement("div", { className: 'mt-10px' },
                React.createElement(TextButton, { variant: 'dark', width: 'large', type: 'submit' }, "Sign Up")))));
};
/**
 * Step one requires the user to read the privacy policy, and optionally
 * gives the user the chance to sign up for Mito pro.
 *
 * We make this part of step two rather than a whole new step in it's own
 * right as it is branches, and we don't want to deal with that complexity
 * in the main signup flow.
*/
const StepTwo = (props) => {
    // We cache isPro so that if the user signs up for pro we can
    // update this immediately with no lag
    const [isPro, setIsPro] = useState(props.isPro);
    const [enteringProAccessCode, setEnteringProAccessCode] = useState(false);
    const [accessCode, setAccessCode] = useState('');
    const [invalidAccessCode, setInvalidAccessCode] = useState(false);
    const attemptSubmitAccessCode = () => {
        if (accessCode !== ACCESS_CODE) {
            setInvalidAccessCode(true);
            return;
        }
        setInvalidAccessCode(false);
        setEnteringProAccessCode(false);
        setIsPro(true);
        // We log this before going pro so that this is the last thing to appear in the logs
        void props.mitoAPI.log('signup_completed_pro');
        void props.mitoAPI.updateGoPro();
        // Then, we go to the final page
        props.next();
    };
    return (React.createElement(React.Fragment, null,
        !enteringProAccessCode &&
            React.createElement("div", { className: 'signup-modal-left-column' },
                React.createElement("div", null,
                    React.createElement("h1", { className: 'mt-0' }, isPro
                        ? "You've Signed up for Mito Pro!"
                        : "Want More Power? Consider Mito Pro"),
                    React.createElement("p", { className: 'signup-modal-text' }, isPro
                        ? "Thanks for being a Mito Pro user! Paying for Mito gets you access to advanced functionality and turns off telemetry. In turn, it allows us to fund Mito's development."
                        : "Mito Pro gives you access to advanced functionality, and allows you to turn off telemetry. It also allows us to continue to fund Mito's development!")),
                !isPro &&
                    React.createElement("div", { className: 'signup-modal-buttons mt-10px' },
                        React.createElement(TextButton, { variant: 'light', width: 'small', onClick: props.back }, "Back"),
                        React.createElement(TextButton, { variant: 'light', width: 'small', href: 'https://trymito.io/plans', target: '_blank', onClick: () => {
                                setEnteringProAccessCode(true);
                                void props.mitoAPI.log('signup_clicked_pro');
                            } }, "See Plans"),
                        React.createElement(TextButton, { variant: 'dark', width: 'small', onClick: props.next, autoFocus: true }, "No Thanks")),
                isPro &&
                    React.createElement("div", { className: 'signup-modal-buttons mt-10px' },
                        React.createElement(TextButton, { variant: 'light', width: 'small', onClick: props.back }, "Back"),
                        React.createElement(TextButton, { variant: 'dark', width: 'small', onClick: props.next, autoFocus: true }, "Continue"))),
        enteringProAccessCode &&
            React.createElement("div", { className: 'signup-modal-left-column' },
                React.createElement("div", null,
                    React.createElement("h1", { className: 'mt-0' }, "Complete Pro Signup"),
                    React.createElement("p", { className: 'signup-modal-text' },
                        "Complete the checkout flow. In the Pro documentation, click ",
                        React.createElement("b", null, "Get Access Code"),
                        " and enter it here."),
                    React.createElement("label", null,
                        React.createElement("h3", { className: 'mt-5px mb-10px' }, "Access Code:")),
                    React.createElement(Input, { placeholder: 'mito-pro-access-code-XXXXXXXXXXXX', value: accessCode, onChange: (e) => { setAccessCode(e.target.value); }, onKeyDown: (e) => {
                            if (e.key === 'Enter') {
                                attemptSubmitAccessCode();
                            }
                        } }),
                    invalidAccessCode &&
                        React.createElement("p", { className: 'signup-modal-text signup-modal-text-error' }, "Invalid access code")),
                React.createElement("div", { className: 'signup-modal-buttons mt-10px' },
                    React.createElement(TextButton, { variant: 'light', width: 'small', onClick: () => {
                            setEnteringProAccessCode(false);
                        } }, "Back"),
                    React.createElement(TextButton, { variant: 'dark', width: 'small', onClick: attemptSubmitAccessCode, autoFocus: true }, "Submit")))));
};
/* Step three asks the user want they want to use Mito for */
const StepThree = (props) => {
    const onSubmit = async () => {
        // Log the answers to the questions each as their own piece of feedback
        await props.mitoAPI.updateFeedback("company/organization" /* COMPANY */, props.numUsages, [{ 'question': FirstQuestion, 'answer': props.firstResponse }]);
        // Advance to the next step
        props.next();
    };
    return (React.createElement("div", { className: 'signup-modal-left-column' },
        React.createElement("div", null,
            React.createElement("h1", { className: 'mt-0' },
                props.isPro && `Mito Pro is Totally Private`,
                !props.isPro && `Mito is Built for Privacy`),
            React.createElement("p", { className: 'signup-modal-text' },
                props.isPro && `As a Mito Pro user, no data leaves your computer, ever. Check out our CCPA compliant privacy policy`,
                !props.isPro && `We make sure none of your private data leaves your computer. Read our CCPA compliant privacy policy`,
                ' ',
                " ",
                React.createElement("a", { href: 'https://privacy.trymito.io/privacy-policy', target: '_blank', rel: "noreferrer" },
                    React.createElement("u", null, "here")),
                ".")),
        React.createElement("div", { className: 'signup-modal-buttons mt-10px' },
            React.createElement(TextButton, { variant: 'light', width: 'small', onClick: props.back }, "Back"),
            React.createElement(TextButton, { variant: 'dark', width: 'small', onClick: onSubmit, autoFocus: true }, "Accept"))));
};
/*
    First signup modal, which collects the users email, shows them the privacy policy,
    and then asks them what they want to do with the tool.
*/
const SignupModal = (props) => {
    const [step, setStep] = useState(1);
    const [email, setEmail] = useState('');
    // Store the first and second response here, so if they use the back button, their answers are stored
    const [firstResponse, setFirstResponse] = useState('');
    const next = () => {
        // Note that if the user is pro, we don't show them the final signup step
        // as we don't collect that data anyways
        if (step + 1 > 3) {
            props.setUIState((prevUIState) => {
                return Object.assign(Object.assign({}, prevUIState), { currOpenModal: { type: ModalEnum.None } });
            });
            void props.mitoAPI.log('finished_signup');
        }
        else {
            const newStep = Math.min(step + 1, 3);
            setStep(newStep);
            void props.mitoAPI.log('switched_signup_step', {
                'old_signup_step': step,
                'new_signup_step': newStep
            });
        }
    };
    const back = () => {
        const newStep = Math.max(step - 1, 0);
        setStep(newStep);
        void props.mitoAPI.log('switched_signup_step', {
            'old_signup_step': step,
            'new_signup_step': newStep
        });
    };
    // Background colors of the different steps right column
    const backgroundColors = {
        1: '#FFEBEB',
        2: '#F0C5BB',
        3: '#FFDAAE',
    };
    return (React.createElement("div", { className: 'overlay' },
        React.createElement("div", { className: 'signup-modal-container' },
            React.createElement("div", { className: 'signup-modal-left-column-container' },
                step === 1 &&
                    React.createElement(StepOne, { next: next, email: email, setEmail: setEmail, firstResponse: firstResponse, setFirstResponse: setFirstResponse, mitoAPI: props.mitoAPI }),
                step === 2 &&
                    React.createElement(StepTwo, { next: next, back: back, isPro: props.isPro, mitoAPI: props.mitoAPI }),
                step === 3 &&
                    React.createElement(StepThree, { next: next, back: back, firstResponse: firstResponse, setFirstResponse: setFirstResponse, numUsages: props.numUsages, mitoAPI: props.mitoAPI, isPro: props.isPro })),
            React.createElement("div", { className: 'signup-modal-right-column-container', style: { backgroundColor: backgroundColors[step] } },
                step === 1 &&
                    React.createElement(BlueMitoFolk, null),
                step === 2 &&
                    React.createElement(PinkMitoFolk, null),
                step === 3 &&
                    React.createElement(YellowMitoFolk, null)))));
};
export default SignupModal;
//# sourceMappingURL=SignupModal.js.map