"""Code that evaluates the quality of the MVM products generated by the drizzlepac package.

Visualization of these Pandas DataFrames with Bokeh can follow the example
from:

https://programminghistorian.org/en/lessons/visualizing-with-bokeh

PUBLIC INTERFACE FOR THIS MODULE
 - build_mvm_plots(HDF5_FILE, output_basename='', display_plot=False):

python mvm_quality_graphics.py mvm_qa_dataframe.h5

"""

# Standard library imports
import argparse
import copy
import math
import os
import re
import sys

# Related third party imports
import numpy as np
import pandas as pd
from astropy.io import fits
from bokeh.layouts import row, column, gridplot
from bokeh.plotting import figure, output_file, save, show
from bokeh.models import ColumnDataSource, Label, CDSView, Div, LinearColorMapper, ColorBar
from PIL import Image, ImageDraw
from spherical_geometry.polygon import SphericalPolygon

# Local application imports
import drizzlepac.haputils.astrometric_utils as au
from drizzlepac.haputils.pandas_utils import PandasDFReader, get_pandas_data
from drizzlepac.haputils.graph_utils import HAPFigure, build_tooltips
import drizzlepac.haputils.mvm_quality_analysis as mqa
from stsci.tools import logutil

__taskname__ = 'mvm_quality_graphics'

MSG_DATEFMT = '%Y%j%H%M%S'
SPLUNK_MSG_FORMAT = '%(asctime)s %(levelname)s src=%(name)s- %(message)s'
log = logutil.create_logger(__name__, level=logutil.logging.NOTSET, stream=sys.stdout,
                            format=SPLUNK_MSG_FORMAT, datefmt=MSG_DATEFMT)

# ----------------------------------------------------------------------------------------------------------------------
# Module level variables
color20 = ['#1f77b4', '#393b79', '#aec7e8', '#ffbb78', '#2ca02c',
           '#6b6ecf', '#d62728', '#ff9896', '#9467bd', '#c5b0d5',
           '#8c564b', '#e377c2', '#f7b6d2', '#a55194', '#7f7f7f',
           '#c7c7c7', '#bcbd22', '#dbdb8d', '#ce6dbd', '#9edae5']

# ------------------------------------------------------------------------------------------------------------


def build_mvm_plots(data_source, output_basename='', display_plot=False, log_level=logutil.logging.INFO):
    """Create all the plots for the results generated by these comparisons

    Parameters
    ----------
    data_source : str
        name of the .h5 file produced by diagnostic_json_harvester.py that holds the information to plot.

    output_basename : str
        text string that will be used as the basis for all .html files generated by this script. Unless
        explicitly specified, the default value is 'mvm_qa'. If a value is explicitly specified, the text
        string '_mvm_qa' will be automatically appended to the end of the user-specified value.

    display_plot : bool, optional
        If set to Boolean 'True', plots .html files will be automatically opened in the default web browser
        as they are generated. Unless explicitly specified, the default value is Boolean 'False', meaning
        that plots will only be written to output .html files but not displayed on-screen.

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default value is 20, or 'info'.

    Returns
    -------
    Nothing.
    """
    log.setLevel(log_level)
    if output_basename == '':
        output_basename = "mvm_qa"
    else:
        output_basename = "{}_mvm_qa".format(output_basename)

    # Generate overlap crossmatch plots
    try:
        build_overlap_crossmatch_plots(data_source,
                                       display_plot,
                                       output_basename=output_basename,
                                       log_level=log_level)
    except Exception:
        log.warning("Overlap crossmatch plot generation encountered a problem.")
        log.exception("message")
        log.warning("Continuing to next plot...")

    # Generate the WCS graphics
    try:
        wcs_graphics_driver(data_source, output_basename, display_plot, log_level=log_level)
    except Exception:
        log.warning("WCS comparison plot generation encountered a problem.")
        log.exception("message")
        log.warning("Continuing to next plot...")

# ------------------------------------------------------------------------------------------------------------
# Functions for generating each data plot
# -----------------------------------------------------------------------------
# WCS graphic
#


def build_overlap_crossmatch_plots(data_source, display_plot=False, output_basename='mvm_qa', log_level=logutil.logging.INFO):
    """retrieve required data and reformat the dataframe in preperation for the generation of overlap
    crossmatch plots.

    Parameters
    ----------
    data_source : str
        name of the .h5 file produced by diagnostic_json_harvester.py that holds the information to plot.

    display_plot : bool, optional.
        If set to Boolean 'True', plots .html files will be automatically opened in the default web browser
        as they are generated. Unless explicitly specified, the default value is Boolean 'False', meaning
        that plots will only be written to output .html files but not displayed on-screen.

    output_basename : str, optional.
        text string that will be used as the basis for all .html files generated by this script. Unless
        explicitly specified, the default value is 'mvm_qa'.

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default value is 20, or 'info'.

    Returns
    -------
    Nothing
    """
    log.setLevel(log_level)
    # lists of column titles that will be used.
    details_column_basenames = ["overlap_region_size",
                                "reference_catalog_name",
                                "comparison_catalog_name",
                                "total_reference_catalog_size",
                                "total_comparison_catalog_size",
                                "number_of_reference_sources_available_for_crossmatch",
                                "number_of_comparison_sources_available_for_crossmatch",
                                "number_of_crossmatched_sources"]

    difference_column_basenames = ["X-axis_differences",
                                   "Y-axis_differences",
                                   "On-sky_separation_(X-Y)",
                                   "On-sky_separation_(RA-Dec)"]

    stats_column_basenames = ["Non-clipped_minimum",
                              "Non-clipped_maximum",
                              "Non-clipped_mean",
                              "Non-clipped_median",
                              "Non-clipped_standard_deviation",
                              "3x3-sigma_clipped_mean",
                              "3x3-sigma_clipped_median",
                              "3x3-sigma_clipped_standard_deviation",
                              "Percent_of_all_diff_values_within_1-sigma_of_the_clipped_mean",
                              "Percent_of_all_diff_values_within_2-sigma_of_the_clipped_mean",
                              "Percent_of_all_diff_values_within_3-sigma_of_the_clipped_mean"]

    data_table_column_basename = "crossmatched_reference_X,_Y,_RA,_Dec,_and_crossmatched_comparison_-_reference_difference_values"
    data_table_colnames = ["X-Skycell",
                           "Y-Skycell",
                           "RA",
                           "DEC",
                           "X-axis differences",
                           "Y-axis differences",
                           "On-sky separation (X-Y)",
                           "On-sky separation (RA-Dec)"]

    # list of 26 hand-selected bokeh colors that will be used in the stats plots
    color_list = ["black", "blue", "brown", "fuchsia", "gold", "green", "olive", "orange", "purple",
                  "rebeccapurple", "red", "rosybrown", "royalblue", "saddlebrown", "salmon", "sandybrown",
                  "seagreen", "skyblue", "springgreen", "steelblue", "tan", "teal", "tomato", "turquoise",
                  "violet", "wheat"]

    n_layers_colname = 'gen_info.number of overlap regions present'
    num_layers = get_pandas_data(data_source, [n_layers_colname])[n_layers_colname]

    # retrieve relevant data and "restack" and rename dataframe so that all the information for each overlap
    # region is stored in discrete rows
    restacked_overlap_dataframe = pd.DataFrame()
    color_ctr = 0
    color_list_reset = False
    for df_indexname, layer_val in zip(num_layers.index.values, num_layers.values):
        for layer_ctr in range(1, layer_val + 1):
            columns_to_retrieve = []
            column_basename = "overlap_region_#{}".format(layer_ctr)
            # add "overlap details" columns
            for details_colname in details_column_basenames:
                columns_to_retrieve.append("{}_details.{}".format(column_basename, details_colname))
            # add stats columns for each difference type
            for diff_type in difference_column_basenames:
                for stats_colname in stats_column_basenames:
                    columns_to_retrieve.append("{}_{}.{}".format(column_basename, diff_type, stats_colname))
            # add all the data table columns
            for data_table_colname in data_table_colnames:
                columns_to_retrieve.append(
                    "{}_{}.{}".format(column_basename, data_table_column_basename, data_table_colname))
            overlap_dataframe = get_pandas_data(data_source, columns_to_retrieve)
            overlap_dataframe = overlap_dataframe[overlap_dataframe['gen_info.dataframe_index'] == df_indexname]
            dataframe_indexname = "{}_overlap_region_{}".format(overlap_dataframe['gen_info.dataframe_index'][0], layer_ctr)
            overlap_dataframe['gen_info.dataframe_index'] = dataframe_indexname
            col_rename_dict = {}
            for colname in columns_to_retrieve:
                col_rename_dict[colname] = colname.replace(column_basename, "overlap_region")
            overlap_dataframe = overlap_dataframe.rename(columns=col_rename_dict)
            overlap_dataframe['colormap'][0] = color_list[color_ctr]
            if color_list_reset == True:
                log.info("NOTE: There are more data points than colors defined for the statistics plots. The color assigned to '{}' is NOT unique.".format(dataframe_indexname))
            restacked_overlap_dataframe = restacked_overlap_dataframe.append(overlap_dataframe)
            color_ctr += 1
            # reset the counter so that it starts back at the first color in the list if there are more
            # data points to plot than colors in color_list.
            if color_ctr == len(color_list):
                color_list_reset = True
                color_ctr = 0
    # Sort columns alphabetically to make it more human-friendly
    restacked_overlap_dataframe = restacked_overlap_dataframe[overlap_dataframe.columns.sort_values()]
    # optionally write dataframe to .csv file.
    if log_level == logutil.logging.DEBUG:
        output_csv_filename = "testout.csv"
        if os.path.exists(output_csv_filename):
            os.remove(output_csv_filename)
        restacked_overlap_dataframe.to_csv(output_csv_filename)
        log.debug("Wrote restacked dataframe to csv file {}".format(output_csv_filename))

    # generate plots!
    generate_overlap_crossmatch_graphics(restacked_overlap_dataframe,
                                         display_plot=display_plot,
                                         output_basename=output_basename,
                                         log_level=log_level)


# ------------------------------------------------------------------------------------------------------------

def generate_overlap_crossmatch_graphics(dataframe, display_plot=False, output_basename='mvm_qa', log_level=logutil.logging.INFO):
    """Generate plots to statistically quantify the quality of the alignment of crossmatched sources in
    regions where observations from different proposal/visits overlap.

    Parameters
    ----------
    dataframe : pandas dataframe
        dataframe containing results from the overlap crossmatch(s) to plot.

    display_plot : bool, optional.
        If set to Boolean 'True', plots .html files will be automatically opened in the default web browser
        as they are generated. Unless explicitly specified, the default value is Boolean 'False', meaning
        that plots will only be written to output .html files but not displayed on-screen.

    output_basename : str, optional.
        text string that will be used as the basis for all .html files generated by this script. Unless
        explicitly specified, the default value is 'mvm_qa'.

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default value is 20, or 'info'.

    Returns
    -------
    Nothing
    """
    log.setLevel(log_level)
    xmatch_cds = ColumnDataSource(dataframe)
    # generate plots of x vs. y components of various stat. measures for each difference
    output = "{}_overlap_crossmatch_stats_plots".format(output_basename)
    if not output.endswith('.html'):
        output = output + '.html'
    # Set the output file immediately as advised by Bokeh.
    output_file(output)

    # Generate the graphic-specific tooltips - be mindful of the default tooltips defined in graph_utils.py
    hover_columns = ["gen_info.dataframe_index",
                     "overlap_region_details.overlap_region_size",
                     "overlap_region_details.reference_catalog_name",
                     "overlap_region_details.comparison_catalog_name",
                     "overlap_region_details.number_of_crossmatched_sources"]
    tooltips_list = ['Overlap region name',
                     'Overlap region size (pixels)',
                     'Ref. SVM catalog name',
                     'Comp. SVM catalog name',
                     'Number of crossmatched sources']
    tooltips_list_dynamic = ["X value", "Y value"]
    # hover_tips = build_tooltips(tooltips_list, hover_columns, list(range(0, len(hover_columns))))
    # Define the graphics
    plot_list = []
    # Create title text at the top of the html file
    html_title_text = Div(text="""<h1>Distribution characteristics of crossmatched sources identified in regions of overlapping observations in the MVM product</h1>""")
    plot_list.append(html_title_text)
    # Scatter plots!
    hover_columns_dynamic = ['overlap_region_X-axis_differences.Non-clipped_minimum',
                             'overlap_region_Y-axis_differences.Non-clipped_minimum']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p0 = HAPFigure(title='Minimum difference value',
                   x_label='X minimum difference (pixels)',
                   y_label='Y minimum difference (pixels)', hover_tips=hover_tips)
    p0.build_glyph('circle',
                   x='overlap_region_X-axis_differences.Non-clipped_minimum',
                   y='overlap_region_Y-axis_differences.Non-clipped_minimum',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')

    hover_columns_dynamic = ['overlap_region_X-axis_differences.Non-clipped_maximum',
                             'overlap_region_Y-axis_differences.Non-clipped_maximum']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p1 = HAPFigure(title='Maximum difference value',
                   x_label='X maximum difference (pixels)',
                   y_label='Y maximum difference (pixels)', hover_tips=hover_tips)
    p1.build_glyph('circle',
                   x='overlap_region_X-axis_differences.Non-clipped_maximum',
                   y='overlap_region_Y-axis_differences.Non-clipped_maximum',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')
    row1 = row(p0.fig, p1.fig)
    plot_list.append(row1)

    hover_columns_dynamic = ['overlap_region_X-axis_differences.Non-clipped_median',
                             'overlap_region_Y-axis_differences.Non-clipped_median']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p2 = HAPFigure(title='Median difference value',
                   x_label='X median difference (pixels)',
                   y_label='Y median difference (pixels)', hover_tips=hover_tips)
    p2.build_glyph('circle',
                   x='overlap_region_X-axis_differences.Non-clipped_median',
                   y='overlap_region_Y-axis_differences.Non-clipped_median',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')

    hover_columns_dynamic = ['overlap_region_X-axis_differences.Non-clipped_mean',
                             'overlap_region_Y-axis_differences.Non-clipped_mean']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p3 = HAPFigure(title='Mean difference value',
                   x_label='X mean difference (pixels)',
                   y_label='Y mean difference (pixels)', hover_tips=hover_tips)
    p3.build_glyph('circle',
                   x='overlap_region_X-axis_differences.Non-clipped_mean',
                   y='overlap_region_Y-axis_differences.Non-clipped_mean',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')
    row2 = row(p2.fig, p3.fig)
    plot_list.append(row2)

    hover_columns_dynamic = ['overlap_region_X-axis_differences.Non-clipped_standard_deviation',
                             'overlap_region_Y-axis_differences.Non-clipped_standard_deviation']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p4 = HAPFigure(title='Difference standard deviation value',
                   x_label='X standard deviation difference (pixels)',
                   y_label='Y standard deviation difference (pixels)', hover_tips=hover_tips)
    p4.build_glyph('circle',
                   x='overlap_region_X-axis_differences.Non-clipped_standard_deviation',
                   y='overlap_region_Y-axis_differences.Non-clipped_standard_deviation',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')

    hover_columns_dynamic = ['overlap_region_X-axis_differences.3x3-sigma_clipped_median',
                             'overlap_region_Y-axis_differences.3x3-sigma_clipped_median']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p5 = HAPFigure(title='3x3 sigma-clipped median difference value',
                   x_label='X sigma-clipped median difference (pixels)',
                   y_label='Y sigma-clipped median difference (pixels)', hover_tips=hover_tips)
    p5.build_glyph('circle',
                   x='overlap_region_X-axis_differences.3x3-sigma_clipped_median',
                   y='overlap_region_Y-axis_differences.3x3-sigma_clipped_median',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')
    row3 = row(p4.fig, p5.fig)
    plot_list.append(row3)

    hover_columns_dynamic = ['overlap_region_X-axis_differences.3x3-sigma_clipped_mean',
                             'overlap_region_Y-axis_differences.3x3-sigma_clipped_mean']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))
    p6 = HAPFigure(title='3x3 sigma-clipped mean difference value',
                   x_label='X sigma-clipped mean difference (pixels)',
                   y_label='Y sigma-clipped mean difference (pixels)', hover_tips=hover_tips)
    p6.build_glyph('circle',
                   x='overlap_region_X-axis_differences.3x3-sigma_clipped_mean',
                   y='overlap_region_Y-axis_differences.3x3-sigma_clipped_mean',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')

    hover_columns_dynamic = ['overlap_region_X-axis_differences.3x3-sigma_clipped_standard_deviation',
                             'overlap_region_Y-axis_differences.3x3-sigma_clipped_standard_deviation']
    hover_tips = build_tooltips(tooltips_list+tooltips_list_dynamic, hover_columns+hover_columns_dynamic,
                                list(range(0, len(hover_columns)+2)))

    p7 = HAPFigure(title='3x3 sigma-clipped Difference standard deviation value',
                   x_label='X sigma-clipped difference standard_deviation (pixels)',
                   y_label='Y sigma-clipped difference standard_deviation (pixels)', hover_tips=hover_tips)
    p7.build_glyph('circle',
                   x='overlap_region_X-axis_differences.3x3-sigma_clipped_standard_deviation',
                   y='overlap_region_Y-axis_differences.3x3-sigma_clipped_standard_deviation',
                   sourceCDS=xmatch_cds,
                   glyph_color='colormap',
                   legend_group='gen_info.dataframe_index')
    row4 = row(p6.fig, p7.fig)
    plot_list.append(row4)

    if display_plot:
        show(column(plot_list))
    # Just save
    else:
        save(column(plot_list))
    log.info("Output HTML graphic file {} has been written.\n".format(output))

    # generate quad resid (x vs. dx, y vs. dx, x vs. dy, y vs. dy) plots for each DF row

    for dfindex, dfrow in dataframe.iterrows():
        qr_df = pd.DataFrame()
        qr_df = qr_df.append(dfrow)
        qr_cds = ColumnDataSource(qr_df)
        qr_dict = {"x": qr_cds.data['overlap_region_crossmatched_reference_X,_Y,_RA,_Dec,_and_crossmatched_comparison_-_reference_difference_values.X-Skycell'][0],
                   "y": qr_cds.data['overlap_region_crossmatched_reference_X,_Y,_RA,_Dec,_and_crossmatched_comparison_-_reference_difference_values.Y-Skycell'][0],
                   "dx": qr_cds.data['overlap_region_crossmatched_reference_X,_Y,_RA,_Dec,_and_crossmatched_comparison_-_reference_difference_values.X-axis differences'][0],
                   "dy": qr_cds.data['overlap_region_crossmatched_reference_X,_Y,_RA,_Dec,_and_crossmatched_comparison_-_reference_difference_values.Y-axis differences'][0]}
        qr_cds = ColumnDataSource(qr_dict)
        output = "{} {}_overlap_crossmatch_residuals_plots".format(output_basename, qr_df['gen_info.dataframe_index'].values[0])
        if not output.endswith('.html'):
            output = output + '.html'
        # Set the output file immediately as advised by Bokeh.
        output_file(output)
        # Define plots
        plot_list = []
        html_title_text = Div(text="""<h1>Crossmatched comparison - reference residuals:<br>{}</h1>""".format(qr_df['gen_info.dataframe_index'].values[0]))
        plot_list.append(html_title_text)
        # add descriptive info
        for detail_title, detail_value in zip(tooltips_list[1:], hover_columns[1:]):
            if detail_title in ["Overlap region size (pixels)", "Number of crossmatched sources"]:
                dv = int(qr_df[detail_value].values[0])
            else:
                dv = qr_df[detail_value].values[0]
            detail_html_text = Div(text="""<h3>{}: {}</h3>""".format(detail_title, dv))
            plot_list.append(detail_html_text)
        p1 = HAPFigure(title='X vs DX',
                       x_label="X (pixels)",
                       y_label='Delta[X] (pixels)',
                       use_hover_tips=False)
        p1.build_glyph('circle',
                       x='x',
                       y='dx',
                       sourceCDS=qr_cds)

        p2 = HAPFigure(title='X vs DY',
                       x_label="X (pixels)",
                       y_label='Delta[Y] (pixels)',
                       use_hover_tips=False)
        p2.build_glyph('circle',
                       x='x',
                       y='dy',
                       sourceCDS=qr_cds)
        row1 = row(p1.fig, p2.fig)
        plot_list.append(row1)

        p3 = HAPFigure(title='Y vs DX',
                       x_label="Y (pixels)",
                       y_label='Delta[X] (pixels)',
                       use_hover_tips=False)
        p3.build_glyph('circle',
                       x='y',
                       y='dx',
                       sourceCDS=qr_cds)

        p4 = HAPFigure(title='Y vs DY',
                       x_label="Y (pixels)",
                       y_label='Delta[Y] (pixels)',
                       use_hover_tips=False)
        p4.build_glyph('circle',
                       x='y',
                       y='dy',
                       sourceCDS=qr_cds)
        row2 = row(p3.fig, p4.fig)
        plot_list.append(row2)

        # Display and save
        if display_plot:
            show(column(plot_list))
        # Just save
        else:
            save(column(plot_list))
        log.info("Output HTML graphic file {} has been written.\n".format(output))

# ------------------------------------------------------------------------------------------------------------


def wcs_graphics_driver(storage_filename, output_base_filename='', display_plot=False,
                        log_level=logutil.logging.INFO):
    """Driver to load the data from the storage file and generate the graphics.

    Parameters
    ==========
    storage_filename : str
        Name of the storage file for the Pandas dataframe created by the harvester.

    output_base_filename : str, optional
        Base name for the HMTL file generated by Bokeh.

    display_plot : bool, optional
        Option to display the plot in addition to writing out the file.

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default value is 20, or 'info'.

    Returns
    =======
    Nothing
    """
    log.setLevel(log_level)

    # Retrieve the relevant dataframe columns as specified by the column_restring, as
    # well as some additional columns for context where the latter is done by default
    log.info('Retrieve Pandas dataframe from file {}.\n'.format(storage_filename))
    wcs_dataDF = get_wcs_data(storage_filename, column_restring='p\d\d\d\dx\d\dy\d\d')

    # Dictionary which defines how old column names map to new column names
    wcs_columns = {}

    # Rename the columns to abbreviated text as the graph titles further
    # document the information.
    if bool(wcs_columns):
        for old_col_name, new_col_name in wcs_columns.items():
            wcs_dataDF.rename(columns={old_col_name: new_col_name}, inplace=True)

    # Generate the WCS graphic for specific skycell layer
    generate_wcs_graphic(wcs_dataDF, output_base_filename, display_plot, log_level)

    generate_wcs_footprint(wcs_dataDF, output_base_filename, display_plot, log_level)

# ------------------------------------------------------------------------------------------------------------

def generate_wcs_footprint(wcs_dataDF, output_base_filename='', display_plot=False,
                         log_level=logutil.logging.NOTSET):
    """Generate the graphics associated with this particular type of data.

    Parameters
    ==========
    wcs_dataDF : Pandas dataframe
        A subset of the input Dataframe consisting only of columns associated
        with the gathering of WCS data

    output_base_filename : str
        Base name for the HMTL file generated by Bokeh.

    display_plot : bool, optional
        Option to display the plot to the screen
        Default: False

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default: 20 or 'info'.

    Returns
    =======
    Nothing

    HTML file is generated and written to the current directory.
    """
    log.setLevel(log_level)

    # Set the output base filename immediately as advised by Bokeh.
    if output_base_filename == '':
        output_base_filename = 'mvm_qa_wcs'
    else:
        output_base_filename = '{}_wcs'.format(output_base_filename)

    # Create a clean dictionary of the wcsnames
    wcs_bins = copy.copy(mqa.wcs_pref_list)
    for key, value in wcs_bins.items():
        wcs_bins[key] = 0

    column_names = [item for item in wcs_dataDF.columns if re.search('p\d\d\d\dx\d\dy\d\d.filename', item)]
    skycell_names = [item.split('.')[0] for item in column_names]

    # Get all the row names (dataframe index) - these are the layers of the skycell
    all_rows = wcs_dataDF.index.tolist()

    for skycell in skycell_names:

        # There is one html page for each skycell
        output_filename = skycell + '_' + output_base_filename + '.html'
        output_file(output_filename)

        # Get only the rows (aka layers) for the specified skycell
        layer_names = [item for item in all_rows if re.search(skycell, item)]
        num_layers = len(layer_names)

        # Image containing the footprints for all the exposures in all layers of a skycell
        cnt_all_layers_mask = None
        all_layers_mask = None

        log.info("Processing skycell {}".format(skycell))
        layer_stats = "<h3>Overlap Region Statistics for Mismatched WCS Solutions</h3>"
        layer_stats += "<p>Values of '--' indicate constituent exposures had same WCS solution.</p>"
        layer_stats += "<table><tr><th>Layer</th><th>Min</th><th>Max</th><th>Mean</th><th>Std</th><th># Pixels in Use</th><th># Pixels in Overlap</th><th>%Pixels in Overlap</th></tr>"
        exposure_info = "<table><tr><th>Layer</th><th>Exposure</th><th>WCSNAME</th><th>WCS Value</th></tr>"
        
        # Loop over the layers
        for layer in layer_names:
            output_layer_name = '_'.join((layer.split('-')[1]).split('_')[0:-1])
            # Get the X- and Y-corners for all exposures in one layer in the skycell
            # Each all_expos_* is a list containing a list which contains one or two lists.
            # The inner-most one or two lists represent each possible chip for the exposure.
            all_expo_X = wcs_dataDF.loc[layer, skycell + '.X']
            all_expo_Y = wcs_dataDF.loc[layer, skycell + '.Y']

            # The shape is the same value for all the exposures in a layer as it is the
            # shape of the output drizzled/layered image.  The data was stored/duplicated
            # for each exposure (just a bit redundant).
            all_shape = wcs_dataDF.loc[layer, skycell + '.shape']
            all_wcs = wcs_dataDF.loc[layer, skycell + '.wcs_value']
            all_wcsnames = wcs_dataDF.loc[layer, skycell + '.wcsname']
            all_filenames = wcs_dataDF.loc[layer, skycell + '.filename']
            filenames = [item.split("_")[-2].split("-")[-1] for item in all_filenames]
            for index in range(len(all_wcsnames)):
                exposure_info += f"<tr><td>{layer[12:-4]}</td><td>{filenames[index]}</td><td>{all_wcsnames[index]}</td><td>{all_wcs[index]}</td></tr>"

            # Define the all_layers_mask here due to the way shape is stored
            if cnt_all_layers_mask is None:
                cnt_all_layers_mask = np.zeros(all_shape[0], dtype=np.uint8)
                all_layers_mask = np.zeros(all_shape[0], dtype=np.int32)

            # Image containing the footprints for all the exposures in a single layer
            layer_mask = None
            cnt_layer_mask = None

            # Loop over exposures contributing to the layer (where each exposure may have one or two chips)
            for expo in range(len(all_expo_X)):
                wcs_value = all_wcs[expo]
                x_corners = all_expo_X[expo]
                y_corners = all_expo_Y[expo]
                shape = all_shape[expo]
                if layer_mask is None:
                    layer_mask = np.zeros(shape, dtype=np.int32)
                    cnt_layer_mask = np.zeros(shape, dtype=np.uint8)

                expo_mask = None
                # Loop over each chip in an exposure
                for ichip in range(len(all_expo_X[0])):
                    polygon = list(zip(x_corners[ichip], y_corners[ichip]))
                    naxis1, naxis2 = shape

                    # Take care to reverse the dimensions to match those of Numpy
                    #
                    # Be careful with the mode - the layer_mask can have a max value
                    # of less than 2^21 (2^20 + other), and the cnt_layer_mask is the
                    # number of exposures in that portion of the mask which we will
                    # assume is no larger than 2^16.
                    cnt_img = Image.new("L", (naxis2, naxis1), color=0)
                    ImageDraw.Draw(cnt_img).polygon(polygon, outline=1, fill=1)

                    # Update the image counter mask
                    cnt_expo_mask = np.array(cnt_img).astype(np.uint8)
                    cnt_layer_mask += cnt_expo_mask
                    cnt_all_layers_mask += cnt_expo_mask

                    # Set filled portion of the exposure mask to the WCS 2^N value
                    # and update the bitwise_OR mask
                    layer_img = Image.new("I", (naxis2, naxis1), color=0)
                    ImageDraw.Draw(layer_img).polygon(polygon, outline=1, fill=1)
                    expo_mask = np.array(layer_img).astype(np.int32)
                    expo_mask *= wcs_value
                    np.bitwise_or(layer_mask, expo_mask, out=layer_mask, dtype=np.int32)
                    np.bitwise_or(all_layers_mask, expo_mask, out=all_layers_mask, dtype=np.int32)

            # Compute statistics for a specific layer
            stats = compute_overlap_stats(layer_mask, layer)
            layer_stats += stats

        # Compute statistics for the composite of all the layers
        stats = compute_overlap_stats(all_layers_mask, "All Layers")
        layer_stats += stats

        # Resample the output footprint overlay image of all the layers in the skycell
        # as the original image is large and deep (Based on code in astrometric_utils.py.)
        scale = 8
        yend = all_layers_mask.shape[0] % scale
        xend = all_layers_mask.shape[1] % scale
        yend = -1 * yend if yend > 0 else None
        xend = -1 * xend if xend > 0 else None
        new_shape = (all_layers_mask.shape[0] // scale, all_layers_mask.shape[1] // scale)
        all_layers_mask = np.log10(all_layers_mask)
        max_color_wcs = all_layers_mask.max() + 1
        all_layers_mask = au.rebin(all_layers_mask[:yend, :xend], new_shape)

 
        # Slice the output *count* footprint overlay image of all the layers in the skycell.
        # We do NOT want the data values to change - we just want a smaller image.
        cnt_scale = 4
        max_color = cnt_all_layers_mask.max() + 1
        cnt_all_layers_mask = cnt_all_layers_mask[::cnt_scale, ::cnt_scale]
        cnt_naxis1, cnt_naxis2 = cnt_all_layers_mask.shape

        # Image of footprints where the value of a pixel is the number of images
        # contributing to the pixel
        cnt_color_mapper = LinearColorMapper(palette="Spectral11", low=0, high=max_color)
        pcnt = figure(title="Skycell: " + skycell.upper() 
                      + "       Image of Layer Overlap       Total Number of Layers: "
                      + str(num_layers), x_range=(0, cnt_naxis2), y_range=(0, cnt_naxis1))
        pcnt.image(image=[cnt_all_layers_mask], color_mapper=cnt_color_mapper,
                   x=0, y=0, dw=cnt_naxis2, dh=cnt_naxis1)
        cnt_color_bar = ColorBar(color_mapper=cnt_color_mapper, location=(0,0))
        pcnt.add_layout(cnt_color_bar, "right")

        # Image of footprints where the value of a pixel is a bitwise_or of all contributing images
        # This is the resampled image.
        color_mapper_wcs = LinearColorMapper(palette="Spectral11", low=0, high=max_color_wcs)
        pwcs = figure(title="Skycell: " + skycell.upper() + 
                      "       Image of Mismatched Overlap       Value is log10(WCS Value)",
                      x_range=(0, new_shape[1]), y_range=(0, new_shape[0]))
        pwcs.image(image=[all_layers_mask], color_mapper=color_mapper_wcs,
                   x=0, y=0, dw=new_shape[1], dh=new_shape[0])
        color_bar_wcs = ColorBar(color_mapper=color_mapper_wcs, location=(0,0))
        pwcs.add_layout(color_bar_wcs, "right")

        # Add an explanatory comment
        layer_stats += "</table><p>The statistics are computed on the combined 2^N WCS Value associated with WCSNAME.  A smaller WCS Value means a better alignment solution.</p>"
        exposure_info += "</table>"
        exposure_table = Div(text = exposure_info)
        stat_table = Div(text = layer_stats)

        wcs_value_legend = "<h3>Legend: WCS Value Assignment to WCSNAMEs</h3>"
        wcs_value_legend += "<table><tr><th>WCSNAME</th><th>WCS Value (2^N)</th></tr>"
        for key, value in mqa.wcs_pref_list.items():
            wcs_value_legend += f"<tr><td>{key}</td><td align='right'>{value}</td></tr>"
        wcs_value_legend +="</table>"
        wcs_legend_table = Div(text = wcs_value_legend)

        # Display and save
        if display_plot:
            show(gridplot([[pcnt, pwcs], [stat_table, exposure_table], [wcs_legend_table, None]]))
            log.info("Output HTML graphic file {} displayed in browser and has been written.\n".format(output_filename))
        else:
            save(gridplot([[pcnt, pwcs], [stat_table, exposure_table], [wcs_legend_table, None]]))
            log.info("Output HTML graphic file {} has been written.\n".format(output_filename))

# ------------------------------------------------------------------------------------------------------------

def compute_overlap_stats(footprint, layer_name, log_level=logutil.logging.NOTSET):
    """Compute general statistics on the overlap regions. 

    Parameters
    ----------
    footprint : `numpy.ndarray`
        Mask of the footprint of a single skycell layer or all the skycell layers combined where
        the pixels in the footprint are bitwise_OR of the input footprints

    """

    # Number of pixels in footprint which are overlaid with data (i.e., pixels are not 0)
    total_pixels_in_use = (footprint > 0).sum()

    # Identify all the *overlap* pixels (value is NOT 2^N and NOT zero)
    #
    # These pixels are a *combination* of the 2^N values in wcs_list.
    # The overlap_pixel_mask retains the value of the combination 2^N values or zero.
    overlap_pixel_mask = (np.ceil(np.log2(footprint)) != np.floor(np.log2(footprint))) * footprint
    masked_overlap_pixel_mask = np.ma.masked_equal(overlap_pixel_mask, 0, copy=False)

    # Collect some statistics on the overlap pixels
    dmin = masked_overlap_pixel_mask.min()
    dmax = masked_overlap_pixel_mask.max()
    dmean = masked_overlap_pixel_mask.mean()
    dstd = masked_overlap_pixel_mask.std()
    num_overlap_pixels = (np.ceil(np.log2(footprint)) != np.floor(np.log2(footprint))).sum()
    perc_overlap = num_overlap_pixels / total_pixels_in_use

    if layer_name.upper().startswith("ALL"):
        ln = layer_name
    else:
        ln = layer_name[12:-4]

    text = f"<tr><td>{ln:30s}</td><td>{dmin:6d}</td><td>{dmax:6d}</td><td>{dmean:9.2f}</td><td>{dstd:8.3f}</td><td>{total_pixels_in_use:8d}</td><td>{num_overlap_pixels:8d}</td><td>{perc_overlap:6.3f}</td></tr>"

    return text

# ------------------------------------------------------------------------------------------------------------

def generate_wcs_graphic(wcs_dataDF, output_base_filename='', display_plot=False,
                         log_level=logutil.logging.NOTSET):
    """Generate the graphics associated with this particular type of data.

    Parameters
    ==========
    wcs_dataDF : Pandas dataframe
        A subset of the input Dataframe consisting only of columns associated
        with the gathering of WCS data

    output_base_filename : str
        Base name for the HMTL file generated by Bokeh.

    display_plot : bool, optional
        Option to display the plot to the screen
        Default: False

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default: 20 or 'info'.

    Returns
    =======
    Nothing

    HTML file is generated and written to the current directory.
    """
    log.setLevel(log_level)

    # Set the output file immediately as advised by Bokeh.
    if output_base_filename == '':
        output_base_filename = 'mvm_qa_wcs'
    else:
        output_base_filename = '{}_mvm_qa_wcs'.format(output_base_filename)
    output_file(output_base_filename + '.html')

    # Suffix for the column data needed for the bar chart
    suffix = ['wcsname', 'wcs_value']

    # Gather information regarding constituents of each skycell (p####x##y##)
    # 0) Determine all the skycells in use
    # 1) Identify all layers of a skycell
    # 2) Determine number of exposures in each layer
    # 3) For each layer make a set of the WCSNAMEs
    # 4) Compute the % of exposures using each WCSNAME
    # 5) Plot as a bar chart

    # Determine all the skycell names
    column_names = [item for item in wcs_dataDF.columns if re.search('p\d\d\d\dx\d\dy\d\d.filename', item)]
    skycell_names = [item.split('.')[0] for item in column_names]
    log.info('There are {} skycell datasets.'.format(len(skycell_names)))

    # Get all the row names (dataframe index) - these are the layers of the skycell
    all_rows = wcs_dataDF.index.tolist()

    # Set up for a list of plots
    plots = []

    # Loop over the skycell names for the graphics to be generated
    for skycell in skycell_names:

        # Set up the names of the columns needed for the bar graphic
        bar_columns = [skycell + '.' + item for item in suffix]

        # Get all the layer names for just this skycell
        layer_names = [item for item in all_rows if re.search(skycell, item)]
        num_layers = len(layer_names)

        # This is a dataframe of just one skycell
        df = wcs_dataDF.loc[layer_names, bar_columns]

        # Determine the number of exposures and unique WCS names in the layer
        # row is a list of two lists, wcsnames and wcs_values
        wcsnames_all_layers = []
        layer_dict = {}
        max_num_expo = 0
        for layer in layer_names:
            wcs_dict = {}
            row = df.loc[layer].tolist()
            num_exposures = len(row[0])
            max_num_expo = num_exposures if num_exposures > max_num_expo else max_num_expo
            set_wcsnames = set(row[0])
            wcsnames_all_layers.extend(row[0])
            for item in set_wcsnames:
                # How many times is WCSNAME in the list of WCSNAMEs for this layer
                wcs_dict[item] = row[0].count(item)
            layer_dict[layer] = wcs_dict

        # Get the unique WCSNAMEs as a list
        wcsnames_all_layers = list(set(wcsnames_all_layers))

        # These machinations were done to push the WCS information into a dataframe
        # which can then be utilized by Bokeh for a vertical stacked bar graph

        # The new columns of the skycell dataframe are all the WCSNAMEs in use
        # Want to ensure the data is added to the correct row (aka layer), so add
        # a column of zeros first.
        for item in wcsnames_all_layers:
            empty_col = [0 * i for i in range(num_layers)]
            df[item] = empty_col

        # Now fill in the proper data
        for layer in layer_names:
            for item in wcsnames_all_layers:
                try:
                    df.loc[layer, item] = layer_dict[layer][item]
                # This layer just did not have any exposures with this WCSNAME
                except KeyError:
                    pass

        # Create a new column using the index values -- seems to be easier to use
        df['LAYER_NAME'] = df.index.str[24:-4]
        grouped = df.groupby('LAYER_NAME')[wcsnames_all_layers].sum()

        # Setup the source of the data to be plotted so the axis variables can be
        # referenced by column name in the Pandas dataframe
        sourceCDS = ColumnDataSource(grouped)
        layers = sourceCDS.data['LAYER_NAME'].tolist()

        # Map the WCSNAMEs to specific colors so any particular WCSNAME will always have
        # the same color
        dict_wcs_color = dict(zip(mqa.wcs_pref_list.keys(), color20))
        colors = [dict_wcs_color.get(item) for item in wcsnames_all_layers]

        # The graphic
        p = figure(x_range=layers)
        p.xaxis.major_label_orientation = math.pi / 4
        p.title.text = 'Sky Cell ' + skycell.upper()
        p.xaxis.axis_label = 'Sky Cell Layer'
        p.yaxis.axis_label = 'Count of WCS Names'
        p.x_range.range_padding = 0.1
        p.y_range.end = max_num_expo + (0.3 * max_num_expo)
        if p.y_range.end % 2 > 0:
            p.y_range.end += 1

        legend = wcsnames_all_layers
        p.vbar_stack(stackers=wcsnames_all_layers,
                     x='LAYER_NAME', source=sourceCDS,
                     legend_label=wcsnames_all_layers,
                     width=0.5, color=colors)
        p.legend.location = 'top_left'

        plots.append(p)

    # Setup the grid to have two columns and multiple rows as it is not known a priori
    # how many datasets will be processed
    grid = gridplot(plots, ncols=2)

    # Display and save
    if display_plot:
        show(grid)
        log.info("Output HTML graphic file {} displayed in browser and has been written.\n".format(
            output_base_filename + ".html"))
    # Just save
    else:
        save(grid)
        log.info("Output HTML graphic file {} has been written.\n".format(output_base_filename + ".html"))


# -----------------------------------------------------------------------------
# General Utility functions for plotting
#


def get_wcs_data(storage_filename, wcs_columns=None, column_restring='', log_level=logutil.logging.NOTSET):
    """Load the harvested data, stored in a storage file, into local arrays.

    Parameters
    ==========
    storage_filename : str
        Name of the storage file for the Pandas dataframe created by the harvester.

    wcs_columns : dict
        Dictionary of original column names (keys) as stored in the Pandas dataframe
        and the corresponding simple name (values) which is often more practical for use.

    column_restring : str
        Substring to use to match as a regular expression against all available
        column names. This parameter is ONLY used in the instance column_names
        is None.

    log_level : int, optional
        The desired level of verboseness in the log statements displayed on the screen and written to the
        .log file. Default: 20 or 'info'.

    Returns
    =======
    wcs_dataDF : Pandas dataframe
        Dataframe which is a subset of the input Pandas dataframe which
        consists of only the requested columns and rows, as well as any columns provided
        by pandas_utils for free.

    Note: This routine is different from the nominal get_pandas_data() in that it tries to
    compensate in case on the of requested columns (apriori or aposteriori) is missing.
    """
    log.setLevel(log_level)

    # Instantiate a Pandas Dataframe Reader (lazy instantiation)
    df_handle = PandasDFReader(storage_filename, log_level=log_level)

    # Get the relevant column data
    wcs_dataDF = df_handle.get_columns_HDF5(column_restring=column_restring)

    # If no dataframe were returned, there was a KeyError because columns were
    # not present in the original dataframe versus the columns contained NaNs.
    if wcs_dataDF.empty:
        log.critical("Critical columns not found in storage Pandas dataframe: {}.\n".format(
            storage_filename))
        sys.exit(1)

    log.info("WCS data has been retrieved from the storage Pandas dataframe: {}.\n".format(storage_filename))

    return wcs_dataDF

# ------------------------------------------------------------------------------------------------------------


if __name__ == "__main__":
    """Simple command-line interface. That is all.
    """
    # process command-line inputs with argparse
    parser = argparse.ArgumentParser(description='Generate MVM quality assessment plots based on information'
                                                 ' stored in the user-specified .h5 file.')
    parser.add_argument('input_filename',
                        help='Input .h5 file produced by diagnostic_json_harvester.py that holds the '
                             'information to plot.')
    parser.add_argument('-d', '--display_plot', required=False, action='store_true',
                        help='If specified, plots will be automatically opened in the default web browser as '
                             'they are generated. Otherwise, .html plot files will be generated but not '
                             'opened.')
    parser.add_argument('-l', '--log_level', required=False, default='info',
                        choices=["critical", "error", "warning", "info", "debug", "notset"],
                        help='The desired level of verboseness in the log statements displayed on the screen '
                             'and written to the .log file.')
    parser.add_argument('-o', '--output_basename', required=False, default='',
                        help='Base name for the HMTL file generated by Bokeh.')
    user_args = parser.parse_args()

    # verify that input file exists
    if not os.path.exists(user_args.input_filename):
        err_msg = "File {} doesn't exist.".format(user_args.input_filename)
        log.critical(err_msg)
        sys.exit(err_msg)

    # set up logging
    log_dict = {"critical": logutil.logging.CRITICAL,
                "error": logutil.logging.ERROR,
                "warning": logutil.logging.WARNING,
                "info": logutil.logging.INFO,
                "debug": logutil.logging.DEBUG,
                "notset": logutil.logging.NOTSET}
    log_level = log_dict[user_args.log_level]
    log.setLevel(log_level)

    # execute plot generation!
    build_mvm_plots(user_args.input_filename, output_basename=user_args.output_basename,
                    display_plot=user_args.display_plot, log_level=log_level)
