# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['asynckivy']

package_data = \
{'': ['*']}

install_requires = \
['asyncgui>=0.5,<0.6']

setup_kwargs = {
    'name': 'asynckivy',
    'version': '0.5.1',
    'description': 'Async library for Kivy',
    'long_description': '# AsyncKivy\n\n[Youtube](https://www.youtube.com/playlist?list=PLNdhqAjzeEGjTpmvNck4Uykps8s9LmRTJ)  \n[日本語doc](README_jp.md)  \n\n`asynckivy` is an async library that saves you from ugly callback-based code,\njust like other async libraries do.\nLet\'s say you want to do:\n\n1. `print(\'A\')`\n1. wait for 1sec\n1. `print(\'B\')`\n1. wait for a button to be pressed\n1. `print(\'C\')`\n\nin that order.\nYour code would look like this:\n\n```python\nfrom kivy.clock import Clock\n\ndef what_you_want_to_do(button):\n    print(\'A\')\n\n    def one_sec_later(__):\n        print(\'B\')\n        button.bind(on_press=on_button_press)\n    Clock.schedule_once(one_sec_later, 1)\n\n    def on_button_press(button):\n        button.unbind(on_press=on_button_press)\n        print(\'C\')\n```\n\nIt\'s barely readable and not easy to understand.\nIf you use `asynckivy`, the code above will become like this:\n\n```python\nimport asynckivy as ak\n\nasync def what_you_want_to_do(button):\n    print(\'A\')\n    await ak.sleep(1)\n    print(\'B\')\n    await ak.event(button, \'on_press\')\n    print(\'C\')\n```\n\n## Installation\n\n```\n# stable version\npip install asynckivy\n```\n\n## Pin the minor version\n\nIf you use this module, it\'s recommended to pin the minor version, because if\nit changed, it usually means some breaking changes occurred.\n\n## Usage\n\n```python\nimport asynckivy as ak\n\nasync def some_task(button):\n    # wait for 1sec\n    dt = await ak.sleep(1)\n    print(f\'{dt} seconds have passed\')\n    \n    # wait until a button is pressed\n    await ak.event(button, \'on_press\')\n\n    # wait until \'button.x\' changes\n    __, x = await ak.event(button, \'x\')\n    print(f\'button.x is now {x}\')\n\n    # wait until \'button.x\' becomes greater than 100\n    if button.x <= 100:\n        __, x = await ak.event(button, \'x\', filter=lambda __, x: x>100)\n        print(f\'button.x is now {x}\')\n\n    # wait until EITHER a button is pressed OR 5sec passes\n    tasks = await ak.or_(\n        ak.event(button, \'on_press\'),\n        ak.sleep(5),\n    )\n    print("The button was pressed" if tasks[0].done else "5sec passed")\n\n    # wait until BOTH a button is pressed AND 5sec passes"\n    tasks = await ak.and_(\n        ak.event(button, \'on_press\'),\n        ak.sleep(5),\n    )\n\nak.start(some_task(some_button))\n```\n\n### animation\n\n```python\nimport asynckivy as ak\n\nasync def some_task(widget):\n    # start an animation and wait for the completion.\n    # the keyword-arguments are the same as kivy.animation.Animation\'s.\n    await ak.animate(widget, width=200, t=\'in_out_quad\', d=.5)\n\n    # interpolate between the values 0 and 200.\n    # the keyword-arguments are the same as kivy.animation.Animation\'s.\n    async for v in ak.interpolate(0, 200, s=.2, d=2, t=\'linear\'):\n        print(v)\n        # await ak.sleep(1)  # Do not await anything during the iteration\n\n    # change the text of Label with fade-transition\n    label = Label(...)\n    async with ak.fade_transition(label):\n        label.text = \'new text\'\n```\n\n### touch handling\n\nYou can easily handle `on_touch_xxx` events via `asynckivy.rest_of_touch_moves()`.\n\n```python\nimport asynckivy as ak\n\nclass Painter(RelativeLayout):\n    def on_touch_down(self, touch):\n        if self.collide_point(*touch.opos):\n            ak.start(self.draw_rect(touch))\n            return True\n    \n    async def draw_rect(self, touch):\n        from kivy.graphics import Line, Color, Rectangle\n        from kivy.utils import get_random_color\n        with self.canvas:\n            Color(*get_random_color())\n            line = Line(width=2)\n        ox, oy = self.to_local(*touch.opos)\n        async for __ in ak.rest_of_touch_moves(self, touch):\n            # This part is iterated everytime \'on_touch_move\' is fired.\n            x, y = self.to_local(*touch.pos)\n            min_x = min(x, ox)\n            min_y = min(y, oy)\n            max_x = max(x, ox)\n            max_y = max(y, oy)\n            line.rectangle = [min_x, min_y, max_x - min_x, max_y - min_y]\n            # await ak.sleep(1)  # Do not await anything during the iteration\n\n        # If you want to do something when \'on_touch_up\' is fired, do it here.\n        do_something_on_touch_up()\n```\n\n### threading\n\n`asynckivy` currently does not have any I/O primitives like Trio and asyncio do,\nthus threads are the only way to perform them without blocking the main-thread:\n\n```python\nfrom concurrent.futures import ThreadPoolExecuter\nimport asynckivy as ak\n\nexecuter = ThreadPoolExecuter()\n\nasync def some_task():\n    # create a new thread, run a function inside it, then\n    # wait for the completion of that thread\n    r = await ak.run_in_thread(thread_blocking_operation)\n    print("return value:", r)\n\n    # run a function inside a ThreadPoolExecuter, and wait for the completion\n    r = await ak.run_in_executer(thread_blocking_operation, executer)\n    print("return value:", r)\n```\n\nExceptions(not BaseExceptions) are propagated to the caller,\nso you can handle them like you do in synchronous code:\n\n```python\nimport requests\nfrom requests.exceptions import Timeout\nimport asynckivy as ak\n\nasync def some_task():\n    try:\n        r = await ak.run_in_thread(lambda: requests.get(\'htt...\', timeout=10))\n    except Timeout:\n        print("TIMEOUT!")\n    else:\n        print(\'GOT A RESPONSE\')\n```\n\n### synchronization primitive\n\nThere is a Trio\'s [Event](https://trio.readthedocs.io/en/stable/reference-core.html#trio.Event) equivalent.\n\n```python\nimport asynckivy as ak\n\nasync def task_A(e):\n    print(\'A1\')\n    await e.wait()\n    print(\'A2\')\nasync def task_B(e):\n    print(\'B1\')\n    await e.wait()\n    print(\'B2\')\n\ne = ak.Event()\nak.start(task_A(e))\n# A1\nak.start(task_B(e))\n# B1\ne.set()\n# A2\n# B2\n```\n\n### misc\n\n```python\nimport asynckivy as ak\n\n# schedule a coroutine/Task to start after the next frame\nak.start_soon(coro_or_task)\n```\n\n## Structured Concurrency\n\nBoth `asynckivy.and_()` and `asynckivy.or_()` follow the concept of "structured concurrency".\nWhat does that mean?\nThey promise two things:\n\n* The tasks passed into them never outlive them.\n* Exceptions occured in the tasks are propagated to the parent task.\n\nRead [this post][njs_sc] if you want to know more about "structured concurrency".\n\n## Test Environment\n\n- CPython 3.7 + Kivy 2.0.0\n- CPython 3.8 + Kivy 2.0.0\n- CPython 3.9 + Kivy 2.0.0\n\n## Why this does exist\n\nKivy supports two legit async libraries, [asyncio][asyncio] and [Trio][trio], from version 2.0.0 so developing another one seems [reinventing the wheel][reinventing]. Actually, I started developing this one just for learning how async/await works so it *was* initially `reinventing the wheel`.\n\nBut after playing with Trio and Kivy for a while, I noticed that Trio is not suitable for the situation where fast reactions are required e.g. touch events. The same is true of asyncio. You can confirm it by running `examples/misc/why_xxx_is_not_suitable_for_handling_touch_events.py`, and masshing a mouse button. You\'ll see sometimes `up` is not paired with `down`. You\'ll see the coordinates aren\'t relative to the `RelativeLayout` even though the `target` belongs to it.\n\nThe cause of those problems is that `trio.Event.set()` and `asyncio.Event.set()` don\'t *immediately* resume the tasks that are waiting for the `Event` to be set. They just schedule the tasks to resume.\nSame thing can be said to `nursery.start_soon()` and `asyncio.create_task()`.\n\nTrio and asyncio are async **I/O** libraries after all. They probably don\'t need the functionality that immediately resumes/starts tasks, which I think necessary for Kivy\'s touch handling.\nTheir core design may not be suitable for GUI in the first place.\nThat\'s why I\'m still developing this `asynckivy` library to this day.\n\n[asyncio]:https://docs.python.org/3/library/asyncio.html\n[trio]:https://trio.readthedocs.io/en/stable/\n[reinventing]:https://en.wikipedia.org/wiki/Reinventing_the_wheel\n[njs_sc]:https://vorpus.org/blog/notes-on-structured-concurrency-or-go-statement-considered-harmful/\n',
    'author': 'Nattōsai Mitō',
    'author_email': 'flow4re2c@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gottadiveintopython/asynckivy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
