"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends core_1.Construct {
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        this.requestDestinationArn = false;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3_deployment_BucketDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.ephemeralStorageSize, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            ephemeralStorageSize: props.ephemeralStorageSize,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.accessControl) {
            props.destinationBucket.grantPutAcl(handler);
        }
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // to avoid redundant stack updates, only include "SourceMarkers" if one of
        // the sources actually has markers.
        const hasMarkers = sources.some(source => source.markers);
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.ephemeralStorageSize, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                SourceMarkers: hasMarkers ? sources.map(source => { var _d; return (_d = source.markers) !== null && _d !== void 0 ? _d : {}; }) : undefined,
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependencees on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? props.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.slice(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(props.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Doing this replaces calling `otherResource.node.addDependency(deployment)`.
     */
    get deployedBucket() {
        var _d;
        this.requestDestinationArn = true;
        this._deployedBucket = (_d = this._deployedBucket) !== null && _d !== void 0 ? _d : s3.Bucket.fromBucketArn(this, 'DestinationBucket', cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')));
        return this._deployedBucket;
    }
    renderUniqueId(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '';
        // if the user specifes a custom memory limit, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error("Can't use tokens when specifying 'memoryLimit' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if the user specifies a custom ephemeral storage size, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (ephemeralStorageSize) {
            if (ephemeralStorageSize.isUnresolved()) {
                throw new Error("Can't use tokens when specifying 'ephemeralStorageSize' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${ephemeralStorageSize.toMebibytes().toString()}MiB`;
        }
        // if the user specifies a VPC, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, ephemeralStorageSize, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        var _d;
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return (_d = stack.node.tryFindChild(uuid)) !== null && _d !== void 0 ? _d : new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.159.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.159.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atDate", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atDate);
        }
        throw error;
    } return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atTimestamp", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atTimestamp);
        }
        throw error;
    } return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#after", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.after);
        }
        throw error;
    } return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#fromString", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.fromString);
        }
        throw error;
    } return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.159.0" };
//# sourceMappingURL=data:application/json;base64,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