# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gitws', 'gitws._cli']

package_data = \
{'': ['*']}

install_requires = \
['appdirs>=1.4.4,<2.0.0',
 'click>=8.0.0,<9.0.0',
 'coloredlogs>=15.0.1,<16.0.0',
 'pydantic>=1.10.0,<2.0.0',
 'tomlkit>=0.11.5,<1.0.0']

entry_points = \
{'console_scripts': ['git-ws = gitws._cli:main']}

setup_kwargs = {
    'name': 'git-ws',
    'version': '0.2.0',
    'description': 'Git Workspace - Multi Repository Management Tool',
    'long_description': '[![PyPI Version](https://badge.fury.io/py/git-ws.svg)](https://badge.fury.io/py/git-ws)\n[![PyPI Downloads](https://img.shields.io/pypi/dm/git-ws.svg?label=pypi%20downloads)](https://pypi.python.org/pypi/git-ws)\n[![Python Build](https://github.com/c0fec0de/git-ws/actions/workflows/main.yml/badge.svg)](https://github.com/c0fec0de/git-ws/actions/workflows/main.yml)\n[![Documentation](https://readthedocs.org/projects/git-ws/badge/?version=latest)](https://git-ws.readthedocs.io/en/latest/?badge=latest)\n[![Coverage Status](https://coveralls.io/repos/github/c0fec0de/git-ws/badge.svg?branch=main)](https://coveralls.io/github/c0fec0de/git-ws?branch=main)\n[![python-versions](https://img.shields.io/pypi/pyversions/git-ws.svg)](https://pypi.python.org/pypi/git-ws)\n[![pylint](https://img.shields.io/badge/linter-pylint-%231674b1?style=flat)](https://www.pylint.org/)\n[![black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n# Git Workspace - Multi Repository Management Tool\n\n* [Installation](#installation)\n* [Usage](#usage)\n* [Cheat-Sheet](#cheat-sheet)\n* [Python API](#api)\n* [Alternatives](#alternatives)\n\nGit Workspace is a lightweight tool for creating and managing *workspaces* consisting of several interdependent `git` repositories. Starting from a *main repository*, Git Workspace discovers dependencies specified in a *manifest file*, fetching any specified required repositories and assembling them into a single workspace.\n\n![Workspace](https://github.com/c0fec0de/git-ws/raw/main/docs/images/workspace.png)\n\n👉 You can read more about the used [nomenclature](https://git-ws.readthedocs.io/en/latest/manual/nomenclature.html) in the [documentation](https://git-ws.readthedocs.io/en/latest/index.html).\n\n\n# 📦 Installation\n\nGit Workspace is written in Python and - as usual - installing it is pretty easy:\n\n```bash\npip install git-ws\n```\n\nAnd that\'s it! Ideally, if your project also uses Python, we recommend adding Git Workspace as a dependency to it as well, so that you can track the exact version of it together with your other dependencies. For example, if you use `poetry`, add it by running\n\n```bash\n# Add Git Workspace as development dependency:\npoetry add --group dev git-ws\n```\n\n\n# 📔 Usage\n\nGit Workspace is integrated into git `git ws` - this is what you will be using most of the time.\n\nLet\'s assume we have a project called `myapp`, which requires a library `mylib` that is maintained in another `git` repository. In order to use this project with Git Workspace, `myapp` needs to provide a so called *manifest*. An Git Workspace manifest is a simple [TOML](https://toml.io/) file - by default called `git-ws.toml` in the project\'s root folder - which defines the dependencies a project has as well as some other meta information. A minimal manifest for our project could look like this:\n\n```toml\n[[dependencies]]\nname = "mylib"\nurl = "git@github.com:example/mylib.git"\nrevision = "v2.3.4"\n```\n\nIf `myapp` and `mylib` are stored on the same server (side-by-side), than the manifest can even be simpler:\n\n```toml\n[[dependencies]]\nname = "mylib"\nrevision = "v2.3.4"\n```\n\nThe project will be searched via a relative path (which is either `../mylib` or `../mylib.git` depending on the main repository\'s URL). Relative paths are in general useful as they allow to use the same protocol for the main repository as well as any of its dependencies.\n\nTo build a workspace from a project prepared like that, simply clone it via `git ws`:\n\n```bash\ncd $HOME/Projects\nmkdir my_app_workspace\ncd my_app_workspace\ngit ws clone --update git@github.com:example/myapp.git\n```\n\n👉 Without the `--update` option, only the main repository will be fetched.\n\nThe above will clone the app repository and also the library side-by-side:\n\n```bash\nls -a\n# Should print something like\n.\n..\n.git-ws\nmyapp\nmylib\n```\n\nAs you can see, besides the two repositories we wanted, there is also a hidden `.git-ws` folder where the tool stores the needed configuration data.\n\nSometimes there are use cases where using `git ws clone` cannot be used. For example, when a CI/CD system creates the initial clone of the main repository, you may need a way to fetch the remaining projects. This can be done by simply running the following within the main project:\n\n```bash\ngit ws init --update\n```\n\n👉 As with `git ws clone`, without the `--update`, no dependencies will be fetched.\n\nAnother important use case is keeping a workspace up-to-date. Lets say you pull in an update in the main repository, which in turn might cause changes in the manifest to be pulled in as well. Updating the existing workspace is as simple as\n\n```bash\n# Update the workspace (main and all dependent repositories):\ngit ws update\n\n# Alternatively, run `git rebase` instead of `git pull` in dependencies:\ngit ws update --rebase\n```\n## Cheat-Sheet\n\n#### Initialization\n\n| Command | Description |\n| --- | --- |\n| `git ws clone URL` | Clone git repository from `URL` as main repository and initialize Git Workspace |\n| `git ws init` | Initialize Git Workspace. Use current git clone as main repository |\n| `git ws manifest create` | Create well documented, empty manifest |\n\n#### Basic\n\n| Command | Description |\n| --- | --- |\n| `git ws update` | Pull latest changes on main repository and all dependent repositories (clone them if needed) |\n| `git ws update --rebase` | Same as above, but fetch and rebase instead of pull |\n| `git ws status` | Run `git status` on all repositories (displayed paths include the actual clone path) |\n| `git ws add FILES` | Run `git add FILE` on `FILES` in the corresponding repositories |\n| `git ws reset FILES` | Run `git reset FILE` on `FILES` in the corresponding repositories. Undo `git add` |\n| `git ws commit FILES -m MESSAGE` | Run `git commit FILE` on `FILES` in the corresponding repositories |\n| `git ws commit -m MESSAGE` | Run `git commit` repositories with changes |\n| `git ws checkout FILES` | Run `git checkout FILE` on `FILES` in the corresponding repositories |\n| `git ws checkout` | Checkout git revision specified as in the manifest(s) (clone them if needed) |\n\n#### Run Commands on all repositories\n\n| Command | Description |\n| --- | --- |\n| `git ws push` | Run `git push` on all repositories |\n| `git ws fetch` | Run `git fetch` on all repositories |\n| `git ws rebase` | Run `git rebase` on all repositories |\n| `git ws pull` | Run `git pull` on all repositories |\n| `git ws diff` | Run `git diff` on all repositories |\n| `git ws git CMD` | Run `git CMD` on all repositories |\n| `git ws foreach CMD` | Run `CMD` on all repositories |\n\n#### Other\n\n| Command | Description |\n| --- | --- |\n| `git ws manifest freeze`   | Print The Resolved Manifest With SHAs For All Project Revisions. |\n| `git ws manifest path`     | Print Path to Main Manifest File. |\n| `git ws manifest paths`    | Print Paths to ALL Manifest Files. |\n| `git ws manifest resolve`  | Print The Manifest With All Projects And All Their Dependencies. |\n| `git ws manifest upgrade`  | Update Manifest To Latest Version. |\n| `git ws manifest validate` | Validate The Current Manifest, Exiting With An Error On Issues. |\n| `git ws info main-path`      | Print Path to Main Git Clone. |\n| `git ws info project-paths`  | Print Paths to all git clones. |\n| `git ws info workspace-path` | Print Path to Workspace. |\n\n\n## 🐍 Python API\n\nGit Workspace is written in Python. Besides the `git ws` command line tool, there is also an API which you can use to further automate workspace creation and maintenance. If you are interested, have a look into the [API documentation](https://git-ws.readthedocs.io/en/latest/api/gitws.html).\n\n\n## 🤝 Alternatives\n\nBefore writing Git Workspace, we investigates several other existing tools in the hope they would fulfil out needs. In particular, we looked into the following tools and methodologies which are widely used to organize large projects:\n\n- [`git submodules`](https://git-scm.com/book/en/v2/Git-Tools-Submodules).\n- Google\'s [repo](https://gerrit.googlesource.com/git-repo/) tool.\n- The [`west`](https://docs.zephyrproject.org/latest/develop/west/index.html) tool developed in the scope of [Zephyr](https://www.zephyrproject.org/).\n- Leaving the pure `git` domain, one can also use a package manager like [`conan`](https://conan.io/).\n- And lastly, there are also approaches to still pack everything into a large so called *monorepo*.\n\nUnfortunately, none of the tools we tested really satisfied us. But hey, as we are developers - *why not starting our own tool for the purpose?*\n\nAnd that\'s what we did - Git Workspace is a our tool for managing a large workspace consisting of several smaller `git` projects. Here is how it compares to the other tools we evaluated:\n\n\n|                           | `git submodules` | `repo` | `west` | *Monorepos* | `git ws` |\n| ------------------------- | ---------------- | ------ | ------ | ----------- | --------- |\n| Reusable Components       | ✅               | ✅     | ✅     | ➖          | ✅        |\n| Ease of Use               | ➖               | ✅     | ✅     | ✅          | ✅        |\n| Editable Components       | ➖               | ✅     | ➖     | ✅          | ✅        |\n| Freezing Configurations   | ✅               | ✅     | ✅     | ✅          | ✅        |\n| Transitive Dependencies   | ➖               | ➖     | ✅     | ➖          | ✅        |\n| Relative Dependency Paths | ✅               | ✅     | ➖     | ➖          | ✅        |\n\n👉 Please note that our view on the various features might be biased. As we did, always look at all the options available to you before deciding for one tool or the other. While the other tools in comparison did not model what we needed for our workflow, they might just be what you are looking for.\n\nIf you want to learn more, have a look into the [documentation](https://git-ws.readthedocs.io/en/latest/manual/why.html).\n\n',
    'author': 'c0fec0de',
    'author_email': 'c0fec0de@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
