from __future__ import annotations
import logging
from python_graphql_client import GraphqlClient
from typing import Optional
import boto3
from requests_aws4auth import AWS4Auth
from thousandwords.auth import CognitoJwtAuth
from thousandwords.config import CONFIG
from thousandwords.credentials import CognitoCredentials

logger = logging.getLogger("thousandwords.client")

class Client:
  """
  Client to connect to Thousandwords GraphQL API

  __Humans:__ First, log in via the CLI: `thousandwords login`

  __Bots:__ Set the `THOUSANDWORDS_INSTANCE` and `THOUSANDWORDS_API_KEY` environment variables
  """

  def __init__(
    self,
    instance: Optional[str] = None,
  ):
    """
    Create connection object.

    Arguments:
      instance: Thousandwords instance url. If unspecified, the your config default will be used.
    """
    if instance:
      CONFIG.instance = instance

    self._s3 = None
    self._cognito_creds = CognitoCredentials()
  
  def _get_gql_client(self, auth_type):
    endpoint = CONFIG.api_endpoint
    if auth_type == 'AMAZON_COGNITO_USER_POOLS':
      auth = CognitoJwtAuth()
    elif auth_type == 'AWS_IAM':
      creds = self._cognito_creds.credentials['Credentials']
      is_mock = (endpoint == 'http://192.168.1.30:20002/graphql')
      auth = AWS4Auth(
        # see https://docs.amplify.aws/cli/usage/mock/
        'ASIAVJKIAM-AuthRole' if is_mock else creds['AccessKeyId'],
        creds['SecretKey'],
        CONFIG.api_region,
        'appsync',
        session_token=creds['SessionToken'],
      )
    return GraphqlClient(auth=auth, endpoint=endpoint)

  @property
  def instance(self) -> str:
    """
    Instance name from client config, e.g. `'yourco.thousandwords.ai'`
    """
    return CONFIG.instance
  
  def create_cell(self, input):    
    query = """
      mutation CreateCell($input: CreateCellInput!) {
        createCell(input: $input) {
          id
        }
      }
    """
    ret = self._get_gql_client('AMAZON_COGNITO_USER_POOLS').execute(
      query=query, variables={"input": input}
    )
    if "errors" in ret:
      raise Exception(ret["errors"][0]["message"])

    return ret["data"]["createCell"]["id"]
  
  def run_cell(self, req):
    query = """
      mutation RunCell($request: ExecuteRequestInput) {
        runCell(request: $request) {
          stdout
          stderr
          outputs {
            representations {
              mime
              key
              width
              height
              data
            }
            metadata
          }
          traceback
          userNS {
            name
            key
            value
            serializationType
          }
        }
      } 
    """
    ret = self._get_gql_client('AWS_IAM').execute(
      query=query, variables={"request": req}
    )
    if "errors" in ret:
      raise Exception(ret["errors"][0]["message"])
    return ret["data"]["runCell"]
  
  @property
  def s3(self):
    if not self._s3:
      self._s3 = self._get_session().client('s3')
    return self._s3
  
  def put_protected(self, key, value):
    identity_id = self._cognito_creds.credentials['IdentityId']
    key = f'protected/{identity_id}/{key}'
    resp = self.s3.put_object(
      Key=key,
      Bucket=CONFIG.storage_bucket,
      Body=value,
    )
    logger.debug(f"s3 put_object response: {resp}")
    return key
  
  def get(self, key):
    resp = self.s3.get_object(
      Key=key,
      Bucket=CONFIG.storage_bucket,
    )
    logger.debug(f"s3 get_object response: {resp}")
    return resp['Body'].read()

  def _get_session(self):
    creds = self._cognito_creds.credentials['Credentials']
    return boto3.Session(
      aws_access_key_id=creds['AccessKeyId'],
      aws_secret_access_key=creds['SecretKey'],
      aws_session_token=creds['SessionToken'],
    )