"""
MWCP parser for reverting Quarantined files back to their original state
"""
import configparser
import struct
from itertools import cycle
import pathlib
import re

import olefile
from Crypto.Cipher import ARC4, Blowfish
from Crypto.Util import Padding

from mwcp import Parser, FileObject, metadata


class McAfee(Parser):
    DESCRIPTION = "McAfee Quarantined"
    MINIMUM_OLE_SIZE = 1536

    @classmethod
    def identify(cls, file_object):
        """
        Identify a McAfee Qurantined File.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is a Foo Trojan.
        """
        # Don't run on children files and only if it is an OLE
        # (and if you pass in bytes with len < MINIMUM_OLE_SIZE it crashes)
        return not file_object.parent and (
            len(file_object.data) > cls.MINIMUM_OLE_SIZE
            and olefile.isOleFile(file_object.data)
        )

    def run(self):
        """
        Revert the file back to its original state.
        """
        # quarantined file is in ole format, parse it with the olefile
        ole = olefile.OleFileIO(self.file_object.data)

        # Get a config parse to parse the details in the ole file
        details = configparser.ConfigParser()
        # Details lets us know the details for the quarantined files we want to extract
        if ["Details"] in ole.listdir():
            # Parse and load the configuration details
            details_str = bytes(b ^ 0x6a for b in ole.openstream("Details").read()).decode("utf8")
            details.read_string(details_str)
            file_sections = [f for f in details.sections() if f.startswith("File_")]

            # Iterate through each section to decode an original file
            for f in file_sections:
                # Get original name
                if original_name := details.get(f, "OriginalName", fallback=None):
                    self.report.add(metadata.FilePath(original_name).add_tag("original_location"))
                    file_name = pathlib.PureWindowsPath(original_name).name
                else:
                    file_name = None

                # Shouldn't be necessary since this comes from the details, yet here we are,
                # sometimes there are phantom files listed
                if ole.exists(f):
                    # Decode the files and create the decoded FileObject and add it to the list
                    original_data = bytes(b ^ 0x6a for b in ole.openstream(f).read())
                    self.dispatcher.add(FileObject(original_data, file_name=file_name))
        else:
            self.logger.warning("Unable to revert file!")


class Defender(Parser):
    DESCRIPTION = "Windows Defender Quarantined"
    HEADER_LEN = 0x3C
    RC4_KEY = bytearray(
        b"\x1E\x87\x78\x1B\x8D\xBA\xA8\x44\xCE\x69\x70\x2C\x0C\x78\xB7\x86\xA3\xF6\x23\xB7\x38\xF5\xED\xF9"
        b"\xAF\x83\x53\x0F\xB3\xFC\x54\xFA\xA2\x1E\xB9\xCF\x13\x31\xFD\x0F\x0D\xA9\x54\xF6\x87\xCB\x9E\x18"
        b"\x27\x96\x97\x90\x0E\x53\xFB\x31\x7C\x9C\xBC\xE4\x8E\x23\xD0\x53\x71\xEC\xC1\x59\x51\xB8\xF3\x64"
        b"\x9D\x7C\xA3\x3E\xD6\x8D\xC9\x04\x7E\x82\xC9\xBA\xAD\x97\x99\xD0\xD4\x58\xCB\x84\x7C\xA9\xFF\xBE"
        b"\x3C\x8A\x77\x52\x33\x55\x7D\xDE\x13\xA8\xB1\x40\x87\xCC\x1B\xC8\xF1\x0F\x6E\xCD\xD0\x83\xA9\x59"
        b"\xCF\xF8\x4A\x9D\x1D\x50\x75\x5E\x3E\x19\x18\x18\xAF\x23\xE2\x29\x35\x58\x76\x6D\x2C\x07\xE2\x57"
        b"\x12\xB2\xCA\x0B\x53\x5E\xD8\xF6\xC5\x6C\xE7\x3D\x24\xBD\xD0\x29\x17\x71\x86\x1A\x54\xB4\xC2\x85"
        b"\xA9\xA3\xDB\x7A\xCA\x6D\x22\x4A\xEA\xCD\x62\x1D\xB9\xF2\xA2\x2E\xD1\xE9\xE1\x1D\x75\xBE\xD7\xDC"
        b"\x0E\xCB\x0A\x8E\x68\xA2\xFF\x12\x63\x40\x8D\xC8\x08\xDF\xFD\x16\x4B\x11\x67\x74\xCD\x0B\x9B\x8D"
        b"\x05\x41\x1E\xD6\x26\x2E\x42\x9B\xA4\x95\x67\x6B\x83\x98\xDB\x2F\x35\xD3\xC1\xB9\xCE\xD5\x26\x36"
        b"\xF2\x76\x5E\x1A\x95\xCB\x7C\xA4\xC3\xDD\xAB\xDD\xBF\xF3\x82\x53"
    )

    @classmethod
    def identify(cls, file_object):
        """
        Identify a Defender Qurantined File.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is a Defender Quarantine File.
        """
        return len(file_object.data) > cls.HEADER_LEN and (
            file_object.data.startswith((b"\xDB\xE8\xC5\x01", b"\x0B\xAD"))
        )

    def run(self):
        """
        Revert the file back to its original state.
        """
        header_len = self.HEADER_LEN
        rc4_key = self.RC4_KEY
        file_data = self.file_object.data
        if file_data.startswith(b"\xDB\xE8\xC5\x01"):
            rc4 = ARC4.new(key=rc4_key)
            header = rc4.decrypt(file_data[:header_len])
            data_end_1 = struct.unpack("<I", header[0x28:0x2C])[0] + header_len
            data_end_2 = struct.unpack("<I", header[0x2C:0x30])[0] + data_end_1

            rc4 = ARC4.new(key=rc4_key)
            decrypt_1 = rc4.decrypt(file_data[header_len:data_end_1])

            rc4 = ARC4.new(key=rc4_key)
            decrypt_2 = rc4.decrypt(file_data[data_end_1:data_end_2])

            decrypted_data = header + decrypt_1 + decrypt_2
            self.dispatcher.add(FileObject(decrypted_data))

        elif file_data.startswith(b"\x0B\xAD"):
            rc4 = ARC4.new(key=rc4_key)
            decrypt_all = rc4.decrypt(file_data)
            header_len = 0x28 + struct.unpack("<I", decrypt_all[0x08:0x0c])[0]
            orig_len = struct.unpack("<I", decrypt_all[header_len-12:header_len-8])[0]
            decrypted_data = decrypt_all[header_len:header_len + orig_len]
            self.dispatcher.add(FileObject(decrypted_data))


class SymantecBase(Parser):
    # List of file headers we want to check the decoded files against to see if they are valid
    # language=PythonRegExp
    FILE_HEADERS = [
        b"^MZ.+PE\x00\x00",  # Match a PE,
        b"^(PK\x03\x04|Cr24|Rar!|\xCA\xFE\xBA\xBE|CAB|SZDD)",  # Match archive
        b"^%PDF",  # Match a pdf
        b"^\xD0\xCF\x11\xE0\xA1\xB1\x1A\xE1",  # Match Ole File
        b"^\x20\x00\x02\x00\x00\x00\x12\x00\x0c\x00\x00\x00\x0D",  # Match provided samples of unknown type
    ]


class SymantecSubSDK(SymantecBase):
    DESCRIPTION = "Symantec Quarantined Sub SDK"
    KEY = 0x00

    @classmethod
    def identify(cls, file_object):
        """
        Identify an symantec Qurantined Sub SDK File.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is a Symantec Quarantine File.
        """
        # Don't bother checking the header if this is an extracted file
        if not file_object.parent and len(file_object.data) > 48:
            key = file_object.data[16:32]
            data = file_object.data[32:48]
            try:
                header = cls.decode_data(data, key)
            except ValueError:
                # ValueError occurs if blocks aren't padded correctly.
                return False
            # Because these files are encrypted there is no standard header.
            # Decrypt the first 16 bytes and use that to see if the header matches any known types we care about.
            for file_header in cls.FILE_HEADERS:
                if re.search(file_header, header):
                    return True
        return False

    @classmethod
    def decode_data(cls, data: bytes, key: bytes) -> bytes:
        decrypted_data = bytearray()
        blowfish = Blowfish.new(key, Blowfish.MODE_ECB)

        chunk_size = 16
        offset = 0
        while offset < len(data):
            # Grab an 8 byte chunk for format in Network Order
            chunk = cls.swap_to_network_order(data[offset:offset + chunk_size])
            # decrypt
            decrypted_chunk = blowfish.decrypt(chunk)
            decrypted_chunk = Padding.unpad(decrypted_chunk, Blowfish.block_size)
            decrypted_chunk = cls.swap_to_network_order(decrypted_chunk)
            # reverse back on the decrypted chunk
            decrypted_data += decrypted_chunk
            # Setup the next chunk
            offset += chunk_size
        return bytes(decrypted_data)

    @staticmethod
    def swap_to_network_order(data: bytes) -> bytes:
        offset = 0
        swapped = bytearray()
        while offset + 4 <= len(data):
            swapped += bytearray(reversed(data[offset:offset+4]))
            offset += 4
        return bytes(swapped)

    def run(self):
        """
        Revert the file back to its original state.
        """
        key = self.file_object.data[16:32]
        data = self.file_object.data[32:]
        decoded_data = self.decode_data(data, key)
        # Because this won't run unless we decoded version matches a file type, no need to check again
        self.dispatcher.add(FileObject(decoded_data))


class SymantecQB(SymantecBase):
    DESCRIPTION = "Symantec Quarantined QB"
    KEY = 0xB3

    @classmethod
    def identify(cls, file_object):
        """
        Identify a symantec Qurantined QBD or QBI File.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is a Symantec Quarantine File.
        """
        has_magic = False
        if not file_object.parent:
            # Get the first 32 decoded bytes
            header = bytes(datum ^ cls.KEY for datum in file_object.data[:32])
            # Because these files are simply xored together there is no good detection method,
            # compare the decoded header against various known file types we might care about
            for file_header in cls.FILE_HEADERS:
                has_magic = has_magic or re.search(file_header, header)

        return has_magic

    def run(self):
        """
        Revert the file back to its original state.
        """
        decrypted_data = bytes(datum ^ self.KEY for datum in self.file_object.data)
        self.dispatcher.add(FileObject(decrypted_data))


class AhnLab(Parser):
    DESCRIPTION = "AhnLab"
    KEY = bytearray((k ^ 0xFF) for k in b"v3backup!@#$%^&)")

    @classmethod
    def identify(cls, file_object):
        """
        Identify an AhnLab Qurantined.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is a AhnLab Quarantine File.
        """
        return file_object.data.startswith(b"AhnLab Inc. 2006")

    def run(self):
        file_data = self.file_object.data
        data_offset = int.from_bytes(file_data[0x58:0x5C], "little") + 0x58
        data = file_data[data_offset:]
        decrypted_data = bytes(d ^ k for (d, k) in zip(data, cycle(self.KEY)))
        self.dispatcher.add(FileObject(decrypted_data))


class Avast(Parser):
    DESCRIPTION = "Avast"
    KEY = bytearray(
        (0x33, 0xB6, 0x59, 0x83, 0x8B, 0x43, 0x75, 0xFB, 0x35, 0xB6, 0x8A, 0x37, 0xAE, 0x29, 0x16, 0x47, 0xA2, 0x51,
         0x41, 0x4F, 0x69, 0x9A, 0x07, 0xF5, 0xF1, 0x69, 0x80, 0x89, 0x60, 0x15, 0x8E, 0xF6, 0xB2, 0x3B, 0x89, 0xC4,
         0x9F, 0xFF, 0x65, 0x2E, 0x36, 0xD3, 0xF2, 0x10, 0xEA, 0x76, 0x88, 0xAD, 0x19, 0x39, 0x44, 0xEF, 0x7E, 0xBC,
         0xAF, 0xA0, 0x26, 0x7D, 0x83, 0xC9, 0x13, 0xC7, 0xBD, 0xE1, 0x16, 0xEB, 0x27, 0x69, 0x2C, 0x17, 0xE2, 0xF9,
         0xF8, 0x8A, 0x7F, 0x6E, 0x6F, 0xEB, 0x16, 0x16, 0x60, 0x48, 0x86, 0x12, 0xC5, 0x9A, 0x91, 0x6B, 0xB3, 0xA2,
         0x71, 0x38, 0xC6, 0x2F, 0x4E, 0x05, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71,
         0xD3, 0xF6, 0x08, 0x0F, 0x0F, 0x08, 0xE3, 0x73, 0xA6, 0xA0, 0x2E, 0x02, 0x76, 0x7F, 0x97, 0x9F, 0x8E, 0x5D,
         0x80, 0xDE, 0x75, 0xDB, 0x41, 0x31, 0x62, 0xCC, 0x68, 0x73, 0x79, 0x33, 0x3F, 0xE8, 0xDC, 0xCD, 0xF5, 0x9A,
         0x9E, 0x1F, 0x21, 0xD7, 0x97, 0xDF, 0x66, 0xC8, 0x50, 0x0F, 0xBD, 0x2E, 0x35, 0x11, 0x1D, 0x77, 0xE1, 0x62,
         0xA1, 0xCA, 0x4C, 0xC7, 0x4C, 0xE3, 0xB5, 0x5C, 0x86, 0xD5, 0xE4, 0xCE, 0xF5, 0xD3, 0xCF, 0xA5, 0xE6, 0x54,
         0xA7, 0x2E, 0x7B, 0xA8, 0xBA, 0xA5, 0x8B, 0x02, 0x15, 0x4E, 0xEE, 0xD6, 0xB1, 0xE4, 0xEB, 0x46, 0x9B, 0x8B,
         0xB5, 0x26, 0xCA, 0x88, 0xAF, 0xE6, 0xF8, 0x56, 0xFA, 0x6F, 0x39, 0x48, 0x6B, 0xFA, 0xF0, 0x7A, 0x4F, 0xC4,
         0xE3, 0xA7, 0x2C, 0x62, 0x44, 0x84, 0x39, 0xE3, 0xDD, 0xED, 0xA4, 0xF6, 0xFD, 0x4E, 0xB8, 0x92, 0x0C, 0x1D,
         0x3A, 0x78, 0x7E, 0xDD, 0x03, 0x3E, 0xD1, 0x7B, 0xE6, 0x2C, 0xBE, 0xD3, 0x87, 0x75, 0xD5, 0xE1, 0x2F, 0x07,
         0x19, 0x37, 0x01, 0x40, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6,
         0x08, 0x0F, 0x6F, 0x43, 0x73, 0x1E, 0x13, 0x7D, 0x30, 0x3D, 0xFA, 0x30, 0x5B, 0x81, 0x68, 0x7C, 0xF9, 0xEA,
         0x52, 0xA9, 0xE3, 0xF4, 0x28, 0x8C, 0x01, 0x38, 0xAF, 0xE9, 0xD0, 0xA8, 0x2C, 0xD4, 0x62, 0xE8, 0x41, 0xA5,
         0xB1, 0x71, 0xC1, 0x2E, 0x2B, 0x79, 0xE3, 0xFF, 0xA8, 0x24, 0x12, 0xAF, 0x89, 0xA7, 0x9A, 0x6D, 0x73, 0xE6,
         0xCD, 0xE8, 0x11, 0x75, 0xFF, 0xE6, 0x70, 0x8A, 0x8A, 0xE5, 0x4F, 0x08, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF,
         0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0x32, 0x91, 0x88, 0xC4, 0x01, 0x19, 0x08, 0x2F,
         0x48, 0x84, 0xD2, 0x30, 0x5F, 0x56, 0xB7, 0x15, 0xAF, 0x90, 0x52, 0x1A, 0x1B, 0x16, 0xCD, 0x84, 0xEA, 0x6E,
         0xE6, 0x9A, 0x89, 0xC3, 0x48, 0xF0, 0x9C, 0xBD, 0xC3, 0x95, 0x18, 0xF9, 0x30, 0xC0, 0xD2, 0x50, 0x21, 0x66,
         0x41, 0xA1, 0x82, 0xB6, 0xA5, 0x57, 0x64, 0x6A, 0x96, 0x96, 0x8C, 0x1D, 0x94, 0x74, 0xD6, 0x2F, 0x77, 0x12,
         0x4B, 0x3F, 0xF6, 0x5C, 0x33, 0x8C, 0xA1, 0x21, 0x7F, 0x3C, 0x5B, 0x9D, 0xCB, 0x2C, 0x1F, 0x6E, 0xA0, 0xAA,
         0x8F, 0xEA, 0x09, 0x90, 0x4B, 0x47, 0xD0, 0x27, 0xA7, 0x9A, 0x89, 0x78, 0x9E, 0x20, 0xF2, 0x8E, 0x81, 0x12,
         0xDD, 0x09, 0xDF, 0x0D, 0x7F, 0x4B, 0x76, 0xB5, 0x89, 0xA7, 0x69, 0x7F, 0x42, 0x15, 0x22, 0x7F, 0xE9, 0x59,
         0xA6, 0xFD, 0x59, 0x2A, 0x8E, 0x58, 0x25, 0xCC, 0xFC, 0x63, 0x88, 0xC5, 0x64, 0x22, 0x71, 0x48, 0x2E, 0xC9,
         0xC8, 0xE3, 0xAE, 0xB4, 0x94, 0xBD, 0x73, 0x7A, 0x11, 0x04, 0x37, 0x0C, 0xF0, 0x04, 0x1D, 0x57, 0x62, 0x6F,
         0x65, 0xFE, 0x57, 0x97, 0xBA, 0x44, 0x81, 0xA7, 0x9D, 0xB4, 0xFC, 0x16, 0x3E, 0xB5, 0x05, 0xED, 0xD1, 0x67,
         0xD5, 0x2A, 0xE7, 0xCF, 0xD5, 0xD5, 0xD4, 0xC8, 0x4A, 0x1F, 0x85, 0x39, 0x79, 0xDF, 0xB7, 0x1E, 0x79, 0x75,
         0xC3, 0xB3, 0xB4, 0xCA, 0xA1, 0x3D, 0x38, 0x54, 0xC6, 0x42, 0x17, 0x3B, 0x57, 0x02, 0xB9, 0xDC, 0xDE, 0xC8,
         0x1F, 0x88, 0x0A, 0x9B, 0xB7, 0x4E, 0x95, 0x25, 0xCE, 0xC8, 0x3F, 0x02, 0x74, 0xFD, 0xB8, 0xA2, 0x5C, 0x12,
         0x1D, 0xD6, 0x25, 0x37, 0xE3, 0xD3, 0x6C, 0x2A, 0x9D, 0x8F, 0x93, 0x26, 0xE3, 0x3F, 0xC1, 0xD6, 0xBE, 0x14,
         0x8E, 0xEF, 0x08, 0xBC, 0x4D, 0xB4, 0xDE, 0xE9, 0x05, 0x14, 0x29, 0xA0, 0x3A, 0x57, 0xEB, 0x52, 0x47, 0xD7,
         0xDE, 0x1D, 0x42, 0x97, 0x51, 0xD3, 0x85, 0x0D, 0xA7, 0xF0, 0xF7, 0xD3, 0x30, 0xFC, 0x78, 0xC3, 0x77, 0x73,
         0x8D, 0x8E, 0xF4, 0x18, 0x2D, 0x33, 0xC6, 0xE9, 0x9F, 0xE3, 0x1E, 0x81, 0x8A, 0x99, 0xDB, 0x26, 0x36, 0x8D,
         0xC1, 0x9D, 0xC0, 0x03, 0xF2, 0xB7, 0xBC, 0xA3, 0xBD, 0x22, 0xA8, 0xAF, 0x5D, 0x37, 0x1A, 0x3E, 0x1D, 0x9C,
         0x48, 0xD6, 0xA8, 0x0D, 0x21, 0xA6, 0x7A, 0xC0, 0x11, 0x37, 0xB3, 0x97, 0x5D, 0x50, 0x5A, 0xE8, 0xDC, 0xC6,
         0x7F, 0xF8, 0x16, 0x35, 0xED, 0x28, 0xEF, 0x51, 0xEE, 0x05, 0x43, 0x4F, 0x62, 0x15, 0xBE, 0x8B, 0x88, 0x0F,
         0x6C, 0xB0, 0x57, 0x69, 0x57, 0xB1, 0xFF, 0x76, 0x44, 0x64, 0x6C, 0xA3, 0x12, 0x9C, 0x54, 0x5C, 0x7B, 0xA2,
         0xC3, 0xAD, 0x26, 0x60, 0xDE, 0xF0, 0x5D, 0x23, 0xA3, 0x28, 0xF5, 0xAB, 0x88, 0x8B, 0x61, 0x85, 0x02, 0xC8,
         0xB3, 0x1D, 0x41, 0xD2, 0x30, 0x0B, 0x22, 0x2C, 0xEA, 0xD3, 0xC9, 0xEB, 0x3F, 0x23, 0x80, 0xDA, 0xC3, 0x84,
         0x0F, 0x6C, 0xD4, 0xCC, 0x1E, 0x5F, 0x6D, 0x22, 0xAC, 0x74, 0x1D, 0xD2, 0x09, 0x3A, 0x1B, 0x28, 0x08, 0xB6,
         0xED, 0x92, 0xE6, 0xA5, 0xAD, 0x9B, 0x84, 0x09, 0x40, 0xA2, 0x91, 0x6C, 0x1A, 0x90, 0x72, 0x8E, 0x54, 0x63,
         0x16, 0xA6, 0xDD, 0xFC, 0xF3, 0x23, 0x3E, 0x11, 0xF0, 0x0A, 0x32, 0x2C, 0x7B, 0x81, 0x60, 0xFF, 0x61, 0xF7,
         0x4A, 0x7A, 0x07, 0x10, 0x7B, 0xEB, 0x68, 0x90, 0x56, 0xD9, 0x6A, 0x3D, 0xDB, 0x22, 0x31, 0x65, 0xA2, 0x2D,
         0xA4, 0xB4, 0x82, 0xC3, 0xEA, 0x7C, 0xF8, 0x24, 0x88, 0x80, 0x4A, 0x2F, 0xA9, 0x56, 0x55, 0xBD, 0x2E, 0x88,
         0xF7, 0x26, 0x36, 0x1D, 0x18, 0x65, 0xF9, 0xB7, 0xB3, 0x63, 0xB2, 0xE6, 0x0F, 0x76, 0x5A, 0xF4, 0x3A, 0x0A,
         0xB7, 0x6C, 0x11, 0x1F, 0x00, 0xCC, 0x31, 0x16, 0x5C, 0x6F, 0x51, 0x2C, 0xC4, 0xD9, 0xEE, 0xF6, 0xC4, 0xF9,
         0x2D, 0x81, 0x6E, 0xF1, 0xCD, 0x13, 0x44, 0x77, 0x43, 0x67, 0xBD, 0xB6, 0x14, 0x91, 0xD9, 0x32, 0xB2, 0xFF,
         0x96, 0x46, 0x26, 0xC4, 0x62, 0x1E, 0x6C, 0xB4, 0x3A, 0xC5, 0xBC, 0xB8, 0xEA, 0x40, 0x7B, 0x72, 0x7F, 0xF8,
         0x82, 0xDC, 0x67, 0x47, 0xF2, 0x38, 0x66, 0xC6, 0x73, 0xD5, 0xA9, 0x8F, 0x48, 0xE5, 0x10, 0x54, 0x7B, 0x43,
         0xFD, 0xEE, 0x9B, 0x1C, 0x12, 0x55, 0x15, 0x20, 0xF8, 0x23, 0x0F, 0x95, 0x85, 0xCB, 0x5F, 0xF5, 0x1B, 0x57,
         0x4F, 0x25, 0x0B, 0xCA, 0xA1, 0x6A, 0x6D, 0xF2, 0x69, 0xD8, 0x8E, 0x40, 0xEC, 0x64, 0x60, 0x93, 0x43, 0x3A,
         0x01, 0x77, 0xBC, 0x4B, 0x72, 0x73, 0x23, 0xF5, 0xD1, 0x0C, 0x70, 0x1B, 0x71, 0xE3, 0x95, 0xF9, 0x51, 0x7D,
         0xAB, 0x40, 0x46, 0xB3, 0xEC, 0x38, 0x2A, 0xFB, 0x9C, 0xB9, 0xA2, 0x69, 0xE4, 0x22, 0xBD, 0xB1, 0x6B, 0x08,
         0xA9, 0xE0, 0xE7, 0x66, 0x48, 0xF1, 0xB0, 0x2D, 0x6B, 0x53, 0x8B, 0x62, 0x54, 0xD4, 0xE8, 0x48, 0xDA, 0xFB,
         0xE0, 0x8C, 0x0B, 0x8A, 0x9D, 0xA4, 0x46, 0x88, 0x5B, 0x47, 0xE5, 0x2A, 0xF6, 0x84, 0xB3, 0xAE, 0x31, 0xF6,
         0xC5, 0x3F, 0x11, 0x59, 0x40, 0x7C, 0x66, 0x74, 0x8D, 0xA3, 0x1A, 0x13, 0x7E, 0xD6, 0x90, 0xCD, 0xC8, 0x97,
         0xAF, 0x33, 0xBA, 0xAD, 0xF6, 0xDE, 0x64, 0x41, 0xD3, 0xAE, 0x08, 0x00, 0xF6, 0x2F, 0x73, 0xD3, 0x21, 0x64,
         0x4E, 0xC5, 0x27, 0xFA, 0xB6, 0x62, 0x02, 0xB9, 0x37, 0xD2, 0x22, 0x9D, 0xE6, 0x9F, 0x08, 0x47, 0xFB, 0x05,
         0x1E, 0xB8, 0x2C, 0xE4, 0xC5, 0x92, 0x91, 0xAA, 0x50, 0xA3, 0xE4, 0x78, 0x80, 0xC5, 0xB0, 0x14, 0x4F, 0x90,
         0x11, 0x4D, 0x80, 0xEB, 0x20, 0x25, 0x3C, 0x3F, 0x03, 0xF6, 0xFC, 0xFE, 0xAE, 0xCD, 0x0F, 0x48, 0xF5, 0x90,
         0xE2, 0x9E, 0xC1, 0x6C, 0xA3, 0x33, 0xEB, 0xD9, 0xA4, 0xE9, 0x33, 0x0D, 0xE2, 0x5D, 0x4A, 0x48, 0xC9, 0xCE,
         0xF4, 0xDF, 0xE9, 0xD2, 0x8D, 0xDB, 0x3D, 0x2E, 0xE9, 0x0C, 0xBE, 0x8D, 0x36, 0x4B, 0xD3, 0xA9, 0xA7, 0xC9,
         0xE5, 0xB6, 0xFB, 0x83, 0x37, 0xD5, 0x65, 0x31, 0x61, 0x7F, 0x30, 0xDC, 0x4B, 0xB6, 0x30, 0xCD, 0x44, 0xE0,
         0x9B, 0x07, 0x4E, 0x00, 0x89, 0x80, 0xAC, 0xC9, 0xFB, 0x86, 0x4E, 0x78, 0xA2, 0x48, 0x17, 0x34, 0x67, 0x94,
         0x11, 0x60, 0x67, 0xFC, 0x61, 0x74, 0xC1, 0x6B, 0x40, 0x47, 0xA1, 0x22, 0x75, 0xCA, 0x56, 0x99, 0x0B, 0xBB,
         0x33, 0x07, 0xA2, 0x88, 0x78, 0xAE, 0xF6, 0x1B, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03,
         0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0x59, 0x40, 0xBA, 0x35, 0x0A, 0x34, 0x21, 0x3C, 0x8D, 0xDA, 0x9B, 0x9D,
         0x16, 0xCB, 0x22, 0x63, 0x27, 0xA4, 0x3A, 0xD1, 0x0E, 0xCB, 0x79, 0x87, 0x34, 0x89, 0xE9, 0x8F, 0x9B, 0x0E,
         0xC8, 0x05, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F,
         0xF4, 0x03, 0xFC, 0xBE, 0x12, 0xEB, 0xDA, 0x57, 0x28, 0x28, 0x26, 0x43, 0xCF, 0xC4, 0x71, 0xBF, 0x60, 0x51,
         0x33, 0xBF, 0x3D, 0xC1, 0xB6, 0xBB, 0x18, 0x39, 0xCC, 0xE0, 0x8B, 0x66, 0x33, 0xC2, 0x82, 0x79, 0x3E, 0xFC,
         0x73, 0x7C, 0xD2, 0xD5, 0xF9, 0x7D, 0x83, 0xA6, 0x2C, 0x1B, 0x0B, 0x76, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF,
         0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0x2D, 0x72, 0x31, 0x49, 0xD9, 0x9F, 0xEB, 0x32,
         0x72, 0xDC, 0xE0, 0x59, 0xF8, 0xE2, 0xAA, 0xBB, 0x94, 0x48, 0xE3, 0x65, 0x89, 0x2B, 0xB6, 0xA4, 0xCA, 0x38,
         0x39, 0x82, 0x92, 0x1D, 0xED, 0x9E, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71,
         0xD3, 0xF6, 0x08, 0x0F, 0xE3, 0xAB, 0xE2, 0x85, 0x50, 0x08, 0xD1, 0xA8, 0x27, 0x88, 0x2D, 0x92, 0x65, 0x5D,
         0x30, 0xBF, 0xA1, 0x61, 0x69, 0xA1, 0x32, 0xAC, 0xC0, 0x68, 0x55, 0x5A, 0x82, 0x98, 0x2C, 0x2C, 0x02, 0x64,
         0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0x8D, 0x7E,
         0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0xAD, 0xF6, 0x5C, 0x7E,
         0xC3, 0x7B, 0x8A, 0x5C, 0xC3, 0xF6, 0x40, 0x28, 0x0F, 0x4E, 0x30, 0x0F, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF,
         0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0xFF, 0x1D, 0x50, 0x0E, 0xFE, 0x67, 0xF3, 0x44,
         0x9C, 0xDF, 0xEE, 0xC2, 0x4A, 0xFB, 0xCA, 0x04, 0x96, 0x16, 0x10, 0xF1, 0x30, 0x17, 0x4E, 0x7F, 0x4C, 0xF8,
         0x86, 0x53, 0x7C, 0x73, 0x15, 0x0A, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71,
         0xD3, 0xF6, 0x08, 0x0F, 0xF4, 0xAD, 0x23, 0x55, 0x42, 0x96, 0xED, 0xC0, 0x71, 0x75, 0x9B, 0x1C, 0x70, 0x17,
         0x07, 0x25, 0x0A, 0x83, 0x42, 0xEB, 0x81, 0xAB, 0x4B, 0x1B, 0xE5, 0x39, 0xA0, 0x9E, 0x98, 0xD7, 0x33, 0x67,
         0x3E, 0xF0, 0xFA, 0xBB, 0xB7, 0xBD, 0x71, 0xD1, 0x4D, 0x11, 0x52, 0xC3, 0x2B, 0x9C, 0xDA, 0x43, 0xA3, 0x48,
         0xEB, 0x27, 0x04, 0x0E, 0x6E, 0x0D, 0xF6, 0x42, 0xA5, 0xB7, 0xB3, 0xC5, 0x78, 0x45, 0xD8, 0x28, 0xCE, 0x52,
         0x22, 0xC5, 0x82, 0x0F, 0x1F, 0xA9, 0x0F, 0xCE, 0xD9, 0xD2, 0x1F, 0xA5, 0x8A, 0xFE, 0x93, 0xBC, 0x02, 0x18,
         0x2F, 0x7C, 0x27, 0x5C, 0x67, 0xC1, 0xE0, 0xF6, 0xBD, 0x94, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2,
         0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0x53, 0xF1, 0x82, 0xF9, 0xDF, 0x59, 0xF3, 0x3C, 0xA8, 0xAF,
         0x18, 0x68, 0xFF, 0xD8, 0x1B, 0x19, 0x19, 0xCF, 0x42, 0xEB, 0xE2, 0xC5, 0x6B, 0x22, 0x90, 0x4A, 0x34, 0xE3,
         0x36, 0x8A, 0x49, 0xE1, 0xEA, 0x79, 0xFA, 0x1F, 0x79, 0x8A, 0x3C, 0xE3, 0xC4, 0x8C, 0x46, 0x49, 0x56, 0x7B,
         0x80, 0x8F, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2, 0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F,
         0xF9, 0x85, 0x6C, 0xC3, 0x6E, 0x9A, 0x3D, 0x9C, 0x3D, 0xAC, 0xD0, 0x9C, 0x11, 0x8F, 0x82, 0x1C, 0x45, 0xC6,
         0xF3, 0xE6, 0xD6, 0x12, 0x6A, 0xFC, 0x3D, 0x61, 0x6C, 0xED, 0xBB, 0x7A, 0x08, 0xB5, 0x13, 0x9D, 0x43, 0x91,
         0x75, 0x69, 0x20, 0x4D, 0xE2, 0xBE, 0xA7, 0x30, 0x93, 0xF7, 0xA7, 0xC1, 0x5A, 0x6C, 0x35, 0x53, 0x86, 0x03,
         0x32, 0x26, 0x6B, 0xD8, 0xD8, 0xA7, 0xDC, 0xE7, 0xF3, 0x42, 0x8D, 0x7E, 0xE8, 0x98, 0x30, 0xFF, 0x6F, 0xB2,
         0xBD, 0x03, 0x35, 0x71, 0xD3, 0xF6, 0x08, 0x0F, 0xBE, 0x85, 0x0C, 0x45, 0x9E, 0x70, 0xC6, 0x84, 0x15, 0xA3,
         0xC4, 0x2E, 0x60, 0xAD, 0xC9, 0xB9, 0x99, 0x95, 0x4A, 0xBD, 0xD9, 0x43, 0xA5, 0x56, 0x28, 0xE9, 0xCF, 0x96,
         0x8A, 0x05, 0x40, 0xCF, 0x34, 0x64, 0x71, 0x69, 0xCA, 0x92, 0x9F, 0x1F, 0x01, 0x48, 0xDB, 0xFD, 0x73, 0x29,
         0x41, 0x94, 0x9A, 0x92, 0xEF, 0x74, 0x71, 0xA5, 0xE0, 0xF9, 0x7D, 0x11, 0xCF, 0x6B, 0x5A, 0x81, 0x74, 0x77,
         0x1D, 0x25, 0xC1, 0x18, 0x9B, 0xDE, 0xF3, 0x88, 0x18, 0x88, 0x2B, 0x3A, 0xD7, 0xD4, 0xDA, 0x57, 0xF6, 0xE0,
         0xE1, 0xCC, 0xD2, 0x83, 0x61, 0xEA, 0xDF, 0xBE, 0xC6, 0xEE, 0x82, 0x29, 0x31, 0x03, 0xC3, 0xB7, 0x44, 0xD2,
         0x8A, 0x00, 0x1E, 0x4F, 0x40, 0x74, 0x56, 0x72, 0xCA, 0x4F, 0xAD, 0x44, 0x6C, 0x20, 0xD7, 0xA3, 0xA2, 0xF4,
         0xDA, 0x32, 0x51, 0xCE, 0xAA, 0x8D, 0x7F, 0xD1, 0xD1, 0xFB, 0xC0, 0xA3, 0x3E, 0xB9, 0xB4, 0x8E, 0x54, 0xDA,
         0xE9, 0x7A, 0xD6, 0xBA, 0xFD, 0x54, 0xB7, 0x50, 0x2D, 0x02, 0xA4, 0xC9, 0x8A, 0x1E, 0x82, 0xB5, 0x11, 0x8E,
         0x38, 0xAA, 0xDE, 0x2F, 0x84, 0x64, 0x44, 0x03, 0xA6, 0xF9, 0x4C, 0xD7, 0x6A, 0x8F, 0x25, 0x6F, 0xD6, 0x7B,
         0xF8, 0x47, 0x71, 0x33, 0x28, 0x2A, 0x25, 0x01, 0x1D, 0xEA, 0x8C, 0xBA, 0x62, 0x65, 0xE0, 0x5B, 0x5E, 0xA8,
         0x6F, 0x5F, 0xEE, 0xA9, 0x09, 0x6B, 0xB2, 0x35, 0xBB, 0x07, 0x40, 0x8A, 0x99, 0xB9, 0xBB, 0xDE, 0xEB, 0x7B,
         0x92, 0x73, 0xC5, 0x7A, 0x5E, 0x71, 0xA9, 0x5A, 0x3A, 0xC1, 0x76, 0x74, 0x4F, 0x0C, 0x37, 0x9A, 0x9C, 0x66,
         0xDE, 0x41, 0x40, 0xB3, 0xBF, 0x58, 0x25, 0x88, 0x05, 0x0B, 0x3B, 0x62, 0x06, 0x89, 0x12, 0x5E, 0x2C, 0x82,
         0x7E, 0x18, 0x93, 0x60, 0xAE, 0x71, 0x6A, 0x79, 0x1C, 0x5F, 0xB6, 0x70, 0x7C, 0xEF, 0x16, 0x3E, 0x6C, 0x9F,
         0x88, 0x0F, 0x6C, 0x39, 0xE9, 0xED, 0xCB, 0x01, 0x3C, 0x27, 0x08, 0x78, 0xFA, 0x60, 0xB5, 0xD4, 0x14, 0xAB,
         0x10, 0xB0, 0xE4, 0xA2, 0x8F, 0x80, 0x4F, 0x6E, 0xB7, 0x45, 0xCB, 0x81, 0x84, 0x86, 0x83, 0xFE, 0x15, 0x8B,
         0x21, 0x59, 0xD7, 0x83, 0x2A, 0x4F, 0x68, 0x5B, 0x61, 0xC1, 0x49, 0x55, 0xA4, 0xB3, 0x03, 0xD8, 0x2C, 0x7F,
         0x2F, 0xDD, 0xED, 0x8E, 0xD0, 0xF2, 0xBD, 0x12, 0x4D, 0x1E, 0x53, 0x4B, 0x1D, 0x5F, 0x08, 0x25, 0x9B, 0x3D,
         0xA2, 0xA1, 0xEA, 0x44, 0x99, 0x6D, 0x71, 0x3B, 0x48, 0x49, 0xFD, 0x2E, 0x5A, 0x71, 0xEC, 0xA0, 0x86, 0x13,
         0x26, 0x02, 0x0B, 0x49, 0xF8, 0x10, 0xD6, 0x60, 0xAA, 0x6B, 0x02, 0xDF, 0xE5, 0xAA, 0xBA, 0x26, 0xFA, 0xCD,
         0x50, 0xDE, 0xE3, 0x31, 0x93, 0x53, 0x4E, 0x0F, 0x65, 0x77, 0xF9, 0xBF, 0x29, 0x95, 0x86, 0xB9, 0x7E, 0x83,
         0x45, 0xDB, 0xF2, 0x4A, 0x05, 0x3C, 0x2D, 0xDD, 0x58, 0x6A, 0xEA, 0xFE, 0x53, 0x77, 0x7F, 0x2F, 0x05, 0xF7,
         0x29, 0x3C, 0x17, 0x69, 0x36, 0xB8, 0xDB, 0x33, 0x13, 0x8D, 0x76, 0xFD, 0xA5, 0xBC, 0x65, 0xAE, 0xFD, 0xC7,
         0xED, 0x48, 0xC9, 0x4C, 0x30, 0x40, 0xD6, 0x38, 0x5A, 0xCE, 0x1A, 0x04, 0x8F, 0xF8, 0xEE, 0x44, 0x60, 0xAD,
         0x3A, 0x94, 0xA3, 0x8C, 0x26, 0x63, 0x5B, 0xC1, 0x18, 0xC6, 0x73, 0x7F, 0x12, 0x20, 0x04, 0xB5, 0x67, 0x5A,
         0x29, 0xB5, 0xE9, 0x8D, 0x28, 0x92, 0x05, 0xDA, 0x67, 0xA4, 0x9F, 0xB8, 0xCC, 0x35, 0x0F, 0x2B, 0xF1, 0xB9,
         0x50, 0x46, 0xEF, 0xA0, 0x7D, 0xC6, 0x42, 0xEA, 0x38, 0x4E, 0xE8, 0xB4, 0xE2, 0xB1, 0x4C, 0x65, 0x18, 0xD7,
         0x18, 0x73, 0xD9, 0x6D, 0x28, 0x24, 0x6D, 0xEB, 0x4D, 0xF7, 0xBD, 0x31, 0xAC, 0x8C, 0x80, 0xC6, 0xF8, 0x72,
         0x5B, 0xAE, 0xF0, 0x16, 0x88, 0x25, 0x9A, 0xEF, 0xB2, 0xE5, 0x8E, 0xAD, 0xD0, 0xC2, 0x8D, 0x96, 0x80, 0xB6,
         0x00, 0xD0, 0x08, 0x2C, 0x8C, 0xE1, 0x0D, 0xC3, 0x57, 0xC0, 0x2B, 0xF7, 0xA5, 0x5A, 0x82, 0xFF, 0x69, 0xAD,
         0x01, 0xE6, 0x42, 0xCD, 0x38, 0x4D, 0x4E, 0xB6, 0x6D, 0xC6, 0xA4, 0x6A, 0x53, 0x5C, 0x13, 0x92, 0x05, 0x36,
         0x02, 0xB3, 0x98, 0x8E, 0x9E, 0xBB, 0x48, 0x74, 0xE3, 0x2E, 0x86, 0x84, 0xFF, 0x80, 0xB4, 0xEE, 0x94, 0x3F,
         0xBB, 0x37, 0xD7, 0xC3, 0xA8, 0x4B, 0x5A, 0x85, 0x5A, 0x6A, 0xB7, 0x4D, 0x4B, 0xB3, 0x47, 0x6F, 0x6C, 0x43,
         0xBA, 0x12, 0x38, 0x61, 0xA1, 0x61, 0x2C, 0x7B, 0x50, 0xFA, 0x3A, 0x05, 0x4C, 0x16, 0x0E, 0x2F, 0x88, 0x6C,
         0xCF, 0xAC, 0x4F, 0xB7, 0xD9, 0x0E, 0xC7, 0x55, 0x72, 0x9D, 0xF5, 0x64, 0xC4, 0xB8, 0x9A, 0x82, 0x18, 0xDA,
         0x51, 0x2C, 0x7F, 0x39, 0xC1, 0x17, 0xB9, 0xF4, 0x2F, 0x98, 0x3B, 0x55, 0x7B, 0x69, 0xCC, 0x83, 0xE3, 0xDE,
         0x75, 0x44, 0xFD, 0x55, 0x06, 0x73, 0x47, 0xC7, 0xEE, 0x47, 0xA2, 0x3B, 0x35, 0x17, 0xF0, 0xF3, 0x0B, 0xE8,
         0x01, 0x97, 0x77, 0x67, 0x28, 0x14, 0xA4, 0x3F, 0x96, 0x74, 0xA2, 0xA4, 0x35, 0xE3, 0x1F, 0x69, 0x41, 0x5A,
         0x89, 0x4F, 0x19, 0x06, 0xDF, 0xD7, 0x57, 0xC0, 0x54, 0x27, 0x72, 0x0F, 0x36, 0xE0, 0xF9, 0x57, 0xC0, 0x9A,
         0x56, 0xFD, 0x5C, 0x19, 0xC5, 0x79, 0x50, 0x88, 0xB8, 0x70, 0x57, 0xDB, 0x55, 0x98, 0xA8, 0x2F, 0xBB, 0xB8,
         0xAF, 0xF8, 0xE5, 0x33, 0x0D, 0x40, 0xBE, 0xBC, 0x3C, 0x7E, 0xC8, 0xFF, 0x11, 0x7C, 0x12, 0x39, 0x76, 0x12,
         0x9C, 0x8D, 0x9C, 0x95, 0x23, 0x6A, 0x1A, 0xE9, 0xAF, 0xD7, 0x9B, 0xD1, 0x8E, 0xA1, 0x32, 0x2F, 0x17, 0x4D,
         0x84, 0x9C, 0xCC, 0x8F, 0xA9, 0x14, 0xEC, 0x86, 0xCF, 0x1A, 0xFD, 0x1D, 0x45, 0x14, 0x5D, 0x9B, 0x0C, 0x63,
         0x3A, 0xF2, 0x84, 0x44, 0xAA, 0x4D, 0x69, 0x76, 0x15, 0x32, 0xD2, 0xCB, 0x6B, 0x03, 0x30, 0x9A, 0x6A, 0x64,
         0x66, 0x8C, 0x06, 0xEC, 0x70, 0xCC, 0x33, 0x92, 0xAA, 0x43, 0x09, 0x06, 0xE2, 0x52, 0x48, 0xE1, 0x76, 0x09,
         0x0F, 0x44, 0x7B, 0x40, 0x04, 0x7B, 0xD7, 0xBD, 0x8C, 0xEC, 0xBD, 0x8E, 0x16, 0xBF, 0x2F, 0x40, 0x48, 0x72,
         0x78, 0xA0, 0x45, 0x16, 0x90, 0x37, 0xAA, 0x7A, 0xCA, 0x8F, 0x7A, 0x6C, 0xAA, 0x68, 0xB3, 0xEB, 0x63, 0xE9,
         0x7F, 0xD0, 0xE7, 0x3A, 0x82, 0xFF, 0xEB, 0x45, 0x7F, 0x75, 0xDC, 0xB5, 0xF9, 0x6A, 0x27, 0x8E, 0x49, 0x39,
         0x0A, 0xAE, 0x98, 0x24, 0xC4, 0xEE, 0xF5, 0x80, 0xBC, 0xF5, 0x30, 0x4F, 0x7D, 0x5A, 0x0C, 0x79, 0x0F, 0x37,
         0x3D, 0x9C, 0x53, 0xD6, 0xEE, 0x95, 0xEA, 0x85, 0x54, 0xE6, 0x9A, 0x2A, 0xCF, 0x80, 0xDC, 0x14, 0x76, 0x8D,
         0x9C, 0x83, 0x1F, 0x40, 0xAC, 0x66, 0x27, 0x67, 0x69, 0xB6, 0xBC, 0x05, 0x7E, 0xE9, 0xC6, 0xCE, 0xB2, 0xD6,
         0xB1, 0xAC, 0x79, 0x20, 0x13, 0x24, 0xCE, 0xC4, 0xD1, 0x02, 0x6B, 0x88, 0xD4, 0x0D, 0xD9, 0x41, 0x9D, 0x94,
         0x71, 0xF2, 0x06, 0x15, 0xDF, 0x34, 0xF1, 0x4D, 0xB3, 0x02, 0x84, 0xAE, 0x83, 0x7D, 0xDC, 0xC8, 0xC3, 0x3E,
         0x53, 0x01, 0x5C, 0x1D, 0xB0, 0xFE, 0x84, 0xA3, 0x73, 0xA9, 0x6A, 0x8A, 0x13, 0xD3, 0xEF, 0x8B, 0xEA, 0x90,
         0x78, 0x87, 0xB5, 0xFF, 0xCE, 0x34, 0xF9, 0x78, 0x7B, 0xAD, 0xB8, 0x62, 0xDE, 0xB5, 0xB5, 0x6C, 0xF9, 0xB3,
         0x67, 0x2E, 0x83, 0x9D, 0x52, 0xBE, 0x44, 0xD1, 0x89, 0xD6, 0x2B, 0xE4, 0x58, 0x67, 0x9D, 0xAC, 0xC4, 0x25,
         0xEB, 0x50, 0x20, 0x1F, 0x2E, 0xF5, 0xE7, 0x38, 0x05, 0xEF, 0x70, 0x9C, 0xF3, 0x0C, 0x4A, 0xB4, 0x47, 0xD4,
         0x1C, 0x89, 0x9C, 0xD3, 0x09, 0x6A, 0x97, 0xCC, 0xF3, 0xC6, 0x85, 0x4A, 0xB3, 0x0F, 0xAF, 0xF0, 0x0A, 0xC0,
         0x40, 0xE0, 0x22, 0x19, 0xEB, 0x81, 0x63, 0x3E, 0x5D, 0x36, 0x28, 0x63, 0xCD, 0x48, 0xBB, 0xB5, 0x3F, 0x52,
         0xD2, 0xAB, 0x27, 0x01, 0xC4, 0x75, 0x67, 0xF5, 0xC2, 0x89, 0x43, 0x48, 0xCF, 0x4B, 0x8C, 0xB2, 0xED, 0x11,
         0x87, 0x5F, 0x16, 0xFF, 0xC7, 0xDC, 0x77, 0x4A, 0xFE, 0xEA, 0xEE, 0xC4, 0x9D, 0x9F, 0xB4, 0xA2, 0x3A, 0x01,
         0xD3, 0x20, 0xEB, 0x95, 0x84, 0xAD, 0x4C, 0xCC, 0xD9, 0x12, 0xA1, 0x4B, 0xC7, 0xC3, 0x60, 0x26, 0xBA, 0xDB,
         0x26, 0x9F, 0xF1, 0x3A, 0x66, 0xE1, 0x8C, 0x06, 0xD1, 0x47, 0x0B, 0x77, 0x84, 0xAA, 0xEE, 0x8A, 0x3F, 0xCB,
         0x51, 0x8E, 0x15, 0x71, 0xB7, 0x45, 0x79, 0x8C, 0x34, 0xC3, 0xB2, 0x56, 0xE5, 0xC4, 0x24, 0xAF, 0x05, 0x3F,
         0x56, 0x86, 0x87, 0xAE, 0xA2, 0x96, 0xA4, 0x78, 0xB2, 0x29, 0x34, 0xA5, 0xE3, 0x02, 0x52, 0x5A, 0xD6, 0x5F,
         0x8F, 0xE7, 0x55, 0xE8, 0x89, 0xCA, 0x1F, 0xDD, 0x29, 0x81, 0x80, 0xBA, 0xA0, 0x0C, 0x28, 0xFA, 0x85, 0x8A,
         0xAC, 0xA3, 0x8F, 0x48, 0xB5, 0xC2, 0xB8, 0xD1, 0x87, 0x0F, 0x89, 0x5B, 0xA1, 0x94, 0x2C, 0xE4, 0x05, 0xC6,
         0x60, 0x94, 0x57, 0xC2, 0x69, 0x53, 0x01, 0xA9, 0xEF, 0xC4, 0x3D, 0x92, 0x21, 0x0D, 0x78, 0x2C, 0xFE, 0x58,
         0x65, 0xBB, 0xB7, 0x0A, 0x39, 0x6A, 0x20, 0xC4, 0xAE, 0x4B, 0xF0, 0x57, 0xEE, 0x0A, 0x59, 0xA0, 0x71, 0xD1,
         0xBF, 0xB9, 0x39, 0xC7, 0x12, 0x3C, 0xAA, 0xCA, 0xFB, 0xCE, 0x0F, 0x43, 0x48, 0xDD, 0xA7, 0xB5, 0x0D, 0x25,
         0xF3, 0x88, 0xA7, 0xD4, 0xDA, 0xDC, 0x6B, 0xBE, 0x18, 0x36, 0x09, 0x82, 0xE8, 0x99, 0x74, 0xD1, 0x83, 0xFF,
         0x94, 0x2B, 0xA5, 0xD9, 0xC0, 0x8E, 0xBF, 0x0C, 0xF3, 0xAF, 0xCE, 0x06, 0x4B, 0x86, 0x4D, 0x2E, 0xB8, 0xAA,
         0x0C, 0xA5, 0x0C, 0xC3, 0x38, 0xA8, 0x60, 0x0E, 0xD9, 0x58, 0x71, 0x10, 0x00, 0x37, 0x99, 0x81, 0x9D, 0xB8,
         0x4F, 0xEF, 0x74, 0xAC, 0x37, 0xBC, 0x57, 0x53, 0x4E, 0xAF, 0xAE, 0x87, 0x78, 0xFB, 0x82, 0x18, 0x60, 0x78,
         0x99, 0x98, 0x16, 0x01, 0x41, 0x42, 0x5E, 0xDE, 0xDF, 0xA9, 0xBD, 0x7B, 0x93, 0xDC, 0xC4, 0xB8, 0x9F, 0x4E,
         0xD5, 0x8A, 0x20, 0x15, 0x49, 0x0C, 0xCF, 0x0A, 0x2E, 0x6E, 0xDE, 0xCB, 0x67, 0x59, 0x1B, 0x1B, 0xB8, 0x24,
         0xB0, 0xB0, 0xE4, 0x13, 0x1A, 0x8E, 0x4E, 0x39, 0xF4, 0x52, 0x69, 0xE3, 0xDC, 0x95, 0xC0, 0xC8, 0xC0, 0xB3,
         0x96, 0xE2, 0x72, 0x99, 0xA1, 0xE9, 0xFD, 0xC4, 0x9E, 0xFA, 0x43, 0xD3, 0x62, 0x17, 0x6B, 0x54, 0x35, 0x7D,
         0xB1, 0xBF, 0xA9, 0xB3, 0x03, 0x18, 0x36, 0x73, 0xE2, 0xC7, 0xC2, 0xD9, 0xB2, 0x35, 0x6B, 0x7D, 0x9F, 0x55,
         0x38, 0xF9, 0xA1, 0x61, 0xA0, 0x99, 0x34, 0x02, 0x39, 0x29, 0x21, 0x97, 0xDF, 0x87, 0xDA, 0xCB, 0x87, 0x75,
         0x4F, 0xFA, 0x57, 0x54, 0x69, 0x3A, 0x97, 0x24, 0xFE, 0xBE, 0x2F, 0xB4, 0x7B, 0x12, 0xC3, 0xA7, 0x0A, 0xE9,
         0x39, 0x9A, 0x2C, 0x08, 0x6A, 0x05, 0x66, 0x0F, 0xD3, 0x38, 0xA5, 0xD9, 0xC8, 0x83, 0x2A, 0x6B, 0xC5, 0x44,
         0xE3, 0x7B, 0x01, 0xAB, 0x32, 0x96, 0x8C, 0x23, 0x2A, 0x14, 0x7C, 0x3C, 0xBB, 0x46, 0x51, 0x04, 0x12, 0xB6,
         0x1B, 0xB6, 0xE3, 0xF0, 0xAD, 0x23, 0xEB, 0x67, 0xF8, 0xB9, 0x95, 0xC1, 0x98, 0x55, 0xE3, 0x75, 0xA7, 0x1A,
         0x7C, 0x3A, 0xB8, 0x9D, 0xA7, 0x12, 0xDF, 0xFC, 0xA0, 0x1A, 0xBB, 0x5C, 0xC8, 0x69, 0x95, 0x5A, 0x67, 0xE3,
         0x5A, 0xFC, 0x14, 0x3F, 0x17, 0x6E, 0x54, 0x0A, 0x80, 0xA9, 0x51, 0xE8, 0x41, 0x20, 0xFD, 0x58, 0x12, 0x31,
         0x45, 0xD3, 0x50, 0xF4, 0x46, 0x67, 0x75, 0x4B, 0x21, 0x3C, 0x57, 0x6D, 0xDD, 0x7C, 0xAC, 0xE0, 0xA7, 0x86,
         0xC9, 0xF8, 0xA7, 0x09, 0xB5, 0x45, 0x1C, 0xE9, 0xD5, 0xB5, 0xC0, 0xC5, 0x67, 0x2F, 0xE9, 0x34, 0xA1, 0x0D,
         0xCD, 0xB4, 0xE5, 0x3D, 0xB2, 0x41, 0x9C, 0x72, 0x4C, 0x88, 0x22, 0x0F, 0x48, 0xE6, 0x78, 0x3C, 0xCD, 0xEF,
         0x06, 0x70, 0x10, 0xBF, 0x66, 0x14, 0xE0, 0xA9, 0x2F, 0xBA, 0xF5, 0x96, 0x47, 0xB0, 0x79, 0x59, 0x98, 0x0A,
         0x9B, 0x16, 0xD3, 0xB1, 0x11, 0xB1, 0xBD, 0xDA, 0x81, 0xDB, 0x4F, 0x56, 0xBC, 0x68, 0x15, 0xA8, 0xBA, 0x10,
         0x0B, 0xD1, 0xBE, 0x76, 0x2C, 0x4D, 0xA8, 0x4F, 0x3F, 0xC8, 0x9E, 0xA5, 0xAE, 0x88, 0x6B, 0x7D, 0x11, 0xFE,
         0x89, 0xFC, 0x17, 0x0F, 0x0C, 0x32, 0x51, 0xF4, 0xDB, 0x3E, 0x94, 0x55, 0xC6, 0xDA, 0xED, 0xD5, 0x0F, 0x87,
         0xB9, 0x7F, 0x33, 0xC3, 0xD1, 0x0B, 0x83, 0x63, 0x32, 0x72, 0x36, 0xED, 0xA1, 0x2A, 0xF8, 0x6E, 0x88, 0x42,
         0xC4, 0x94, 0xBB, 0x00, 0x64, 0xC1, 0x72, 0x25, 0x36, 0x53, 0xBB, 0x71, 0x02, 0x68, 0x78, 0xC1, 0x7A, 0x5F,
         0x70, 0xD0, 0x8D, 0x15, 0x94, 0xC0, 0x90, 0xA7, 0x81, 0x4F, 0x87, 0x30, 0x49, 0x44, 0xC8, 0x96, 0xED, 0x9C,
         0x6E, 0xD4, 0x9D, 0x2A, 0x81, 0x73, 0xD3, 0x89, 0x8C, 0x32, 0x3C, 0xAD, 0xAC, 0x0B, 0x99, 0x7A, 0xD8, 0x25,
         0x94, 0x77, 0xB0, 0x21, 0x24, 0x4E, 0xE6, 0x47, 0x6C, 0xB1, 0x6B, 0x87, 0xF9, 0x76, 0x95, 0x65, 0x2D, 0x05,
         0x03, 0x55, 0x24, 0x1F, 0x6B, 0xAF, 0xC2, 0x6B, 0x17, 0xAC, 0x5F, 0x0C, 0x74, 0x06, 0xAB, 0xDD, 0x04, 0x30,
         0xC8, 0xD6, 0x4F, 0xD1, 0xE8, 0xA2, 0xF4, 0xB1, 0xE0, 0x47, 0x19, 0x00, 0x13, 0x83, 0x87, 0xEB, 0xC7, 0xD9,
         0xAB, 0x6C, 0x57, 0xDA, 0x00, 0xD4, 0x9D, 0x4D, 0x6D, 0x66, 0xC7, 0x51, 0xB5, 0xCF, 0x1F, 0x91, 0x04, 0x16,
         0x30, 0xCC, 0xCE, 0x7B, 0xD2, 0x69, 0x77, 0xED, 0xCC, 0x07, 0x6D, 0xEB, 0x63, 0x2B, 0x99, 0x16, 0x14, 0x1E,
         0x0D, 0x1D, 0xDA, 0x2C, 0x98, 0x16, 0xDF, 0xDC, 0xD4, 0x92, 0xD3, 0xCC, 0x6A, 0x35, 0x80, 0xF1, 0xCC, 0x0F,
         0xF1, 0xF7, 0x43, 0xE6, 0x82, 0xC6, 0xF5, 0x6A, 0x2D, 0x16, 0xCC, 0xFB, 0x28, 0x23, 0xE0, 0x27, 0xB5, 0xC1,
         0x83, 0x88, 0x17, 0xB2, 0x9E, 0x24, 0xBD, 0xA6, 0x17, 0x3B, 0xBD, 0xF8, 0xFB, 0x96, 0xC4, 0x77, 0x93, 0x2E,
         0x51, 0x45, 0xB1, 0x45, 0x97, 0x3A, 0x08, 0x78, 0xA0, 0x34, 0x6B, 0x4B, 0x5C, 0xDF, 0xFE, 0x40, 0x04, 0x8A,
         0xE9, 0xFB, 0xCC, 0x6C, 0xD6, 0x56, 0x28, 0x89, 0x7F, 0xDE, 0x2A, 0xC3, 0xD9, 0x92, 0x8F, 0x22, 0x03, 0x1B,
         0xEA, 0x22, 0xAB, 0x99, 0x18, 0x5B, 0x47, 0xC3, 0x06, 0x02, 0xBC, 0xBB, 0x42, 0xB4, 0x2C, 0x05, 0x61, 0x97,
         0x86, 0x9B, 0x77, 0x91, 0x5B, 0x48, 0xAC, 0x68, 0x23, 0x01, 0x62, 0x7F, 0xA5, 0x7F, 0xF0, 0x7B, 0x6F, 0x04,
         0x37, 0x91, 0x13, 0xF4, 0xEB, 0x3C, 0x2E, 0x2C, 0x9A, 0xE6, 0x1F, 0xB1, 0x5F, 0xAF, 0xD4, 0x8A, 0xDB, 0x77,
         0xCC, 0x0A, 0xA7, 0x7C, 0xC4, 0x46, 0x02, 0xA5, 0x8F, 0x81, 0xEF, 0x92, 0x39, 0xB0, 0x8C, 0xEA, 0xFA, 0x4C,
         0x77, 0x30, 0x21, 0xE8, 0x6D, 0xA0, 0x4C, 0x0A, 0x65, 0xFA, 0x20, 0xD1, 0x7F, 0xCE, 0x69, 0x11, 0x1F, 0x63,
         0xFB, 0x70, 0xCB, 0x42, 0xE1, 0xB5, 0x09, 0xB2, 0x02, 0x83, 0x67, 0xE8, 0x60, 0x20, 0x34, 0x7A, 0x59, 0xC0,
         0x7A, 0x7A, 0xC1, 0x8D, 0x83, 0x0E, 0x90, 0x3A, 0x72, 0xAF, 0x6C, 0x9F, 0xA5, 0x95, 0x1B, 0x1D, 0x2D, 0x98,
         0x52, 0xDF, 0xCF, 0x55, 0x2A, 0xAD, 0x13, 0xD2, 0x38, 0xA5, 0x66, 0xBC, 0x69, 0xA0, 0xA3, 0x37, 0x7D, 0x0D,
         0x70, 0x63, 0x17, 0x49, 0x43, 0xA1, 0x0D, 0xDA, 0xAE, 0xD1, 0x02, 0xFC, 0x0F, 0x6E, 0x7F, 0x26, 0xD8, 0x4C,
         0x16, 0xAB, 0xB8, 0x6A, 0x64, 0x03, 0x69, 0xF4, 0x32, 0x69, 0xCD, 0x17, 0x85, 0xF9, 0xCE, 0xD2, 0xBE, 0xAE,
         0x9C, 0x59, 0xD8, 0x50, 0xF1, 0x93, 0x1C, 0x06, 0xC6, 0x32, 0x61, 0xC2, 0x54, 0x85, 0x88, 0xF3, 0x77, 0xFD,
         0xA3, 0x8D, 0x82, 0x94, 0x63, 0xB4, 0xCA, 0x4C, 0x47, 0xE5, 0xB1, 0x02, 0xA4, 0x53, 0xBB, 0xDF, 0x43, 0xE7,
         0x72, 0xD7, 0xE1, 0xD0, 0x18, 0x06, 0xCE, 0x63, 0x63, 0x8A, 0x96, 0xAF, 0x1F, 0xEA, 0xCD, 0x9B, 0x74, 0x21,
         0xEA, 0x52, 0x7A, 0xDE, 0xBA, 0x98, 0xA7, 0xAE, 0x01, 0x9A, 0xD5, 0x94, 0xF7, 0x07, 0x6F, 0x46, 0x49, 0xE8,
         0x08, 0xEB, 0xA7, 0x4A, 0x6D, 0xF0, 0xC6, 0xFD, 0x9E, 0x64, 0xFB, 0xDB, 0x47, 0x0A, 0x22, 0xB5, 0x93, 0xF3,
         0x5C, 0xA4, 0xF7, 0xEE, 0x5E, 0x8C, 0xBD, 0x15, 0x06, 0x65, 0xDA, 0xBE, 0xF5, 0xA1, 0x70, 0x18, 0x1B, 0x36,
         0x91, 0x34, 0x65, 0x7B, 0xE2, 0x20, 0xA9, 0xA2, 0x4B, 0x9E, 0xBC, 0x7F, 0xAE, 0x5C, 0xCE, 0x58, 0xC6, 0x5F,
         0x0B, 0x30, 0x73, 0xB6, 0xD4, 0x92, 0xD9, 0x85, 0xFE, 0x64, 0x53, 0xD4, 0x03, 0xAF, 0x6B, 0x5E, 0x46, 0x69,
         0x5A, 0xEC, 0x65, 0x73, 0xCD, 0x98, 0xE1, 0x11, 0x9C, 0x24, 0xE1, 0xCD, 0x81, 0x7D, 0x91, 0xF9, 0xD3, 0x86,
         0x9E, 0xA7, 0x38, 0x00, 0xA0, 0x98, 0x21, 0xD7, 0x4A, 0x1D, 0x86, 0xF4, 0x4E, 0x96, 0x34, 0x8E, 0x9F, 0x25,
         0x3D, 0x95, 0xA3, 0xD8, 0xB7, 0x7A, 0xFD, 0xB0, 0xA5, 0xC8, 0x31, 0x73, 0xA1, 0x91, 0x33, 0x34, 0x88, 0xAA,
         0xFB, 0xB8, 0x56, 0x2A, 0xE3, 0xD6, 0xA7, 0x9C, 0x98, 0x4B, 0x4B, 0xDC, 0x6D, 0xC5, 0x29, 0xA4, 0xC0, 0xC8,
         0x11, 0xCD, 0x1E, 0x19, 0x6B, 0x9E, 0x82, 0x09, 0x90, 0x48, 0xAF, 0xDE, 0x08, 0xBD, 0xBD, 0xC0, 0x5F, 0x9B,
         0xB2, 0x22, 0x1A, 0xBD, 0xD7, 0x0F, 0x9D, 0x1C, 0x34, 0xFA, 0xA2, 0xFC, 0x96, 0xE0, 0x47, 0xA7, 0x1A, 0xC1,
         0x29, 0xBA, 0x3D, 0xCC, 0x3B, 0x5F, 0x84, 0xCF, 0x24, 0x27, 0x1C, 0x39, 0x3D, 0xB3, 0x99, 0xD1, 0x06, 0x31,
         0x7B, 0x55, 0xB8, 0x1B, 0x46, 0x35, 0x9B, 0x77, 0x21, 0xEC, 0x8F, 0x31, 0x9C, 0x8F, 0xA8, 0x97, 0xE9, 0x08,
         0xC3, 0x86, 0xB4, 0xF0, 0x98, 0x2C, 0xF1, 0xEF, 0x96, 0x52, 0x3A, 0xC5, 0x9C, 0x91, 0x75, 0xD4, 0x5F, 0xC1,
         0x78, 0x79, 0x1A, 0x7A, 0xC8, 0x9A, 0x10, 0x40, 0x7A, 0xFE, 0xAF, 0x28, 0x63, 0x5E, 0xF6, 0x10, 0x42, 0xC6,
         0xC5, 0x2D, 0x79, 0x8C, 0xB2, 0x54, 0x2E, 0xFE, 0x25, 0x7F, 0x12, 0x69, 0x47, 0xD9, 0x71, 0x97, 0xF5, 0xA9,
         0xC8, 0x0C, 0x9A, 0x28, 0x1F, 0xDE, 0xFB, 0xD9, 0xDC, 0x79, 0xD8, 0x6E, 0x70, 0xFD, 0x74, 0x20, 0xD5, 0xC9,
         0x51, 0xCF, 0xC3, 0x2D, 0x57, 0xA9, 0xEE, 0x65, 0xAA, 0xC3, 0x17, 0xD6, 0x21, 0xBA, 0xFA, 0x43, 0x53, 0xD2,
         0x84, 0x86, 0x0C, 0xD5, 0x51, 0x48, 0x64, 0x40, 0x76, 0x47, 0x1D, 0xD5, 0xEF, 0x24, 0x71, 0x85, 0xFC, 0xF7,
         0xCB, 0xCA, 0xE5, 0x3E, 0x56, 0x39, 0x22, 0x50, 0x4F, 0xAF, 0x5D, 0x3A, 0xE2, 0xE4, 0x09, 0x3B, 0x68, 0xCC,
         0xFE, 0xD5, 0x0C, 0x9D, 0xBB, 0xD2, 0x16, 0xA1, 0xA5, 0x82, 0x8D, 0x7B, 0xAC, 0xEF, 0x7A, 0xE2, 0x88, 0xB6,
         0x19, 0x8F, 0x24, 0x1E, 0x53, 0x7A, 0x8C, 0xE9, 0xD3, 0x62, 0xC8, 0x9E, 0xD1, 0x35, 0x60, 0x8E, 0xC3, 0x23,
         0xAB, 0x70, 0x83, 0x38, 0xFC, 0xF4, 0x6F, 0xFF, 0x2E, 0x61, 0xDF, 0x07, 0xA8, 0xA4, 0xBE, 0xD3, 0xD7, 0x97,
         0x3E, 0x3D, 0xC9, 0xAD, 0xE6, 0xCB, 0x47, 0x09, 0x34, 0xC7, 0x1F, 0xD9, 0x7C, 0xBC, 0xE5, 0xB0, 0xE2, 0x4D,
         0x9E, 0x2A, 0xFA, 0xFF, 0x40, 0x30, 0xE5, 0xFE, 0xFF, 0x55, 0x38, 0x26, 0xC5, 0x0E, 0x16, 0xBD, 0x53, 0xC5,
         0x24, 0xA7, 0xE0, 0x5A, 0x89, 0xAF, 0x3B, 0x07, 0xA2, 0xF7, 0xCD, 0x42, 0x21, 0x8C, 0x93, 0x34, 0xCF, 0xF3,
         0x65, 0x7A, 0x33, 0x1A, 0x8B, 0xE4, 0x57, 0x54, 0x9D, 0x1E, 0xCC, 0x38, 0xBE, 0xAC, 0xF2, 0x59, 0x4E, 0x4A,
         0x05, 0x9C, 0x64, 0xDC, 0x6E, 0x87, 0x4A, 0x93, 0x94, 0xE6, 0x04, 0x64, 0xBD, 0xEC, 0xA8, 0xC5, 0xCA, 0x8A,
         0xBE, 0x34, 0xF4, 0x0B, 0xC1, 0x19, 0x39, 0x59, 0xD7, 0xBD, 0x4B, 0xB2, 0xA3, 0xE7, 0xB7, 0x0E, 0x03, 0xDC,
         0x02, 0x03, 0x87, 0x5D, 0xC1, 0xB7, 0x8F, 0xD9, 0x1D, 0xFE, 0xA1, 0xDA, 0x1D, 0x9C, 0x88, 0x60, 0x14, 0xBA,
         0xCD, 0xA7, 0x38, 0xFB, 0x36, 0x45, 0xE2, 0xAB, 0x47, 0x17, 0x41, 0x2C, 0x3E, 0x89, 0x2C, 0xA8, 0x79, 0xB2,
         0x91, 0x17, 0xEF, 0x63, 0x4D, 0xE8, 0x78, 0xB5, 0xEC, 0xFE, 0x32, 0xCA, 0xCC, 0xE3, 0xD3, 0x7F, 0xD7, 0x93,
         0x22, 0xD2, 0x6C, 0xAD, 0xD7, 0xC2, 0xF6, 0x2E, 0x33, 0x7F, 0x4A, 0x7C, 0x99, 0xE5, 0xAB, 0x50, 0x58, 0x4D,
         0x4B, 0x38, 0x00, 0x51, 0xF0, 0x9A, 0x39, 0x7E, 0xB6, 0xFE, 0x86, 0xEC, 0x61, 0x22, 0x30, 0x78, 0xE3, 0x86,
         0xE0, 0xE7, 0xA4, 0x75, 0x5E, 0xF3, 0x09, 0x5C, 0xC5, 0x5E, 0x57, 0xE0, 0x08, 0xDA, 0x3A, 0x50, 0x10, 0xE3,
         0xEF, 0x63, 0xCF, 0x14, 0x45, 0xFF, 0x94, 0xD4, 0x08, 0x0E, 0xD2, 0xA7, 0xE3, 0x49, 0x5B, 0x99, 0xFC, 0x5C,
         0x5B, 0x5F, 0x5F, 0xC1, 0x84, 0x2C, 0x4E, 0x42, 0xF5, 0x62, 0x3F, 0xEC, 0x01, 0x6C, 0x65, 0x38, 0x1A, 0xD5,
         0xD4, 0x58, 0xA4, 0xC5, 0xEA, 0xF2, 0xB8, 0xF3, 0xBE, 0xD6, 0xA2, 0x27, 0x39, 0x78, 0x7B, 0xB3, 0xF6, 0x49,
         0x28, 0x3C, 0x0C, 0xB4, 0x1A, 0x10, 0x14, 0x78, 0xD4, 0xB1, 0x71, 0x5E, 0xFF, 0x6D, 0x12, 0x88, 0xEA, 0xFE,
         0xDE, 0xF4, 0xD4, 0xD2, 0xC2, 0x51, 0x23, 0xD2, 0x33, 0x5B, 0xC4, 0x35, 0x8A, 0xA1, 0x6E, 0x39, 0x8C, 0x31,
         0x16, 0x76, 0x4A, 0x1B, 0xCB, 0x91, 0x44, 0xFD, 0x64, 0xBB, 0xA6, 0xCB, 0x0A, 0x56, 0xEB, 0x55, 0x8D, 0xEA,
         0x94, 0x91, 0xF7, 0x9D, 0xE6, 0x9F, 0x13, 0xC1, 0xB8, 0x1E, 0xBC, 0x1E, 0x8E, 0x2B, 0x3B, 0x87, 0x9A, 0xD8,
         0x32, 0xF8, 0xC4, 0xAD, 0x69, 0x08, 0xF9, 0x10, 0xD3, 0x9C, 0x89, 0x03, 0xA8, 0x96, 0xFF, 0x2D, 0x85, 0x81,
         0xF4, 0x65, 0x2F, 0x72, 0x60, 0x63, 0x54, 0x5F, 0x97, 0xB6, 0x46, 0xB5, 0xB7, 0x7C, 0xE0, 0x16, 0x61, 0x36,
         0x2F, 0x41, 0x4B, 0x81, 0xFE, 0x24, 0x7B, 0xAB, 0x09, 0x34, 0xCD, 0xA0, 0x40, 0x0E, 0xB1, 0xB1, 0x05, 0xA2,
         0x88, 0x14, 0x22, 0xE3, 0xCC, 0x45, 0x62, 0xD2, 0x34, 0xC5, 0x9F, 0xEB, 0x36, 0xF4, 0x26, 0x6C, 0xBF, 0x9E,
         0xC1, 0x72, 0xF8, 0x6A, 0x23, 0xA7, 0x87, 0xCF, 0xD2, 0x0A, 0x99, 0x37, 0xE5, 0x0D, 0x68, 0x47, 0xAC, 0x4D,
         0x52, 0x47, 0x00, 0x53, 0x15, 0x68, 0x43, 0x12, 0x9F, 0x9B, 0xF8, 0xD9, 0xF9, 0x5F, 0x78, 0xAA, 0xB5, 0x41,
         0xB9, 0x84, 0x0A, 0x3A, 0xC8, 0x2A, 0x6F, 0xFF, 0x0B, 0xF0, 0xEE, 0xB4, 0xE8, 0xE4, 0x42, 0xCB, 0xBB, 0xDB,
         0x46, 0x31, 0xD0, 0x3A, 0xAA, 0x13, 0xB9, 0x9D, 0x98, 0x85, 0xE5, 0x8B, 0x6C, 0xA3, 0xB6, 0xD1, 0x97, 0x6D,
         0x81, 0x17, 0xF0, 0x63, 0x65, 0x81, 0x40, 0x25, 0x31, 0xCC, 0xEF, 0x71, 0x56, 0x81, 0xA6, 0x81, 0xCA, 0x43,
         0xBB, 0xDC, 0x79, 0xAE, 0x61, 0x0E, 0x99, 0x14, 0xC2, 0x8F, 0xD8, 0x6E, 0x54, 0xC0, 0x71, 0xCF, 0x98, 0x53,
         0x0E, 0x0B, 0x64, 0x87, 0x6E, 0x88, 0xF8, 0xA1, 0xC0, 0xB6, 0x30, 0x58, 0x35, 0xEC, 0x60, 0x04, 0x01, 0xC8,
         0x73, 0x24, 0xB4, 0xB0, 0x46, 0xAD, 0xC7, 0x4B, 0xFB, 0xEC, 0xC3, 0x0D, 0x86, 0x44, 0x32, 0x07, 0xD1, 0xC9,
         0x94, 0xA7, 0xE4, 0x4E, 0xD4, 0xD7, 0x14, 0xFC, 0x9F, 0x7B, 0xD1, 0x3D, 0x94, 0x56, 0xC2, 0x7C, 0xED, 0x62,
         0xFD, 0x91, 0x87, 0x08, 0xA2, 0x6B, 0x44, 0xE5, 0x8C, 0x28, 0xAD, 0xF7, 0x45, 0x15, 0xD3, 0x76, 0xAA, 0x29,
         0x27, 0x0D, 0xC2, 0x72, 0x93, 0xF0, 0x5A, 0xB8, 0xBE, 0xBB, 0xD4, 0x0E, 0xA7, 0x11, 0x13, 0xFC, 0xAA, 0x47,
         0xD2, 0x4D, 0x05, 0x93, 0x1B, 0xBD, 0x3A, 0x01, 0x0E, 0x53, 0x31, 0x33, 0x3C, 0x23, 0x83, 0x8A, 0x65, 0x1E,
         0x82, 0x66, 0x3C, 0x0F, 0x65, 0x67, 0x20, 0xAE, 0x3A, 0x5A, 0xF8, 0x4D, 0x8F, 0xA9, 0x7C, 0xA0, 0x26, 0xBD,
         0x1F, 0xE3, 0x38, 0x4F, 0xBE, 0x3D, 0xCF, 0x9F, 0x50, 0x42, 0x95, 0x1C, 0x76, 0x1A, 0x9A, 0x41, 0x08, 0xD9,
         0xBA, 0xB9, 0xD6, 0xF6, 0x65, 0x41, 0x22, 0xD4, 0xDF, 0xBD, 0x4E, 0xFD, 0x58, 0xD5, 0x1B, 0x07, 0x14, 0x31,
         0xE5, 0xA8, 0xBF, 0x88, 0x22, 0xCF, 0x70, 0x5B, 0x72, 0x9D, 0xC2, 0x6D, 0x87, 0x0F, 0x6F, 0x1B, 0x6C, 0xEF,
         0xD7, 0xD1, 0xC0, 0xEE, 0x48, 0xB4, 0xA2, 0xCA, 0xC9, 0x7B, 0xB8, 0xE8, 0xA2, 0x5D, 0x39, 0x35, 0x89, 0xFA,
         0xEB, 0x78, 0xCE, 0x51, 0xB1, 0xBA, 0xE6, 0xA4)
    )

    @classmethod
    def identify(cls, file_object):
        """
        Identify an Avast Quarantined.
        :param file_object: FileObject object
        :return: Boolean value indicating if file is an Avast Quarantine File.
        """
        return file_object.data.startswith(b"-chest- ")

    def run(self):
        data = self.file_object.data[8:]
        block_size = 0x1000
        decrypted_data = bytearray()
        for i in range(0, len(data), block_size):
            decrypted_data += bytearray(d ^ k for (d, k) in zip(data[i:i + block_size], cycle(self.KEY)))
        self.dispatcher.add(FileObject(bytes(decrypted_data)))
