"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Skill = exports.SkillType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const alexa_ask_1 = require("aws-cdk-lib/alexa-ask");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Enumeration for different Alexa Skill types.
 */
var SkillType;
(function (SkillType) {
    SkillType["CUSTOM"] = "CUSTOM";
    SkillType["SMART_HOME"] = "SMART_HOME";
})(SkillType || (exports.SkillType = SkillType = {}));
/**
 * Base class for the Alexa Skill construct.
 */
class SkillBase extends constructs_1.Construct {
}
/**
 * Alexa Skill construct for managing an Alexa Skill via CloudFormation.
 */
class Skill extends SkillBase {
    /**
     * Factory method to create an Alexa Skill from its ID.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param skillId - The ID of the Alexa Skill.
     * @returns The Alexa Skill construct.
     */
    static fromSkillId(scope, id, skillId) {
        return new (class extends SkillBase {
            constructor() {
                super(...arguments);
                this.skillId = skillId;
            }
            get skillType() {
                throw new Error('Access to skillType is not supported when using fromSkillId');
            }
            get skillStage() {
                throw new Error('Access to skillStage is not supported when using fromSkillId');
            }
        })(scope, id);
    }
    /**
     * Factory method to create an Alexa Skill from its attributes.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param attributes - The attributes of the Alexa Skill.
     * @returns The Alexa Skill construct.
     */
    static fromAttributes(scope, id, attributes) {
        return new (class extends SkillBase {
            constructor() {
                super(...arguments);
                this.skillId = attributes.skillId;
                this.skillType = attributes.skillType;
                this.skillStage = attributes.skillStage;
            }
        })(scope, id);
    }
    /**
     * Creates an instance of the Alexa Skill construct.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The Alexa Skill properties.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const authPropsCount = [
            props.authenticationConfiguration,
            props.authenticationConfigurationParameter,
            props.authenticationConfigurationSecret,
        ].filter(it => !!it).length;
        if (authPropsCount !== 1) {
            throw new Error('Exactly one authentication configuration needs to be provided!');
        }
        this.packageRole = new aws_iam_1.Role(this, 'PackageRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('alexa-appkit.amazon.com'),
        });
        const authentication = this.authenticationSecret(props.authenticationConfigurationSecret) ??
            this.authenticationParameter(props.authenticationConfigurationParameter) ??
            props.authenticationConfiguration;
        this.resource = new alexa_ask_1.CfnSkill(this, 'Default', {
            skillPackage: {
                s3Bucket: props.skillPackage.asset.s3BucketName,
                s3Key: props.skillPackage.asset.s3ObjectKey,
                s3BucketRole: this.packageRole.roleArn,
                overrides: {
                    ...props.skillPackage.overrides,
                    manifest: {
                        ...props.skillPackage.overrides?.manifest,
                        publishingInformation: {
                            category: props.skillType,
                            ...props.skillPackage.overrides?.manifest?.publishingInformation,
                        },
                    },
                },
            },
            vendorId: props.vendorId,
            authenticationConfiguration: authentication,
        });
        this.resource.node.addDependency(this.packageRole);
        this.resource.node.addDependency(props.skillPackage.asset);
        props.skillPackage.asset.grantRead(this.packageRole);
        this.skillId = this.resource.ref;
        this.skillStage = props.skillStage;
        this.skillType = props.skillType;
    }
    authenticationSecret(secret) {
        if (!secret) {
            return;
        }
        return {
            clientId: secret.secretValueFromJson('clientId').unsafeUnwrap(),
            clientSecret: secret.secretValueFromJson('clientSecret').unsafeUnwrap(),
            refreshToken: secret.secretValueFromJson('refreshToken').unsafeUnwrap(),
        };
    }
    authenticationParameter(parameter) {
        if (!parameter) {
            return;
        }
        return JSON.parse(parameter.stringValue);
    }
}
exports.Skill = Skill;
_a = JSII_RTTI_SYMBOL_1;
Skill[_a] = { fqn: "cdk-skill-management.Skill", version: "1.0.26" };
//# sourceMappingURL=data:application/json;base64,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