import { Construct, IConstruct } from 'constructs';
import { CfnSkill } from 'aws-cdk-lib/alexa-ask';
import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { SkillAuthenticationProps } from './skill-authentication-props';
/**
 * Enumeration for different Alexa Skill types.
 */
export declare enum SkillType {
    CUSTOM = "CUSTOM",
    SMART_HOME = "SMART_HOME"
}
/**
 * Interface representing an Alexa Skill.
 */
export interface ISkill extends IConstruct {
    /** The unique ID of the Alexa Skill. */
    readonly skillId: string;
    /** The stage of the Alexa Skill. */
    readonly skillStage: string;
    /** The type of the Alexa Skill. */
    readonly skillType: SkillType;
}
/**
 * Base class for the Alexa Skill construct.
 */
declare abstract class SkillBase extends Construct implements ISkill {
    /**
     * The unique ID of the Alexa Skill.
     */
    abstract readonly skillId: string;
    /**
     * The stage of the Alexa Skill.
     */
    abstract readonly skillStage: string;
    /**
     * The type of the Alexa Skill.
     */
    abstract readonly skillType: SkillType;
}
/**
 * Type alias for Alexa Skill package overrides.
 */
export type SkillPackageOverrides = CfnSkill.OverridesProperty;
/**
 * Interface representing an Alexa Skill package.
 */
export interface SkillPackage {
    /** The asset associated with the Alexa Skill package. */
    readonly asset: Asset;
    /** Overrides for the Alexa Skill package. */
    readonly overrides?: SkillPackageOverrides;
}
/**
 * Properties for creating an Alexa Skill.
 */
export interface SkillProps extends SkillAuthenticationProps {
    /** The type of the Alexa Skill. */
    readonly skillType: SkillType;
    /** The stage of the Alexa Skill. */
    readonly skillStage: string;
    /** The vendor ID of the Alexa Skill. */
    readonly vendorId: string;
    /** The package information for the Alexa Skill. */
    readonly skillPackage: SkillPackage;
}
/**
 * Properties for creating an Alexa Skill Instance from Attributes.
 */
export interface SkillAttributes {
    /**
     * The unique ID of the Alexa Skill.
     */
    readonly skillId: string;
    /**
     * The stage of the Alexa Skill.
     */
    readonly skillStage: string;
    /**
     * The type of the Alexa Skill.
     */
    readonly skillType: SkillType;
}
/**
 * Alexa Skill construct for managing an Alexa Skill via CloudFormation.
 */
export declare class Skill extends SkillBase {
    /**
     * Factory method to create an Alexa Skill from its ID.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param skillId - The ID of the Alexa Skill.
     * @returns The Alexa Skill construct.
     */
    static fromSkillId(scope: Construct, id: string, skillId: string): ISkill;
    /**
     * Factory method to create an Alexa Skill from its attributes.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param attributes - The attributes of the Alexa Skill.
     * @returns The Alexa Skill construct.
     */
    static fromAttributes(scope: Construct, id: string, attributes: SkillAttributes): ISkill;
    private readonly packageRole;
    private readonly resource;
    /**
     * The unique ID of the Alexa Skill.
     */
    readonly skillId: string;
    /**
     * The stage of the Alexa Skill.
     */
    readonly skillStage: string;
    /**
     * The type of the Alexa Skill.
     */
    readonly skillType: SkillType;
    /**
     * Creates an instance of the Alexa Skill construct.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The Alexa Skill properties.
     */
    constructor(scope: Construct, id: string, props: SkillProps);
    private authenticationSecret;
    private authenticationParameter;
}
export {};
