"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SkillEndpointPermission = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const skill_1 = require("./skill");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Class for configuring and managing Skill Endpoint Permissions.
 */
class SkillEndpointPermission extends constructs_1.Construct {
    /**
     * Creates an instance of the Skill Endpoint Permission.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The Skill Endpoint properties.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.principals = {
            [skill_1.SkillType.CUSTOM]: 'alexa-appkit.amazon.com',
            [skill_1.SkillType.SMART_HOME]: 'alexa-connectedhome.amazon.com',
        };
        this.skillType = props.skillType;
        this.permission = new aws_lambda_1.CfnPermission(this, 'InitialSkillPermission', {
            functionName: props.handler.functionArn,
            principal: this.principals[props.skillType],
            action: 'lambda:InvokeFunction',
        });
        this.policy = custom_resources_1.AwsCustomResourcePolicy.fromStatements([
            new aws_iam_1.PolicyStatement({
                actions: ['lambda:RemovePermission', 'lambda:AddPermission'],
                resources: [props.handler.functionArn],
            }),
        ]);
        this.handler = props.handler;
    }
    /**
     * Configures a Skill Endpoint Permission for a specific skill.
     * This replaces the initially created permission with a new permission checking for the SkillId.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param skill - The skill to configure the permission for.
     * @returns An IDependable object representing the configured permission.
     */
    configureSkillId(scope, id, skill) {
        if (this.skillType !== skill.skillType) {
            throw new Error(`Permission was intended for skillType ${this.skillType}, but skill has ${skill.skillType}`);
        }
        const parent = new constructs_1.Construct(scope, id);
        new custom_resources_1.AwsCustomResource(parent, 'RemovePermission', {
            policy: this.policy,
            onCreate: this.removePermissionCall(parent, skill),
            onDelete: this.addPermissionCall(parent),
            installLatestAwsSdk: false,
        });
        new custom_resources_1.AwsCustomResource(parent, 'AddPermission', {
            policy: this.policy,
            onCreate: this.addPermissionCall(parent, skill),
            onDelete: this.removePermissionCall(parent, skill),
            installLatestAwsSdk: false,
        });
        parent.node.addDependency(this.permission, this.handler);
        return parent;
    }
    removePermissionCall(scope, skill) {
        return {
            service: 'Lambda',
            action: 'removePermission',
            parameters: {
                FunctionName: this.handler.functionArn,
                StatementId: this.permission.attrId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`RemovePermission-${scope.node.addr}-${skill.skillId}`),
        };
    }
    addPermissionCall(scope, skill) {
        return {
            service: 'Lambda',
            action: 'addPermission',
            parameters: {
                FunctionName: this.handler.functionArn,
                StatementId: this.permission.attrId,
                Principal: this.principals[skill?.skillType ?? this.skillType],
                Action: 'lambda:InvokeFunction',
                EventSourceToken: skill?.skillId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`AddPermission-${scope.node.addr}-${skill?.skillId ?? 'no-skill'}`),
        };
    }
}
exports.SkillEndpointPermission = SkillEndpointPermission;
_a = JSII_RTTI_SYMBOL_1;
SkillEndpointPermission[_a] = { fqn: "cdk-skill-management.SkillEndpointPermission", version: "1.0.26" };
//# sourceMappingURL=data:application/json;base64,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