# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mkdocstrings', 'mkdocstrings.handlers.crystal']

package_data = \
{'': ['*'], 'mkdocstrings': ['templates/crystal/material/*']}

install_requires = \
['Jinja2>=2.11.2,<3.0.0',
 'cached-property>=1.5.2,<2.0.0',
 'markupsafe>=1.1.1,<2.0.0',
 'mkdocs-material>=6.0.2,<7.0.0',
 'mkdocstrings>=0.14.0b1']

entry_points = \
{'markdown.extensions': ['deduplicate-toc = '
                         'mkdocstrings.handlers.crystal.deduplicate_toc:DeduplicateTocExtension']}

setup_kwargs = {
    'name': 'mkdocstrings-crystal',
    'version': '0.2.1',
    'description': 'Crystal language doc generator for mkdocstrings',
    'long_description': '# mkdocstrings-crystal\n\nCrystal language doc generator for [MkDocs][], via [mkdocstrings][].\n\n[![PyPI](https://img.shields.io/pypi/v/mkdocstrings-crystal)](https://pypi.org/project/mkdocstrings-crystal/)\n[![GitHub](https://img.shields.io/github/license/oprypin/mkdocstrings-crystal)](LICENSE.md)\n[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/oprypin/mkdocstrings-crystal/CI)](https://github.com/oprypin/mkdocstrings-crystal/actions?query=event%3Apush+branch%3Amaster)\n\n```shell\npip install mkdocstrings-crystal\n```\n\n[mkdocs]: https://www.mkdocs.org/\n[mkdocstrings]: https://pawamoy.github.io/mkdocstrings/\n[plugin]: https://www.mkdocs.org/user-guide/plugins/\n\n## Introduction\n\nCrystal has [its own easy-to-use generator of API documentation sites](https://crystal-lang.org/reference/using_the_compiler/#crystal-docs), but it\'s very rigid and doesn\'t attempt to do anything other than API documentation, so [these sites](https://crystal-lang.org/api/) end up being very isolated and prevent the author from presenting some kind of *story* about using their library.\n\nInstead, this plugin is all about that *story*. It\'s very inspiring to look at [Python\'s documentation for `subprocess`](https://docs.python.org/3/library/subprocess.html), and hard to imagine a world in which this document is just an alphabetic dump of the functions in it.\n\nSo (matching the [idea behind *mkdocstrings*](https://pawamoy.github.io/mkdocstrings/usage/) but for Crystal), this allows you to just write textual documentation in Markdown and, in the middle of it, mention any identifier of a Crystal type, method etc., and get its signature, as well as doc comment, printed out right there.\n\n## Usage\n\nWith [MkDocs][], add/merge this base config as your _mkdocs.yml_:\n\n```yaml\nsite_name: My Project\n\ntheme:\n  name: material\n\nplugins:\n  - search\n  - mkdocstrings:\n      default_handler: crystal\n      watch: [src]\n\nmarkdown_extensions:\n  - pymdownx.highlight\n  - pymdownx.magiclink\n  - pymdownx.saneheaders\n  - pymdownx.superfences\n  - deduplicate-toc\n```\n\n</details>\n\nThen, in any `docs/**/*.md` file, you can **mention a Crystal identifier alone on a line, after `:::`**:\n\n```markdown\n::: MyClass\n\n::: Other::Class#some_method\n\n::: Foo::CONSTANT\n```\n\n-- and in the output this will be replaced with generated API documentation for it, much like Crystal\'s own doc generator does.\n\nThis, of course, happens as part of a normal MkDocs build process:\n\n```shell\nmkdocs build  # generate from docs/ into site/\nmkdocs serve  # live preview\n```\n\n### Identifier linking syntax\n\nThe syntax for these "callouts" is almost exactly the same as in Crystal\'s own doc comments. As you may also know, if you **mention an identifier in backticks within a doc comment (e.g. <code>\\`SomeClass#some_method\\`</code>)**, Crystal\'s doc generator will cross-link to it. The same also works seamlessly here, and you don\'t need to change anything (other than possible edge cases).\n\nBut another powerful feature of this plugin is that you can **[cross-reference](https://pawamoy.github.io/mkdocstrings/usage/#cross-references) items like this *anywhere* on the site**, not just in doc comments. But the syntax is **`[SomeClass#some_method][]`** instead. Or `[with custom text][SomeClass#some_method]`. Note, though, that currently this cross-reference syntax is quite rigid, and you need to specify the exact absolute identifier as *mkdocstrings-crystal* determines it. For that, you can click on the item in the navigation and copy the `#Anchor` that it points to (just drop the `#` part).\n\n### Migrating from Crystal\'s own generator\n\nCrystal\'s API generator creates one HTML file per Crystal class. If you don\'t care about this whole [*story* story](#introduction) and just want to unify your docs with a main MkDocs site, you\'re welcome to re-create such a file structure. The URLs can even be backwards-compatible.\n\nSay, if you have a class `Foo::Bar`, you need to create a file _docs/Foo/Bar.md_ with the contents of just `::: Foo::Bar`. Repeat that for every class.\n\nIf you don\'t wish to write out such files manually, you can create them virtually, using [the **gen-files** plugin for MkDocs](https://github.com/oprypin/mkdocs-gen-files), with the following file *gen_doc_stubs.py*:\n\n```python\nimport mkdocs_gen_files\n\nroot = mkdocs_gen_files.config[\'plugins\'][\'mkdocstrings\'].get_handler(\'crystal\').collector.root\n\nfor typ in root.walk_types():\n    filename = \'/\'.join(typ.abs_id.split(\'::\')) + \'.md\'\n    with mkdocs_gen_files.open(filename, \'w\') as f:\n        f.write(f\'::: {typ.abs_id}\\n\\n\')\n```\n\nYou would also need this addition to _mkdocs.yml_:\n\n```yaml\nuse_directory_urls: false\n\nplugins:\n  - gen-files:\n      scripts:\n        - gen_doc_stubs.py\n```\n\nAlso check out [a more complex example](https://github.com/oprypin/athena-website/blob/mkdocstrings/gen_doc_stubs.py) and [an actual website using this](https://oprypin.github.io/athena-website/Validator/Constraint/).\n\n## Usage details\n\nWe have been talking about seamlessly inserting Crystal documentation, but where is it coming from? The implementation actually still uses `crystal doc` generator but through [its JSON output](https://github.com/crystal-lang/crystal/pull/4746). So the requirement is the same: the source code that the doc comments and signatures are coming from is assumed to be somewhere under the _src/_ directory. If that isn\'t appropriate, you can select the wanted entry files by passing them to `crystal doc`, as part of [`crystal_docs_flags`](#crystal_docs_flags) ([example](https://github.com/oprypin/athena-website/blob/c06906d5933421120c76e15fd6f529eeb5c48221/mkdocs.yml#L33)).\n\n### Options\n\n(See also: [mkdocstrings global options](https://pawamoy.github.io/mkdocstrings/usage/#global-options))\n\n#### `crystal_docs_flags:`\n\nA list of command line arguments to pass to `crystal doc`. Mainly used to choose the source directories.\n\n*The above options have been global-only, while the ones below can also apply per-identifier.*\n\n#### `selection:`\n\n* `nested_types:` (`true` / **`false`**)\n\n  Set to `true` to also recursively render all `Foo::Sub::Types` whenever rendering a given class `Foo`.\n\n* `file_filters:` [list of strings]\n\n  If a particular module spans over several files, you might want to choose to render only the sub-items (see `nested_types`) that came from a particular file. These patterns ared regular expressions (not anchored) applied to the file path. Negating the patterns is done by starting it with `!` (which is then excluded from the following regex).\n\n#### `rendering:`\n\n* `show_source_links:` (**`true`** / `false`)\n\n  Set to `false` to skip adding `[View source]` links after every method etc.\n\n* `heading_level:`: (`1`/**`2`**/`3`/`4`/`5`/`6`)\n\n  Each inserted identifier gets an HTML heading. The default heading is `<h2>`, and sub-headings in it are shifted accordingly (so if you write headings in doc comments, you\'re welcome to start with `#` `<h1>`). You can change this heading level, either the default one or per-identifier.\n\n### Example of a global config\n\n```yaml\nplugins:\n  - mkdocstrings:\n      default_handler: crystal\n      watch: [src]\n      handlers:\n        crystal:\n          crystal_docs_flags:\n            - src/bar.cr\n            - lib/foo/src/foo.cr\n          rendering:\n            show_source_links: false\n```\n\n### Example of a per-identifier config\n\n```markdown\n::: SomeModule\n    selection:\n      nested_types: true\n      file_filters:\n        - \'src/foo/[^/]+\\.cr$\'\n    rendering:\n      heading_level: 3\n```\n\n### Extras\n\n#### "deduplicate-toc" extension\n\nFor most [usages it is recommended](#usage) to enable the "deduplicate-toc" Markdown extension, which comes bundled with mkdocstrings-crystal. It de-duplicates consecutive items that have the same title in the table of contents. This is useful because Crystal can have multiple overloads of a method but in the ToC only their names are shown.\n',
    'author': 'Oleh Prypin',
    'author_email': 'oleh@pryp.in',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/oprypin/mkdocstrings-crystal',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
