"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RDSDatabaseRunningScheduleStack = exports.RDSDatabaseRunningScheduleStackResourceNamingType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_resource_naming_1 = require("@gammarers/aws-resource-naming");
Object.defineProperty(exports, "RDSDatabaseRunningScheduleStackResourceNamingType", { enumerable: true, get: function () { return aws_resource_naming_1.ResourceNamingType; } });
const aws_sns_slack_message_lambda_subscription_1 = require("@gammarers/aws-sns-slack-message-lambda-subscription");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const scheduler = require("aws-cdk-lib/aws-scheduler");
const sns = require("aws-cdk-lib/aws-sns");
const running_control_state_machine_1 = require("./resources/running-control-state-machine");
class RDSDatabaseRunningScheduleStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        // 👇 Create random 8 length string
        const random = aws_resource_naming_1.ResourceNaming.createRandomString(`${cdk.Names.uniqueId(scope)}-${cdk.Names.uniqueId(this)}`);
        // 👇 Definition auto naming
        const autoNaming = {
            notificationTopicName: `rds-db-running-schedule-notification-${random}-topic`,
            notificationTopicDisplayName: 'RDS DB Running Schedule Notification Topic',
            stateMachineName: `rds-db-running-schedule-${random}-state-machine`,
            stateMachineRoleName: `rds-db-running-schedule-${random}-state-machine-role`,
            schedulerRoleName: `rds-db-running-scheduler-${random}-exec-role`,
            startScheduleName: `rds-database-running-stop-${random}-schedule`,
            stopScheduleName: `rds-database-running-start-${random}-schedule`,
        };
        // 👇　final naming
        const names = aws_resource_naming_1.ResourceNaming.naming(autoNaming, props.resourceNamingOption);
        // 👇 SNS Topic for notifications
        const topic = new sns.Topic(this, 'NotificationTopic', {
            topicName: names.notificationTopicName,
            displayName: names.notificationTopicDisplayName,
        });
        // 👇 Subscribe an email endpoint to the topic
        const emails = props.notifications?.emails ?? [];
        for (const [index, value] of emails.entries()) {
            new sns.Subscription(this, `SubscriptionEmail${index.toString().padStart(3, '0')}`, {
                topic,
                protocol: sns.SubscriptionProtocol.EMAIL,
                endpoint: value,
            });
        }
        // 👇 Subscription slack webhook
        if (props.notifications?.slack) {
            new aws_sns_slack_message_lambda_subscription_1.SNSSlackMessageLambdaSubscription(this, 'SNSSlackMessageLambdaSubscription', {
                topic,
                slackWebhookSecretName: props.notifications.slack.webhookSecretName,
            });
        }
        // 👇 StepFunctions State Machine
        const stateMachine = new running_control_state_machine_1.RunningControlStateMachine(this, 'StateMachine', {
            stateMachineName: names.stateMachineName,
            notificationTopic: topic,
            timeout: (() => {
                if (props.timeoutOption?.stateMachineTimeout) {
                    return props.timeoutOption?.stateMachineTimeout;
                }
                return aws_cdk_lib_1.Duration.hours(1);
            })(),
        });
        // 👇 rename role name & description.
        if (names.stateMachineRoleName) {
            const role = stateMachine.node.findChild('Role');
            const cfnRole = role.node.defaultChild;
            cfnRole.addPropertyOverride('RoleName', names.stateMachineRoleName);
            cfnRole.addPropertyOverride('Description', 'RDS DB Running machine role.');
            const policy = role.node.findChild('DefaultPolicy');
            const cfnPolicy = policy.node.defaultChild;
            cfnPolicy.addPropertyOverride('PolicyName', `rds-db-running-schedule-${random}-state-machine-policy`);
        }
        // 👇 StateMachine Exec Role of Scheduler
        const schedulerExecRole = new iam.Role(this, 'SchedulerExecutionRole', {
            roleName: names.schedulerRoleName,
            description: 'RDS DB Running scheduler role',
            assumedBy: new iam.ServicePrincipal('scheduler.amazonaws.com'),
            inlinePolicies: {
                'state-machine-exec-policy': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                'states:StartExecution',
                            ],
                            resources: [
                                stateMachine.stateMachineArn,
                            ],
                        }),
                    ],
                }),
            },
        });
        // 👇 Schedule state
        const sheduleState = (() => {
            if (props.enableScheduling === undefined || props.enableScheduling) {
                return 'ENABLED';
            }
            else {
                return 'DISABLED';
            }
        })();
        // 👇 Stop Schedule expression
        const stopScheduleExpression = (() => {
            // default: weekday 21:10
            const minute = props.stopSchedule?.minute ?? '10';
            const hour = props.stopSchedule?.hour ?? '21';
            const week = props.stopSchedule?.week ?? 'MON-FRI';
            return `cron(${minute} ${hour} ? * ${week} *)`;
        })();
        // 👇 Start Schedule expression
        const startScheduleExpression = (() => {
            // default: weekday 07:50
            const minute = props.startSchedule?.minute ?? '50';
            const hour = props.startSchedule?.hour ?? '7';
            const week = props.startSchedule?.week ?? 'MON-FRI';
            return `cron(${minute} ${hour} ? * ${week} *)`;
        })();
        // 👇 Stop DB schedule
        new DBRuningSchedule(this, 'StopDatabaseRunningSchedule', {
            name: names.startScheduleName,
            description: 'stop database(instance/cluster) running stop schedule.',
            sheduleState,
            timezone: props.stopSchedule?.timezone ?? 'UTC',
            expression: stopScheduleExpression,
            target: {
                stateMachineArn: stateMachine.stateMachineArn,
                roleArn: schedulerExecRole.roleArn,
                resourceTag: {
                    key: props.targetResource.tagKey,
                    values: props.targetResource.tagValues,
                },
                input: { mode: 'Stop' },
            },
        });
        // 👇 Start DB schedule
        new DBRuningSchedule(this, 'StartDatabaseRunningSchedule', {
            name: names.stopScheduleName,
            description: 'start db instance schedule.',
            sheduleState,
            timezone: props.startSchedule?.timezone ?? 'UTC',
            expression: startScheduleExpression,
            target: {
                stateMachineArn: stateMachine.stateMachineArn,
                roleArn: schedulerExecRole.roleArn,
                resourceTag: {
                    key: props.targetResource.tagKey,
                    values: props.targetResource.tagValues,
                },
                input: { mode: 'Start' },
            },
        });
    }
}
exports.RDSDatabaseRunningScheduleStack = RDSDatabaseRunningScheduleStack;
_a = JSII_RTTI_SYMBOL_1;
RDSDatabaseRunningScheduleStack[_a] = { fqn: "@gammarers/aws-rds-database-running-schedule-stack.RDSDatabaseRunningScheduleStack", version: "2.6.0" };
class DBRuningSchedule extends scheduler.CfnSchedule {
    constructor(scope, id, props) {
        super(scope, id, {
            name: props.name,
            description: props.description,
            state: props.sheduleState,
            //groupName: scheduleGroup.name, // default
            flexibleTimeWindow: {
                mode: 'OFF',
            },
            scheduleExpressionTimezone: props.timezone,
            scheduleExpression: props.expression,
            target: {
                arn: props.target.stateMachineArn,
                roleArn: props.target.roleArn,
                input: JSON.stringify({
                    Params: {
                        TagKey: props.target.resourceTag.key,
                        TagValues: props.target.resourceTag.values,
                        Mode: props.target.input.mode,
                    },
                }),
                retryPolicy: {
                    maximumEventAgeInSeconds: 60,
                    maximumRetryAttempts: 0,
                },
            },
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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