#!/usr/bin/env python3
"""
Comprehensive test suite for reconcile package.
"""
import os
import sys
import tempfile
import shutil
import subprocess
from pathlib import Path
from unittest.mock import patch, MagicMock
import pytest
from git import Repo

# Add src directory to Python path for testing
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'src'))
import reconcile


class TestConflictResolution:
    """Unit tests for conflict resolution with mocked OpenAI API."""

    @patch('reconcile._get_openai_client')
    def test_resolve_conflict_section_strips_markers(self, mock_client_getter):
        """Test that resolve_conflict_section correctly strips conflict markers and returns AI response."""
        # Setup mock response
        mock_response = MagicMock()
        mock_response.choices[0].message.content = "def merged_function():\n    return 'resolved'"
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        # Test conflict section with markers
        conflict_section = """<<<<<<< HEAD
def function_a():
    return 'version a'
=======
def function_b():
    return 'version b'
>>>>>>> feature-branch"""
        
        result = reconcile.resolve_conflict_section(conflict_section, model="gpt-4")
        
        # Verify OpenAI was called with correct parameters
        mock_client.chat.completions.create.assert_called_once()
        call_args = mock_client.chat.completions.create.call_args
        
        assert call_args[1]['model'] == 'gpt-4'
        assert call_args[1]['temperature'] == 0
        assert len(call_args[1]['messages']) == 2
        assert 'system' in call_args[1]['messages'][0]['role']
        assert 'user' in call_args[1]['messages'][1]['role']
        assert conflict_section in call_args[1]['messages'][1]['content']
        
        # Verify result doesn't contain conflict markers
        assert '<<<<<<< HEAD' not in result
        assert '=======' not in result
        assert '>>>>>>> ' not in result
        assert result == "def merged_function():\n    return 'resolved'"

    @patch('reconcile._get_openai_client')
    def test_resolve_conflict_section_different_models(self, mock_client_getter):
        """Test that different models can be used."""
        mock_response = MagicMock()
        mock_response.choices[0].message.content = "resolved content"
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        conflict = "<<<<<<< HEAD\nold\n=======\nnew\n>>>>>>> branch"
        
        # Test with gpt-3.5-turbo
        reconcile.resolve_conflict_section(conflict, model="gpt-3.5-turbo")
        assert mock_client.chat.completions.create.call_args[1]['model'] == 'gpt-3.5-turbo'

    @patch('reconcile._get_openai_client')
    def test_resolve_conflict_sections_batch_single(self, mock_client_getter):
        """Test that batch resolution works correctly with a single conflict."""
        # Setup mock response with proper batch format
        mock_response = MagicMock()
        mock_response.choices[0].message.content = """RESOLUTION 1:
def merged_function():
    return 'resolved'"""
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        conflicts = ["<<<<<<< HEAD\ndef func_a():\n    pass\n=======\ndef func_b():\n    pass\n>>>>>>> branch"]
        
        results = reconcile.resolve_conflict_sections_batch(conflicts, model="gpt-4")
        
        assert len(results) == 1
        assert results[0] == "def merged_function():\n    return 'resolved'"
        # Should be called once for the batch (not falling back to individual)
        assert mock_client.chat.completions.create.call_count >= 1

    @patch('reconcile._get_openai_client')
    def test_resolve_conflict_sections_batch_multiple(self, mock_client_getter):
        """Test that batch resolution works correctly with multiple conflicts."""
        # Setup mock response with numbered format
        mock_response = MagicMock()
        mock_response.choices[0].message.content = """RESOLUTION 1:
def merged_function_1():
    return 'resolved 1'

RESOLUTION 2:
def merged_function_2():
    return 'resolved 2'

RESOLUTION 3:
def merged_function_3():
    return 'resolved 3'"""
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        conflicts = [
            "<<<<<<< HEAD\ndef func_a1():\n    pass\n=======\ndef func_b1():\n    pass\n>>>>>>> branch",
            "<<<<<<< HEAD\ndef func_a2():\n    pass\n=======\ndef func_b2():\n    pass\n>>>>>>> branch",
            "<<<<<<< HEAD\ndef func_a3():\n    pass\n=======\ndef func_b3():\n    pass\n>>>>>>> branch"
        ]
        
        results = reconcile.resolve_conflict_sections_batch(conflicts, model="gpt-4")
        
        assert len(results) == 3
        assert results[0] == "def merged_function_1():\n    return 'resolved 1'"
        assert results[1] == "def merged_function_2():\n    return 'resolved 2'"
        assert results[2] == "def merged_function_3():\n    return 'resolved 3'"
        
        # Verify the prompt contains batch format
        call_args = mock_client.chat.completions.create.call_args
        prompt = call_args[1]['messages'][1]['content']
        assert '=== CONFLICT 1 ===' in prompt
        assert '=== CONFLICT 2 ===' in prompt
        assert '=== CONFLICT 3 ===' in prompt
        assert 'RESOLUTION 1:' in prompt
        assert 'RESOLUTION 2:' in prompt

    @patch('reconcile._get_openai_client')
    def test_batch_response_parsing_with_code_blocks(self, mock_client_getter):
        """Test parsing LLM responses that include markdown code blocks."""
        mock_response = MagicMock()
        mock_response.choices[0].message.content = """RESOLUTION 1:
```python
def merged_function_1():
    return 'resolved 1'
```

RESOLUTION 2:
```python
def merged_function_2():
    return 'resolved 2'
```"""
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        conflicts = ["conflict1", "conflict2"]
        results = reconcile.resolve_conflict_sections_batch(conflicts, model="gpt-4")
        
        assert len(results) == 2
        assert results[0] == "def merged_function_1():\n    return 'resolved 1'"
        assert results[1] == "def merged_function_2():\n    return 'resolved 2'"

    def test_parse_batch_response_success(self):
        """Test successful parsing of batch response."""
        response = """RESOLUTION 1:
function merged1() {
    return 'result1';
}

RESOLUTION 2:
function merged2() {
    return 'result2';
}"""
        
        results = reconcile._parse_batch_response(response, 2)
        assert len(results) == 2
        assert "function merged1()" in results[0]
        assert "function merged2()" in results[1]

    def test_parse_batch_response_wrong_count(self):
        """Test error handling when response count doesn't match expected."""
        response = """RESOLUTION 1:
function merged1() {
    return 'result1';
}"""
        
        with pytest.raises(ValueError, match="Missing resolutions for numbers"):
            reconcile._parse_batch_response(response, 2)

    def test_parse_batch_response_out_of_range(self):
        """Test error handling when resolution numbers are out of range."""
        response = """RESOLUTION 5:
function merged5() {
    return 'result5';
}"""
        
        with pytest.raises(ValueError, match="Resolution number 5 is out of range"):
            reconcile._parse_batch_response(response, 2)

    def test_parse_batch_response_no_matches(self):
        """Test error handling when response format is unrecognizable."""
        response = "Just some random text without proper format"
        
        with pytest.raises(ValueError, match="Could not parse LLM response"):
            reconcile._parse_batch_response(response, 1)

    @patch('reconcile._get_openai_client')
    def test_batch_fallback_on_failure(self, mock_client_getter):
        """Test that batch resolution falls back to individual on parsing failure."""
        # First call fails parsing, subsequent calls succeed
        mock_responses = [
            MagicMock(),  # Batch call - malformed response
            MagicMock(),  # Individual call 1
            MagicMock(),  # Individual call 2
        ]
        
        mock_responses[0].choices[0].message.content = "Malformed response"
        mock_responses[1].choices[0].message.content = "resolved 1"
        mock_responses[2].choices[0].message.content = "resolved 2"
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.side_effect = mock_responses
        mock_client_getter.return_value = mock_client
        
        conflicts = ["conflict1", "conflict2"]
        results = reconcile.resolve_conflict_sections_batch(conflicts, model="gpt-4")
        
        assert len(results) == 2
        assert results[0] == "resolved 1"
        assert results[1] == "resolved 2"
        
        # Should have made 3 calls total (1 batch + 2 individual)
        assert mock_client.chat.completions.create.call_count == 3

    @patch('reconcile._get_openai_client')
    def test_batch_max_batch_size(self, mock_client_getter):
        """Test that large batches are split according to max_batch_size."""
        mock_response1 = MagicMock()
        mock_response1.choices[0].message.content = """RESOLUTION 1:
resolved 1

RESOLUTION 2:
resolved 2"""
        
        mock_response2 = MagicMock()
        mock_response2.choices[0].message.content = """RESOLUTION 1:
resolved 3

RESOLUTION 2:
resolved 4"""
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.side_effect = [mock_response1, mock_response2]
        mock_client_getter.return_value = mock_client
        
        # 4 conflicts with batch_size=2 should make 2 calls
        conflicts = ["conflict1", "conflict2", "conflict3", "conflict4"]
        results = reconcile.resolve_conflict_sections_batch(conflicts, model="gpt-4", max_batch_size=2)
        
        assert len(results) == 4
        assert results[0] == "resolved 1"
        assert results[1] == "resolved 2"
        assert results[2] == "resolved 3"
        assert results[3] == "resolved 4"
        
        # Should have made 2 batch calls
        assert mock_client.chat.completions.create.call_count == 2


class TestGitIntegration:
    """Integration tests using actual Git repositories."""

    def setup_method(self):
        """Create a temporary directory for test repos."""
        self.test_dir = tempfile.mkdtemp()

    def teardown_method(self):
        """Clean up temporary test directory."""
        shutil.rmtree(self.test_dir, ignore_errors=True)

    def create_conflicted_repo(self):
        """Helper to create a Git repo with merge conflicts using proper Git merge."""
        repo_path = os.path.join(self.test_dir, 'test_repo')
        os.makedirs(repo_path)
        
        # Initialize repo and create initial commit
        repo = Repo.init(repo_path)
        file_path = os.path.join(repo_path, 'test.py')
        
        # Create a base file that will cause conflicts
        with open(file_path, 'w') as f:
            f.write("""def shared_function():
    return 'original version'

def helper_function():
    return 'helper'
""")
        
        repo.index.add(['test.py'])
        repo.index.commit('Initial commit')
        
        # Create and switch to feature branch
        feature_branch = repo.create_head('feature-branch')
        feature_branch.checkout()
        
        # Make changes on feature branch
        with open(file_path, 'w') as f:
            f.write("""def shared_function():
    return 'feature version with new functionality'

def helper_function():
    return 'helper'

def new_feature_function():
    return 'new feature'
""")
        
        repo.index.add(['test.py'])
        repo.index.commit('Add feature functionality')
        
        # Switch back to master and make conflicting changes
        repo.heads.master.checkout()
        
        with open(file_path, 'w') as f:
            f.write("""def shared_function():
    return 'master version with important fixes'

def helper_function():
    return 'helper'

def bug_fix_function():
    return 'bug fix'
""")
        
        repo.index.add(['test.py'])
        repo.index.commit('Add bug fixes')
        
        # Attempt merge to create real conflict
        try:
            repo.git.merge('feature-branch', '--no-commit')
        except Exception as e:
            # Expected merge conflict
            pass
        
        # Check if we have actual unmerged blobs
        unmerged = repo.index.unmerged_blobs()
        if not unmerged:
            # If merge didn't create unmerged blobs, simulate the conflict manually
            # This can happen with different Git versions or configurations
            conflict_content = """def shared_function():
<<<<<<< HEAD
    return 'master version with important fixes'
=======
    return 'feature version with new functionality'
>>>>>>> feature-branch

def helper_function():
    return 'helper'

<<<<<<< HEAD
def bug_fix_function():
    return 'bug fix'
=======
def new_feature_function():
    return 'new feature'
>>>>>>> feature-branch
"""
            
            with open(file_path, 'w') as f:
                f.write(conflict_content)
            
            # Force Git to recognize this as a merge conflict
            # by manipulating the index directly
            import subprocess
            try:
                # Use git add with --update to add the conflicted file
                subprocess.run(['git', 'add', '-u'], cwd=repo_path, check=False)
                # Then put it back into conflicted state
                subprocess.run(['git', 'reset', 'HEAD', 'test.py'], cwd=repo_path, check=False)
            except:
                pass
        
        return repo_path

    @patch('reconcile._get_openai_client')
    def test_full_reconcile_workflow_dry_run(self, mock_client_getter):
        """Test complete workflow with temporary Git repo in dry-run mode."""
        # Create repo with conflicts
        repo_path = self.create_conflicted_repo()
        
        # Test conflict detection - if no unmerged blobs, test parse_conflicts directly
        conflicts = reconcile.detect_conflicts(repo_path)
        
        if conflicts is None:
            # If detect_conflicts doesn't find unmerged blobs, test parse_conflicts directly
            # This simulates what would happen after a manual git merge conflict
            conflict_files = {'test.py': 'mock_blob_data'}
            parsed_conflicts = reconcile.parse_conflicts(conflict_files, repo_path)
        else:
            parsed_conflicts = reconcile.parse_conflicts(conflicts, repo_path)
        
        assert len(parsed_conflicts) > 0, "Should parse conflict sections"
        
        # Verify conflict content
        for path, sections in parsed_conflicts.items():
            assert len(sections) > 0, f"Should find conflict sections in {path}"
            for section in sections:
                assert '<<<<<<< HEAD' in section
                assert '=======' in section
                assert '>>>>>>> ' in section
        
        # Mock should not be called in dry-run mode
        mock_client_getter.assert_not_called()

    @patch('reconcile._get_openai_client')
    def test_reconcile_without_dry_run(self, mock_client_getter):
        """Test complete workflow with temporary Git repo (not dry-run)."""
        # Setup mock
        mock_response = MagicMock()
        mock_response.choices[0].message.content = """RESOLUTION 1:
def reconciled_function():
    return 'reconciled'
"""
        
        mock_client = MagicMock()
        mock_client.chat.completions.create.return_value = mock_response
        mock_client_getter.return_value = mock_client
        
        # Create repo with conflicts
        repo_path = self.create_conflicted_repo()
        
        # Detect and resolve conflicts
        conflicts = reconcile.detect_conflicts(repo_path)
        
        if conflicts is None:
            # If detect_conflicts doesn't find unmerged blobs, test parse_conflicts directly
            conflict_files = {'test.py': 'mock_blob_data'}
            parsed_conflicts = reconcile.parse_conflicts(conflict_files, repo_path)
        else:
            parsed_conflicts = reconcile.parse_conflicts(conflicts, repo_path)
        
        # Test resolution
        for path, sections in parsed_conflicts.items():
            resolved = reconcile.resolve_conflict_sections_batch(sections)
            assert len(resolved) == len(sections)
            for resolution in resolved:
                assert '<<<<<<< HEAD' not in resolution
                assert '=======' not in resolution
                assert '>>>>>>> ' not in resolution

    def test_detect_conflicts_with_no_conflicts(self):
        """Test conflict detection when there are no conflicts."""
        repo_path = os.path.join(self.test_dir, 'clean_repo')
        os.makedirs(repo_path)
        
        # Create a clean repo
        repo = Repo.init(repo_path)
        file_path = os.path.join(repo_path, 'clean.py')
        
        with open(file_path, 'w') as f:
            f.write("print('hello world')")
        
        repo.index.add(['clean.py'])
        repo.index.commit('Clean commit')
        
        # Should detect no conflicts
        conflicts = reconcile.detect_conflicts(repo_path)
        assert conflicts is None


class TestGitHookInstallation:
    """Tests for Git hook installation functionality."""

    def setup_method(self):
        """Create a temporary Git repository for hook testing."""
        # Save the current working directory
        self.original_cwd = os.getcwd()
        
        self.test_dir = tempfile.mkdtemp()
        self.repo_path = os.path.join(self.test_dir, 'test_repo')
        os.makedirs(self.repo_path)
        
        # Change to the repo directory before creating Git repo
        os.chdir(self.repo_path)
        
        # Initialize a Git repo
        repo = Repo.init(self.repo_path)
        file_path = os.path.join(self.repo_path, 'dummy.txt')
        with open(file_path, 'w') as f:
            f.write("dummy content")
        repo.index.add(['dummy.txt'])
        repo.index.commit('Initial commit')

    def teardown_method(self):
        """Clean up temporary test directory."""
        # Restore original working directory
        os.chdir(self.original_cwd)
        shutil.rmtree(self.test_dir, ignore_errors=True)

    def test_install_hook_creates_symlink(self):
        """Test that install_hook creates a proper symlink."""
        os.chdir(self.repo_path)
        
        # Create a dummy script path
        script_path = os.path.join(self.test_dir, 'dummy_script.py')
        with open(script_path, 'w') as f:
            f.write('#!/usr/bin/env python3\nprint("hook executed")')
        
        # Test hook installation
        reconcile.install_hook('post-merge', script_path)
        
        # Verify symlink was created
        hook_path = os.path.join(self.repo_path, '.git', 'hooks', 'post-merge')
        assert os.path.islink(hook_path), "Hook should be a symlink"
        assert os.readlink(hook_path) == script_path, "Symlink should point to script"
        
        # Verify permissions
        stat_info = os.stat(hook_path)
        assert stat_info.st_mode & 0o755, "Hook should be executable"

    def test_install_different_hook_types(self):
        """Test installing different types of Git hooks."""
        os.chdir(self.repo_path)
        
        script_path = os.path.join(self.test_dir, 'hook_script.py')
        with open(script_path, 'w') as f:
            f.write('#!/usr/bin/env python3\nprint("test hook")')
        
        hook_types = ['pre-merge', 'post-merge', 'pre-push']
        
        for hook_type in hook_types:
            reconcile.install_hook(hook_type, script_path)
            hook_path = os.path.join(self.repo_path, '.git', 'hooks', hook_type)
            assert os.path.islink(hook_path), f"{hook_type} hook should be installed"

    def test_install_subcommand_integration(self):
        """Test that the install subcommand works correctly."""
        os.chdir(self.repo_path)
        
        # Test using subprocess to call the install command with proper environment
        env = os.environ.copy()
        env['OPENAI_API_KEY'] = 'test-api-key'  # Set dummy key for subprocess
        
        # Run the install command
        try:
            result = subprocess.run([
                'python', '-m', 'reconcile', 'install', '--hook', 'post-merge'
            ], capture_output=True, text=True, timeout=10, env=env)
            
            # Check if hook was installed
            hook_path = os.path.join(self.repo_path, '.git', 'hooks', 'post-merge')
            
            # The command should succeed since install doesn't require OpenAI
            if result.returncode == 0:
                assert os.path.exists(hook_path), "Hook should be installed"
            else:
                # If it fails, at least check it's not a syntax error
                assert 'SyntaxError' not in result.stderr, f"Should not have syntax errors: {result.stderr}"
            
        except subprocess.TimeoutExpired:
            pytest.skip("Install command test skipped due to timeout")
        except Exception as e:
            pytest.skip(f"Install command test skipped due to: {e}")


class TestConflictParsing:
    """Tests for conflict parsing functionality."""

    def test_parse_conflicts_extracts_sections(self):
        """Test that parse_conflicts correctly extracts conflict sections."""
        # Create a temporary file with conflicts
        with tempfile.NamedTemporaryFile(mode='w', suffix='.py', delete=False) as f:
            f.write("""import os

def function_1():
<<<<<<< HEAD
    return 'version_a'
=======
    return 'version_b'
>>>>>>> feature-branch

def normal_function():
    return 'no conflict'

def function_2():
<<<<<<< HEAD
    print('main implementation')
    return 42
=======
    print('feature implementation')
    return 'feature'
>>>>>>> feature-branch
""")
            temp_file = f.name
        
        try:
            # Mock conflict blobs
            conflict_blobs = {os.path.basename(temp_file): 'mock_blob_data'}
            
            # Parse conflicts
            parsed = reconcile.parse_conflicts(conflict_blobs, os.path.dirname(temp_file))
            
            # Verify results
            assert len(parsed) == 1
            file_conflicts = list(parsed.values())[0]
            assert len(file_conflicts) == 2, f"Should find 2 conflict sections, found {len(file_conflicts)}: {file_conflicts}"
            
            # Check first conflict
            assert '<<<<<<< HEAD' in file_conflicts[0]
            assert "return 'version_a'" in file_conflicts[0]
            assert "return 'version_b'" in file_conflicts[0]
            assert '>>>>>>> feature-branch' in file_conflicts[0]
            
            # Check second conflict
            assert '<<<<<<< HEAD' in file_conflicts[1]
            assert 'main implementation' in file_conflicts[1]
            assert 'feature implementation' in file_conflicts[1]
            assert '>>>>>>> feature-branch' in file_conflicts[1]
            
        finally:
            os.unlink(temp_file)

    def test_parse_conflicts_no_conflicts(self):
        """Test parse_conflicts when file has no conflict markers."""
        # Create a clean temporary file
        with tempfile.NamedTemporaryFile(mode='w', suffix='.py', delete=False) as f:
            f.write("""def clean_function():
    return 'no conflicts here'
""")
            temp_file = f.name
        
        try:
            # Mock conflict blobs
            conflict_blobs = {os.path.basename(temp_file): 'mock_blob_data'}
            
            # Parse conflicts
            parsed = reconcile.parse_conflicts(conflict_blobs, os.path.dirname(temp_file))
            
            # Should find no conflicts
            assert len(parsed) == 0, "Should find no conflict sections in clean file"
            
        finally:
            os.unlink(temp_file)


if __name__ == "__main__":
    pytest.main([__file__]) 