# import packages
import gqcms
import seaborn as sns
import pandas as pd
import numpy as np
import itertools


def getEDF(molecule: gqcms.Hubbard, wavefunction: pd.DataFrame, electron_parition: list, domain_parition: list) -> float:
    """
    Calculates the EDF of a given partition

    :param molecule: The Hubbard object of interest
    :param wavefunction: the wave function associated with the Hubbard object as generated by the gqcms.FCI method
    :param partition: the partition as a list of lists where each list is a domain
    """

    assert len(electron_parition) == len(domain_parition), "electron parition and domain partition do not have the same length"
    assert sum(electron_parition) == molecule.nalpha + molecule.nbeta, "electron count in parition and molecule do not match"

    # get bitstring representation of all the domains
    dom_bit_list = []
    for domain in domain_parition:
        domain_bits = gqcms.Determinant.orbitals_to_onv(domain)
        dom_bit_list.append(domain_bits)

    probability = 0
    for index, ONV in enumerate(molecule.basis):
        # check for all domains
        viable = True
        for number, domain in enumerate(dom_bit_list):
            # count the overlap of alpha and domain and beta and domain
            alpha_count = len(
                gqcms.Determinant.onv_to_orbitals(ONV._alpha_onv & domain)
            )
            beta_count = len(
                gqcms.Determinant.onv_to_orbitals(ONV._beta_onv & domain)
            )

            # amount of electrons has to be equal to nu in all domains
            if alpha_count + beta_count != electron_parition[number]:
                viable = False
        
        if viable:
            probability += wavefunction["C"][0][index]**2
    
    return probability


def FullEDF(molecule: gqcms.Hubbard, wf: pd.DataFrame, domain_partition: list):
    """
    Will generate the full EDF for a partition over a molecule

    :param molecule: the Hubbard object of interest
    :param wf: the wave function associated with the molecule as generated by the gqcms.FCI method
    :param partition: the domain partition over the molecule
    """
    # get the maximum of electrons that fit in each domain
    max_list = []
    # the total amount of domains
    amount = len(domain_partition)
    
    # get all possible partitions
    pos_per_site_list = []
    for domain in domain_partition:
        max_el = len(domain)*2
        pos_in_dom = list(range(max_el + 1))
        pos_per_site_list.append(pos_in_dom)
    
    electron_partitions = itertools.product(*pos_per_site_list)
    
    # iterate all the electron partitions
    partition_dict = {}
    for partition in electron_partitions:
        if sum(partition) == molecule.nalpha + molecule.nbeta:
            probability  = getEDF(molecule=molecule, wavefunction=wf, domain_parition=domain_partition, electron_parition=partition)
            partition_dict[partition] = probability
    
    return partition_dict