# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nautobot_plugin_nornir',
 'nautobot_plugin_nornir.plugins.credentials',
 'nautobot_plugin_nornir.plugins.inventory',
 'nautobot_plugin_nornir.tests',
 'nautobot_plugin_nornir.tests.inventory']

package_data = \
{'': ['*']}

install_requires = \
['nautobot>=1.0.0,<2.0.0', 'nornir-nautobot>=2.0.0,<3.0.0']

setup_kwargs = {
    'name': 'nautobot-plugin-nornir',
    'version': '0.9.2',
    'description': 'Nautobot Nornir plugin.',
    'long_description': '# ntc-nautobot-plugin-nornir\n\nA plugin for [Nautobot](https://github.com/nautobot/nautobot), that intends to be a small shim layer between\n[nornir-nautobot](https://github.com/nautobot/nornir-nautobot) and other plugins. The primary abilities that the plugin provides is a native Nornir\nORM based inventory and a credential manager.\n\n![Architecture Overview](./docs/img/architecture-overview.png)\n\nAs of the writing of this readme, the only plugin leveraging this plugin is the [golden-config](https://github.com/nautobot/nautobot-plugin-golden-config). However, future plugins are planned, such as the "network importer".\n\nThat being said, there is currently little reason to install this plugin by itself, without an enabler, which can be seen represented as the white\nrectangles inside the yellow rectangle in the diagram above. An enabler could be a Plugin, Job, or another piece of code like a Chatops command.\n\n# Installation\n\nIf using the installation pattern from the Nautobot Documentation, you will need sudo to the `nautobot` user before installing so that you install the package into the Nautobot virtual environment.\n\n```no-highlight\nsudo -iu nautobot\n```\n\nThe plugin is available as a Python package in PyPI and can be installed with `pip3`.\n\n```no-highlight\n$ pip3 install nautobot-plugin-nornir\n```\n\nTo ensure the plugin is automatically re-installed during future upgrades, create a file named `local_requirements.txt` (or append if it already exists) in the [`NAUTOBOT_ROOT`](https://nautobot.readthedocs.io/en/latest/configuration/optional-settings/#nautobot_root) directory and list the `nautobot-plugin-nornir` package:\n\n```no-highlight\n$ echo nautobot-plugin-nornir >> $NAUTOBOT_ROOT/local_requirements.txt\n```\n\n> The plugin is compatible with Nautobot 1.0.0 and higher\n \nOnce installed, the plugin needs to be enabled in your `nautobot_config.py`\n\n```python\n# In your nautobot_config.py\nPLUGINS = ["nautobot_plugin_nornir"]\n\nPLUGINS_CONFIG = {\n  "nautobot_plugin_nornir": {\n    "nornir_settings": {\n      "credentials": "nautobot_plugin_nornir.plugins.credentials.env_vars.CredentialsEnvVars",\n      "runner": {\n        "plugin": "threaded",\n        "options": {\n            "num_workers": 20,\n        },\n      },\n    },\n  }\n```\n\nAlternatively you can use the `CredentialsSettingsVars` class to set the username and password via settings.\n\n```python\nPLUGINS_CONFIG = {\n  "nautobot_plugin_nornir": {\n    "nornir_settings": {\n      "credentials": "nautobot_plugin_nornir.plugins.credentials.settings_vars.CredentialsSettingsVars",\n      "runner": {\n        "plugin": "threaded",\n        "options": {\n            "num_workers": 20,\n        },\n      },\n    },\n    "dispatcher_mapping": None,\n    "username": "ntc",\n    "password": "password123",\n    "secret": "password123",\n  }\n}\n```\nThe plugin behavior can be controlled with the following list of settings. \n\n| Key     | Example | Default | Description                          |\n| ------- | ------ | -------- | ------------------------------------- |\n| nornir_settings | {"nornir_settings": { "credentials": "cred_path"}} | N/A | The expected configuration paramters that Nornir uses, see Nornir documentation. |\n| dispatcher_mapping | {"newos": "dispatcher.newos"} | None | A dictionary in which the key is a platform slug and the value is the import path of the dispatcher in string format |\n| username | ntc | N/A | The username when leveraging the `CredentialsSettingsVars` credential provider. |\n| password | password123 | N/A | The password when leveraging the `CredentialsSettingsVars` credential provider. |\n| secret | password123 | N/A | The secret password when leveraging the `CredentialsSettingsVars` credential provider, **placeholder only, not currently functioning**. |\n\nFinally, as root, restart Nautobot and the Nautobot worker.\n\n```no-highlight\n$ sudo systemctl restart nautobot nautobot-worker\n```\n\n## Setting `dispatcher_mapping`\n\nThe `dispatcher_mapping` configuration option can be set to extend or map the platform slug to a proper nornir class.\n\n```json\n {\n  "dispatcher_mapping": {\n    "newos": "dispatcher.newos",\n    "ios": "nautobot_nornir.plugins.tasks.dispatcher.cisco_ios.NautobotNornirDriver",\n    "ios_xe": "nautobot_nornir.plugins.tasks.dispatcher.cisco_ios.NautobotNornirDriver",\n    "fortinet": "nautobot_nornir.plugins.tasks.dispatcher.default.NetmikoNautobotNornirDriver",\n  }\n}\n```\n\nThe above example demonstrates the following use cases.\n\n* Creating a custom only local dispatcher\n* Mapping a user defined and preffered platform slug name to expected driver (e.g. ios -> cisco_ios)\n* Overloading platform slug keys, by mapping ios and ios_xe to the same class\n* Leveraging the existing "default" Netmiko driver\n\nPlugin developers that leverage the plugin, are recommended to use the `get_dispatcher` function in `nautobot_plugin_nornir.utils` file to provide the ability to\nallow users to define their own mappings as described above.\n\n# Inventory\n\nThe Nautobot ORM inventory is rather static in nature at this point. The user has the ability to define the `default` data. The native capabilites\ninclude. \n\n* Providing an object called within the `obj` key that is a Nautobot `Device` object instance.\n* Provide additional keys for hostname, name, id, type, site, role, config_context, and custom_field_data.\n* Provide grouping for global, site, role, type, and manufacturer based on their slug.\n* Provide credentials for NAPALM and Netmiko.\n* Link to the credential class as defined by the `nornir_settings[\'settings\']` definition.\n\n# Credentials\n\nThere is a `NautobotORMCredentials` class that describes what methods a Nautobot Nornir credential class should have.\n\n```python\nclass NautobotORMCredentials:\n    """Abstract Credentials Class designed to work with Nautobot ORM."""\n\n    def get_device_creds(self, device):\n        """Return the credentials for a given device.\n\n        Args:\n            device (dcim.models.Device): Nautobot device object\n\n        Return:\n            username (string):\n            password (string):\n            secret (string):\n        """\n        return (None, None, None)\n\n    def get_group_creds(self, group_name):\n        """Return the credentials for a given group.\n\n        Args:\n            group_name (string): Name of the group\n\n        Return:\n            string: username\n            string: password\n            string: secret\n        """\n        return (None, None, None)\n```\n\nAny custom credential class should inherit from this model and provide get_device_creds and/or get_group_creds methods. Currently, only the\nget_device_creds is used. Building your own custom credential class allows users to control their own credential destiny. As an example, a user could\nintegrate with their own vaulting system, and obtain credentials that way. To provide a simple but concrete example.\n\n```python\nclass CustomNautobotORMCredentials(NautobotORMCredentials):\n\n    def get_device_creds(self, device):\n        if device.startswith(\'csr\'):\n            return ("cisco", "cisco123", None)\n        return ("net-admin", "ops123", None)\n```\n\nYou would have to set your `nornir_settings[\'credentials\']` path to your custom class, such as `local_plugin.creds.CustomNautobotORMCredentials`.\n\nOut of the box, users have access to the `nautobot_plugin_nornir.plugins.credentials.settings_vars.CredentialsSettingsVars` and \n`nautobot_plugin_nornir.plugins.credentials.env_vars.CredentialsEnvVars` class. This `CredentialsEnvVars` class simply leverages the \nenvironment variables `NAPALM_USERNAME`, `NAPALM_PASSWORD`, and `DEVICE_SECRET`.\n\n> Note: DEVICE_SECRET does not currently work.\n\nThe environment variable must be accessible on the web service. This often means simply exporting the environment variable will not \nsuffice, but instead requiring users to update the `nautobot.service` file, however this will ultimately depend on your own setup. Environment\nvariables are distinctively not nautobot configuration parameters (in `naubot_config.py`), if that does not makes sense, expect to see authentication issues.\n\n```\n[Service]\nEnvironment="NAPALM_USERNAME=ntc"\nEnvironment="NAPALM_PASSWORD=password123"\nEnvironment="DEVICE_SECRET=password123"\n```\n\n# Contributing\n\nPull requests are welcomed and automatically built and tested against multiple version of Python and multiple version of Nautobot through TravisCI.\n\nThe project is packaged with a light development environment based on `docker-compose` to help with the local development of the project and to run the tests within TravisCI.\n\nThe project is following Network to Code software development guideline and is leveraging:\n- Black, Pylint, Bandit and pydocstyle for Python linting and formatting.\n- Django unit test to ensure the plugin is working properly.\n\n# CLI Helper Commands\n\nThe project is coming with a CLI helper based on [invoke](http://www.pyinvoke.org/) to help setup the development environment. The commands are listed below in 3 categories `dev environment`, `utility` and `testing`. \n\nEach command can be executed with `invoke <command>`. All commands support the arguments `--nautobot-ver` and `--python-ver` if you want to manually define the version of Python and Nautobot to use. Each command also has its own help `invoke <command> --help`\n\n## Local dev environment\n```\n  build            Build all docker images.\n  debug            Start Nautobot and its dependencies in debug mode.\n  destroy          Destroy all containers and volumes.\n  restart          Restart Nautobot and its dependencies.\n  start            Start Nautobot and its dependencies in detached mode.\n  stop             Stop Nautobot and its dependencies.\n```\n\n## Utility \n```\n  cli              Launch a bash shell inside the running Nautobot container.\n  create-user      Create a new user in django (default: admin), will prompt for password.\n  makemigrations   Run Make Migration in Django.\n  nbshell          Launch a nbshell session.\n```\n## Testing \n\n```\n  bandit           Run bandit to validate basic static code security analysis.\n  black            Run black to check that Python files adhere to its style standards.\n  flake8           This will run flake8 for the specified name and Python version.\n  pydocstyle       Run pydocstyle to validate docstring formatting adheres to NTC defined standards.\n  pylint           Run pylint code analysis.\n  tests            Run all tests for this plugin.\n  unittest         Run Django unit tests for the plugin.\n```\n\n# Questions\n\nFor any questions or comments, please check the [FAQ](FAQ.md) first and feel free to swing by the [Network to Code slack channel](https://networktocode.slack.com/) (channel #networktocode).\nSign up [here](http://slack.networktocode.com/)\n',
    'author': 'Network to Code, LLC',
    'author_email': 'opensource@networktocode.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nautobot/nautobot-plugin-nornir',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
