"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOrgObj = exports.OrgObjectTypes = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The supported OrgObject types.
 *
 * @stability stable
 */
var OrgObjectTypes;
(function (OrgObjectTypes) {
    OrgObjectTypes["OU"] = "ou";
    OrgObjectTypes["ACCOUNT"] = "account";
})(OrgObjectTypes = exports.OrgObjectTypes || (exports.OrgObjectTypes = {}));
;
;
/**
 * @function processOrgObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} provider The custom resource provider the custom resources will utilized to create the resources.
 * @param {OrgObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOrgObj(provider, obj, parent) {
    var _c;
    if (obj.type === OrgObjectTypes.OU) {
        const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
        let props = obj.properties;
        let id = (_c = obj.id) !== null && _c !== void 0 ? _c : obj.properties.name;
        const ou = new OrganizationOU(this, id, {
            provider,
            parent: parentStr,
            name: props.name,
            importOnDuplicate: props.importOnDuplicate,
            allowMergeOnMove: props.allowMergeOnMove,
            allowRecreateOnUpdate: props.allowRecreateOnUpdate,
        });
        obj.children.forEach(child => {
            processOrgObj.call(this, provider, child, ou);
        });
    }
}
exports.processOrgObj = processOrgObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                            'organizations:ListChildren',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.2.2" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e;
        super(scope, id);
        const importOnDuplicate = (_c = props.importOnDuplicate) !== null && _c !== void 0 ? _c : false;
        const allowMergeOnMove = (_d = props.allowMergeOnMove) !== null && _d !== void 0 ? _d : false;
        const allowRecreateOnUpdate = (_e = props.allowRecreateOnUpdate) !== null && _e !== void 0 ? _e : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowMergeOnMove: allowMergeOnMove,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.2.2" };
//# sourceMappingURL=data:application/json;base64,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