# aclint: py3
import asyncio
import logging
import argparse

from revup import git
from revup.types import RevupUsageException


async def main(args: argparse.Namespace, git_ctx: git.Git) -> int:
    if args.toolkit_cmd == "detect-branch":
        if args.show_all:
            target_branches = await git_ctx.get_best_base_branch_candidates(
                "HEAD", not args.no_limit
            )
            logging.info(", ".join(target_branches))
        else:
            target_branch = await git_ctx.get_best_base_branch("HEAD", not args.no_limit)
            logging.info(target_branch)
    elif args.toolkit_cmd == "cherry-pick":
        await asyncio.gather(
            git_ctx.verify_branch_or_commit(args.commit),
            git_ctx.verify_branch_or_commit(args.parent),
        )

        commit_header = git.parse_rev_list(
            await git_ctx.rev_list(args.commit, max_revs=1, header=True)
        )
        if len(commit_header) != 1:
            raise RevupUsageException(f"Commit {args.commit} doesn't exist!")
        logging.info(await git_ctx.synthetic_cherry_pick_from_commit(commit_header[0], args.parent))
    elif args.toolkit_cmd == "diff-target":
        await asyncio.gather(
            git_ctx.verify_branch_or_commit(args.old_head),
            git_ctx.verify_branch_or_commit(args.new_head),
        )

        if not args.old_base:
            args.old_base = git_ctx.to_commit_hash(args.old_head + "~")
        if not args.new_base:
            args.new_base = git_ctx.to_commit_hash(args.new_head + "~")
        logging.info(
            await git_ctx.make_virtual_diff_target(
                args.old_base, args.old_head, args.new_base, args.new_head, args.parent
            )
        )

    return 0
