import pickle
import numpy as np
import umap
import pandas as pd
import seaborn as sns
import matplotlib.pyplot as plt
import os
import urllib

class Meta2Vec:
    """
    A class for calculating similarity between HMDB (Human Metabolome Database) metabolites
    based on their embeddings generated by TransE.

    Parameters:
    embedding_path (str): path to the HMDB embeddings file

    Attributes:
    hmdb_embeddings (dict): a dictionary containing HMDB IDs as keys and their embeddings as values
    vocab (list): a list of all available HMDB IDs

    Methods:
    load_embedding: loads the HMDB embeddings from the given file path
    get_available_hmdbs: gets a list of all available HMDB IDs
    get_hmdb_embedding: gets the embeddings of one or more HMDB IDs
    cal_similarity: calculates the cosine similarity between two HMDB metabolites
    cal_similarity_all: calculates the cosine similarity between one HMDB metabolite and all others
    most_similar: gets the most similar HMDB metabolites to a given one
    visualize_umap: visualizes the HMDB embeddings using UMAP
    """

    def __init__(self, embedding_path="embedding/TransE_HMDB_Embedding.pkl"):
        """
        Initializes the Meta2Vec class with the given embedding file path and
        loads the embeddings.

        Parameters:
        embedding_path (str): path to the HMDB embeddings file
        """
        if not os.path.exists("embedding"):
            os.mkdir("embedding")
        if not os.path.exists(embedding_path):
            urllib.request.urlretrieve("https://github.com/YuxingLu613/meta2vec/raw/main/embedding/TransE_HMDB_Embedding.pkl",embedding_path)
        self.load_embedding(embedding_path)
        self.get_available_hmdbs()

    def load_embedding(self, embedding_path="embedding/TransE_HMDB_Embedding.pkl"):
        """
        Loads the HMDB embeddings from the given file path and assigns them to
        the hmdb_embeddings attribute.

        Parameters:
        embedding_path (str): path to the HMDB embeddings file

        Returns:
        hmdb_embeddings (dict): a dictionary containing HMDB IDs as keys and
        their embeddings as values
        """
        self.embedding_path = embedding_path
        with open(self.embedding_path, "rb") as f:
            self.hmdb_embeddings = pickle.load(f)
        return self.hmdb_embeddings

    def get_available_hmdbs(self):
        """
        Gets a list of all available HMDB IDs and assigns it to the vocab
        attribute.

        Returns:
        vocab (list): a list of all available HMDB IDs
        """
        self.vocab = list(self.hmdb_embeddings.keys())
        return self.vocab

    def get_hmdb_embedding(self, hmdb_id_list):
        """
        Gets the embeddings of one or more HMDB IDs and returns them in a dictionary.

        Parameters:
        hmdb_id_list (str or list): a single HMDB ID as a string or a list of
        HMDB IDs

        Returns:
        hmdb_embedding (dict): a dictionary containing HMDB IDs as keys and
        their embeddings as values
        """
        if type(hmdb_id_list) == str:
            hmdb_id_list = [hmdb_id_list]
        hmdb_embedding = {}
        for id in hmdb_id_list:
            hmdb_embedding.get(id, self.hmdb_embeddings[id])
        return hmdb_embedding
    
    def cal_similarity_all(self, hmdb_1):
        """
        Calculates the cosine similarity between one HMDB metabolite and all
        others.

        Parameters:
        hmdb_1 (str): the HMDB ID to compare against all others

        Returns:
        similarity_dict (list): a list of tuples containing the HMDB IDs and
        their cosine similarity with the given HMDB ID
        """
        hmdb_1_embedding = self.hmdb_embeddings[hmdb_1]
        similarity_dict = {}
        for hmdb_id, embedding in self.hmdb_embeddings.items():
            if hmdb_id != hmdb_1:
                similarity_dict[hmdb_id] = np.dot(hmdb_1_embedding, embedding) / (np.linalg.norm(hmdb_1_embedding) * np.linalg.norm(embedding))
        return list(similarity_dict.items())

    def most_similar(self, hmdb_1, topk=5):
        """
        Returns a list of most similar HMDB IDs to the input HMDB ID.

        Parameters:
        hmdb_id (str): The HMDB ID for which similar HMDB IDs are to be found.
        topk (int): The number of most similar HMDB IDs to be returned.

        Returns:
        list: A list of tuples containing the HMDB ID and similarity score, ordered by similarity score in decreasing order.
        """
        hmdb_1_embedding = self.hmdb_embeddings[hmdb_1]
        similarity_dict = {}
        for hmdb_id, embedding in self.hmdb_embeddings.items():
            if hmdb_id != hmdb_1:
                similarity_dict[hmdb_id] = np.dot(hmdb_1_embedding, embedding) / (np.linalg.norm(hmdb_1_embedding) * np.linalg.norm(embedding))
        return sorted(list(similarity_dict.items()), key=lambda x: x[1], reverse=True)[:topk]

    
    
    def visualize_umap(self):
        """
        Visualizes the UMAP embedding of HMDB IDs in 2D space.
        """
        reducer = umap.UMAP()
        hmdb_2d_embedding = reducer.fit_transform(list(self.hmdb_embeddings.values()))
        sns.scatterplot(x=hmdb_2d_embedding[:, 0], y=hmdb_2d_embedding[:, 1])
        plt.show()

        
if __name__=="__main__":
    meta2vec=Meta2Vec()
    meta2vec.load_embedding()
    print(len(meta2vec.get_available_hmdbs()))
    meta2vec.get_hmdb_embedding(list(meta2vec.hmdb_embeddings.keys())[:10])
    print(meta2vec.cal_similarity("HMDB0000001","HMDB0000060"))
    meta2vec.visualize_umap()