---
jupytext:
  text_representation:
    extension: .md
    format_name: myst
    format_version: 0.12
    jupytext_version: 1.6.0rc0
kernelspec:
  display_name: Python 3
  language: python
  name: python3
---

(pytest_notebook_by_example)=

# pytest-notebook by example

:::{seealso}
This notebook was rendered with [myst-nb](https://myst-nb.readthedocs.io): {nb-download}`tutorial_intro.ipynb`
:::

+++

## Python API

The principal component of `pytest-notebook` is the
{py:class}`~pytest_notebook.nb_regression.NBRegressionFixture` class,
which is an [attrs](http://www.attrs.org) class, whose parameters can be instantiated or set via attributes.

```{code-cell} ipython3
try:
    # Python <= 3.8
    from importlib_resources import files
except ImportError:
    from importlib.resources import files
from pytest_notebook import example_nbs
from pytest_notebook.nb_regression import NBRegressionFixture
```

```{code-cell} ipython3
fixture = NBRegressionFixture(exec_timeout=50)
fixture.diff_color_words = False
fixture
```

The main method is {py:meth}`~pytest_notebook.nb_regression.NBRegressionFixture.check`, which executes a notebook and compares its initial and final contents.

```{code-cell} ipython3
:tags: [raises-exception]

with files(example_nbs).joinpath("example1.ipynb") as path:
    fixture.check(str(path))
```

To return the results, without raising an exception, use ``raise_errors=False``. This returns a {py:class}`~pytest_notebook.nb_regression.NBRegressionResult` instance.

```{code-cell} ipython3
with files(example_nbs).joinpath("example1.ipynb") as path:
    result = fixture.check(str(path), raise_errors=False)

result
```

```{code-cell} ipython3
print(result.diff_string)
```

The diff of the notebooks is returned as a list of [nbdime](https://nbdime.readthedocs.io) `DiffEntry` objects.

```{code-cell} ipython3
result.diff_filtered
```

The notebooks are returned as {py:class}`nbformat.NotebookNode` instances.

```{code-cell} ipython3
result.nb_final
```

## pytest fixture

+++

:::{seealso}
    {py:mod}`pytest_notebook.ipy_magic`,
    for the notebook magic used to run pytest in a Jupyter Notebook.
:::

```{code-cell} ipython3
%load_ext pytest_notebook.ipy_magic
```

A {py:class}`~pytest_notebook.nb_regression.NBRegressionFixture` instance can accessed *via* the {py:func}`~pytest_notebook.plugin.nb_regression` fixture.
This instance will be instatiated with parameters dictated by arguments parsed from the pytest command-line and configuration file(s).

:::{note}
pytest-notebook command-line and configuration file parameter names
are the same as for ``NBRegressionFixture``, but with the prefix ``nb_``.

The command-line parameter takes precedence over the configuration file one.
:::

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings --nb-exec-timeout 50

---
[pytest]
nb_diff_color_words = True
---
try:
    # Python <= 3.8
    from importlib_resources import files
except ImportError:
    from importlib.resources import files
from pytest_notebook import example_nbs

def test_notebook(nb_regression):
    with files(example_nbs).joinpath("example1.ipynb") as path:
        nb_regression.check(str(path))
```

## pytest file collection

+++

{py:meth}`~pytest_notebook.nb_regression.NBRegressionFixture.check` can be run automatically on all notebooks using the pytest collection mechanism.
To activate this feature, set `--nb-test-files` on the command-line, or `nb_test_files = True` in the configuration file.

```{code-cell} ipython3
notebook1_content = files(example_nbs).joinpath("example1_pass.ipynb").read_text()
notebook2_content = files(example_nbs).joinpath("example1.ipynb").read_text()
```

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings --nb-test-files

***
(notebook1_content, "test_notebook1.ipynb")
(notebook2_content, "test_notebook2.ipynb")
***
```

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings

---
[pytest]
nb_test_files = True
---

***
(notebook1_content, "test_notebook1.ipynb")
(notebook2_content, "test_notebook2.ipynb")
***
```

To restrict the notebook files pytest collects, one or more filename pattern matches can also be set (see {py:mod}`fnmatch`).

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings

---
[pytest]
nb_test_files = True
nb_file_fnmatch = test_*.ipynb tutorial_*.ipynb
---

***
(notebook1_content, "test_notebook1.ipynb")
(notebook2_content, "other_notebook2.ipynb")
***
```

## Live Logging of Cell Execution

+++

If you wish to view the progress of the notebook execution, you can use the [pytest live-logging](https://docs.pytest.org/en/latest/logging.html#live-logs) functionality:

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings --nb-test-files --log-cli-level=info

***
(notebook1_content, "test_notebook1.ipynb")
***
```

(regen_notebooks)=

## Regenerating Notebooks

+++

Failing notebooks can be regenerated by setting `--nb-force-regen`.
This will overwrite failing notebooks with the output from the notebook execution.

:::{note}
Notebooks will not be regenerated if they raise any unexpected exceptions,
during execution.

This approach to regeneration mimics [pytest-regressions](https://pytest-regressions.readthedocs.io).
:::

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings --nb-force-regen

---
[pytest]
nb_test_files = True
nb_force_regen = True
---

***
(notebook1_content, "test_notebook1.ipynb")
(notebook2_content, "test_notebook2.ipynb")
***
```

The regeneration can be observed, if we run two tests on the same notebook.

```{code-cell} ipython3
%%pytest -v --color=yes --disable-warnings

import os, tempfile
try:
    # Python <= 3.8
    from importlib_resources import files
except ImportError:
    from importlib.resources import files
import pytest
from pytest_notebook import example_nbs

@pytest.fixture(scope="module")
def notebook():
    tmphandle, tmppath = tempfile.mkstemp(suffix=".ipynb")
    with open(tmppath, "w") as handle:
        handle.write(
            files(example_nbs).joinpath("example1.ipynb").read_text()
        )
    yield tmppath
    os.remove(tmppath)

def test_notebook1(nb_regression, notebook):
    nb_regression.force_regen = True
    nb_regression.check(notebook)

def test_notebook2(nb_regression, notebook):
    nb_regression.check(notebook)
```
