"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Project = exports.ProjectBase = exports.Glossaries = exports.Forms = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const fs = require("fs");
const cdk = require("aws-cdk-lib");
const datazone = require("aws-cdk-lib/aws-datazone");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const form_metadata_1 = require("./form-metadata");
const glossary_1 = require("./glossary");
const resource_1 = require("./resource");
const IS_PROJECT = Symbol.for('amazon-datazone.Project');
class Forms {
    static fromInline(...options) {
        return new Forms(options);
    }
    static fromFile(path) {
        const options = JSON.parse(fs.readFileSync(path, 'utf8'));
        return new Forms(options);
    }
    constructor(metadataOptions = []) {
        this.metadataOptions = metadataOptions;
    }
}
exports.Forms = Forms;
_a = JSII_RTTI_SYMBOL_1;
Forms[_a] = { fqn: "cdk-data-zone.Forms", version: "0.0.0" };
class Glossaries {
    static fromInline(...options) {
        return new Glossaries(options);
    }
    static fromFile(path) {
        const options = JSON.parse(fs.readFileSync(path, 'utf8'));
        return new Glossaries(options);
    }
    constructor(glossariesList = []) {
        this.glossariesList = glossariesList;
    }
}
exports.Glossaries = Glossaries;
_b = JSII_RTTI_SYMBOL_1;
Glossaries[_b] = { fqn: "cdk-data-zone.Glossaries", version: "0.0.0" };
class ProjectBase extends resource_1.ResourceBase {
}
exports.ProjectBase = ProjectBase;
_c = JSII_RTTI_SYMBOL_1;
ProjectBase[_c] = { fqn: "cdk-data-zone.ProjectBase", version: "0.0.0" };
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.glossaryTerms = [];
        // check if domain name is 64 characters or less
        if (!cdk.Token.isUnresolved(props.name) && props.name.length > 64) {
            throw new Error('Project name must be 64 characters or less');
        }
        if (props.glossaryTerms) {
            props.glossaryTerms.forEach((term) => this.addGlossaryTerm(term));
        }
        const project = new datazone.CfnProject(this, 'Resource', {
            domainIdentifier: props.domain.domainId,
            name: props.name,
            glossaryTerms: cdk.Lazy.list({ produce: () => this.glossaryTerms }, { omitEmpty: true }),
            description: props.description,
        });
        this.projectId = project.attrId;
        this.projectDomainId = project.attrDomainId;
        this.createdBy = project.attrCreatedBy;
        this.createdAt = project.attrCreatedAt;
        this.updatedAt = project.attrLastUpdatedAt;
        this.managementRole = props.managementRole ?? this.createManagementRole();
        this.membership = this._addMember('automation', {
            userIdentifier: this.managementRole.roleArn,
            designation: 'PROJECT_OWNER',
        });
        constructs_1.Dependable.of(this.membership);
        this.membership.node.addDependency(this.managementRole);
        if (props.forms) {
            this.addFormMetadata(props.forms);
        }
        if (props.glossaries) {
            this.addGlossaries(props.glossaries);
        }
        Object.defineProperty(this, IS_PROJECT, { value: true });
    }
    addGlossaries(glossaries) {
        glossaries.glossariesList.forEach((options) => this.addGlossary(options));
    }
    addGlossaryTerm(term) {
        if (this.glossaryTerms.length >= 20) {
            throw new Error('A project can have a maximum of 20 glossary terms');
        }
        this.glossaryTerms.push(term);
    }
    addGlossary(options) {
        return new glossary_1.Glossary(this, `Glossary-${options.name}`, {
            project: this,
            ...options,
        });
    }
    addMember(id, options) {
        if (options.userIdentifier && options.groupIdentifier) {
            throw new Error('Only one of userIdentifier or groupIdentifier should be provided');
        }
        this._addMember(id, options);
    }
    addFormMetadata(forms) {
        forms.metadataOptions.forEach((options) => this._addFormMetadata(options));
    }
    _addFormMetadata(options) {
        return new form_metadata_1.FormMetadata(this, `FormMetadata-${options.name}`, {
            project: this,
            ...options,
        });
    }
    _addMember(id, options) {
        return new datazone.CfnProjectMembership(this, `ProjectMembership-${id}`, {
            domainIdentifier: this.projectDomainId,
            projectIdentifier: this.projectId,
            member: {
                userIdentifier: options.userIdentifier,
                groupIdentifier: options.groupIdentifier,
            },
            designation: options.designation ?? 'PROJECT_OWNER',
        });
    }
    createManagementRole() {
        return new iam.Role(this, 'ManagementRole', {
            assumedBy: new iam.ServicePrincipal('datazone.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AdministratorAccess'),
            ],
        });
    }
}
exports.Project = Project;
_d = JSII_RTTI_SYMBOL_1;
Project[_d] = { fqn: "cdk-data-zone.Project", version: "0.0.0" };
//# sourceMappingURL=data:application/json;base64,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