import * as datazone from 'aws-cdk-lib/aws-datazone';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { IDomain } from './domain';
import { Glossary, GlossaryTermOptions } from './glossary';
import { IResource, ResourceBase } from './resource';
export interface ProjectProps {
    /**
     * The identifier of a Amazon  domain where the project exists.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-datazone-project.html#cfn-datazone-project-domainidentifier
     */
    readonly domain: IDomain;
    /**
     * The description of a project.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-datazone-project.html#cfn-datazone-project-description
     */
    readonly description?: string;
    /**
     * The glossary terms that can be used in this Amazon  project.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-datazone-project.html#cfn-datazone-project-glossaryterms
     */
    readonly glossaryTerms?: Array<string>;
    /**
     * The name of a project.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-datazone-project.html#cfn-datazone-project-name
     */
    readonly name: string;
    readonly managementRole?: iam.Role;
    readonly forms?: Forms;
    readonly glossaries?: Glossaries;
}
export interface FormMetadataOptions {
    readonly name: string;
    readonly description?: string;
    readonly smithyModel?: string;
}
export interface GlossaryOptions {
    readonly name: string;
    readonly description: string;
    readonly terms?: GlossaryTermOptions[];
}
export interface MemberOptions {
    readonly designation?: 'PROJECT_OWNER' | 'PROJECT_CONTRIBUTOR';
    readonly userIdentifier?: string;
    readonly groupIdentifier?: string;
}
export declare class Forms {
    readonly metadataOptions: FormMetadataOptions[];
    static fromInline(...options: FormMetadataOptions[]): Forms;
    static fromFile(path: string): Forms;
    private constructor();
}
export declare class Glossaries {
    readonly glossariesList: GlossaryOptions[];
    static fromInline(...options: GlossaryOptions[]): Glossaries;
    static fromFile(path: string): Glossaries;
    private constructor();
}
export interface IProject extends IResource {
    /**
     * The identifier of a Amazon  domain where the project exists.
     *
     * @attribute
     */
    readonly projectDomainId: string;
    /**
     * The identifier of a project.
     *
     * @attribute
     */
    readonly projectId: string;
    /**
     * The Amazon  user who created the project.
     *
     * @attribute
     */
    readonly createdBy: string;
    addGlossaries(glossaries: Glossaries): void;
    addGlossaryTerm(term: string): void;
    addGlossary(options: GlossaryOptions): Glossary;
    addMember(id: string, options: MemberOptions): void;
    addFormMetadata(forms: Forms): void;
}
export declare abstract class ProjectBase extends ResourceBase implements IProject {
    readonly abstract projectDomainId: string;
    /**
     * The identifier of a project.
     *
     * @attribute
     */
    readonly abstract projectId: string;
    /**
     * The Amazon  user who created the project.
     *
     * @attribute
     */
    readonly abstract createdBy: string;
    abstract addGlossaries(glossaries: Glossaries): void;
    abstract addGlossaryTerm(term: string): void;
    abstract addGlossary(options: GlossaryOptions): Glossary;
    abstract addMember(id: string, options: MemberOptions): void;
    abstract addFormMetadata(forms: Forms): void;
}
export declare class Project extends ProjectBase {
    readonly projectDomainId: string;
    /**
     * The identifier of a project.
     *
     * @attribute
     */
    readonly projectId: string;
    /**
     * The Amazon  user who created the project.
     *
     * @attribute
     */
    readonly createdBy: string;
    /**
     * @attribute
     */
    readonly createdAt: string;
    /**
     * The timestamp of when the project was last updated.
     *
     * @attribute
     */
    readonly updatedAt: string;
    readonly managementRole: iam.Role;
    readonly membership: datazone.CfnProjectMembership;
    private readonly glossaryTerms;
    constructor(scope: Construct, id: string, props: ProjectProps);
    addGlossaries(glossaries: Glossaries): void;
    addGlossaryTerm(term: string): void;
    addGlossary(options: GlossaryOptions): Glossary;
    addMember(id: string, options: MemberOptions): void;
    addFormMetadata(forms: Forms): void;
    private _addFormMetadata;
    private _addMember;
    private createManagementRole;
}
