"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.DomainBase = exports.AssignmentType = exports.SingleSignOnType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const cdk = require("aws-cdk-lib");
const datazone = require("aws-cdk-lib/aws-datazone");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const blueprint_1 = require("./blueprint");
const project_1 = require("./project");
const resource_1 = require("./resource");
const IS_DOMAIN = Symbol.for('amazon-datazone.Domain');
var SingleSignOnType;
(function (SingleSignOnType) {
    SingleSignOnType["IAM_IDC"] = "IAM_IDC";
    SingleSignOnType["DISABLED"] = "DISABLED";
})(SingleSignOnType || (exports.SingleSignOnType = SingleSignOnType = {}));
var AssignmentType;
(function (AssignmentType) {
    AssignmentType["AUTOMATIC"] = "AUTOMATIC";
    AssignmentType["MANUAL"] = "MANUAL";
})(AssignmentType || (exports.AssignmentType = AssignmentType = {}));
class DomainBase extends resource_1.ResourceBase {
    enableBlueprint(blueprintIdentifier, options) {
        return new blueprint_1.Blueprint(this, blueprintIdentifier, {
            ...options,
            domain: this,
            environmentBlueprintIdentifier: blueprintIdentifier,
        });
    }
    createProject(id, options) {
        return new project_1.Project(this, id, {
            ...options,
            domain: this,
        });
    }
}
exports.DomainBase = DomainBase;
_a = JSII_RTTI_SYMBOL_1;
DomainBase[_a] = { fqn: "cdk-data-zone.DomainBase", version: "0.0.0" };
class Domain extends DomainBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.description = props.description;
        this.name = props.name;
        this.encryptionKey = props.encryptionKey ?? this.createEncryptionKey();
        this.domainExecutionRole = props.domainExecutionRole ?? this.createDomainExecutionRole();
        this._singleSignOn = props.singleSignOn;
        this.encryptionKey.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
            resources: ['*'],
        }));
        this.encryptionKey.addToResourcePolicy(new iam.PolicyStatement({
            actions: [
                'kms:Encrypt',
                'kms:Decrypt',
                'kms:ReEncrypt*',
                'kms:GenerateDataKey*',
                'kms:DescribeKey',
                'kms:CreateGrant',
                'kms:ListGrants',
                'kms:RevokeGrant',
            ],
            principals: [new iam.AnyPrincipal()],
            resources: ['*'],
            conditions: {
                StringEquals: {
                    'kms:CallerAccount': cdk.Stack.of(this).account,
                },
            },
        }));
        const resource = new datazone.CfnDomain(this, 'Resource', {
            name: this.name,
            description: this.description,
            kmsKeyIdentifier: this.encryptionKey?.keyArn,
            domainExecutionRole: this.domainExecutionRole.roleArn,
            singleSignOn: this.createSingleSignOn(),
            tags: props.tags?.map((tag) => ({ key: tag, value: tag })) ?? [],
        });
        this.domainId = resource.attrId;
        this.createdAt = resource.attrCreatedAt;
        this.updatedAt = resource.attrLastUpdatedAt;
        this.domainArn = resource.attrArn;
        this.portalUrl = resource.attrPortalUrl;
        this.managedAccount = resource.attrManagedAccountId;
        this.status = resource.attrStatus;
        Object.defineProperty(this, IS_DOMAIN, { value: true });
    }
    addSingleSignOn(singleSignOn) {
        this._singleSignOn = singleSignOn;
    }
    createEncryptionKey() {
        return new kms.Key(this, 'DomainKey', {
            enableKeyRotation: true,
            description: 'Datazone KMS Key',
            alias: `${this.name}-datazone-key`,
        });
    }
    createDomainExecutionRole() {
        const role = new iam.Role(this, 'DomainExecutionRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('cloudformation.amazonaws.com')),
            inlinePolicies: {
                DomainExecutionRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'datazone:*',
                                'ram:GetResourceShareAssociations',
                                'sso:CreateManagedApplicationInstance',
                                'sso:DeleteManagedApplicationInstance',
                                'sso:PutApplicationAssignmentConfiguration',
                                'kms:Decrypt',
                                'kms:DescribeKey',
                                'kms:GenerateDataKey',
                            ],
                            effect: iam.Effect.ALLOW,
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        const dataZoneAssumeRoleStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('datazone.amazonaws.com')],
            actions: ['sts:AssumeRole', 'sts:TagSession'],
            conditions: {
                'StringEquals': {
                    'aws:SourceAccount': cdk.Stack.of(this).account,
                },
                'ForAllValues:StringLike': {
                    'aws:TagKeys': 'datazone*',
                },
            },
        });
        role.assumeRolePolicy?.addStatements(dataZoneAssumeRoleStatement);
        return role;
    }
    createSingleSignOn() {
        return cdk.Lazy.any({
            produce: () => {
                if (!this._singleSignOn)
                    return undefined;
                return {
                    type: this._singleSignOn?.ssoType ?? SingleSignOnType.IAM_IDC,
                    userAssignment: this._singleSignOn?.userAssignment ?? AssignmentType.AUTOMATIC,
                };
            },
        });
    }
}
exports.Domain = Domain;
_b = JSII_RTTI_SYMBOL_1;
Domain[_b] = { fqn: "cdk-data-zone.Domain", version: "0.0.0" };
//# sourceMappingURL=data:application/json;base64,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